/**
 *   COMICS - Computing Minimal Counterexamples for Discrete-time Markov Chains
 *
 *   COMICS is a stand-alone tool which performs model checking and the generation
 *   of counterexamples for discrete-time Markov Chains (DTMCs). *
 *
 *   Copyright (C) <2012> <RWTH Aachen University>
 *   Authors: Nils Jansen, Erika Abraham, Jens Katelaan, Maik Scheffler, Matthias Volk, Andreas Vorpahl
 *
 *   This program is free software: you can redistribute it and/or modify
 *   it under the terms of the GNU General Public License as published by
 *   the Free Software Foundation, either version 3 of the License, or
 *   (at your option) any later version.
 *
 *   This program is distributed in the hope that it will be useful,
 *   but WITHOUT ANY WARRANTY; without even the implied warranty of
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *   GNU General Public License for more details.
 *
 *   You should have received a copy of the GNU General Public License
 *   along with this program.  If not, see <http://www.gnu.org/licenses/>.
 *
 *   Main Contact:
 *
 *   Nils Jansen
 *   Theory of Hybrid Systems
 *   RWTH Aachen
 *   52056 Aachen
 *   Germany
 *   nils.jansen@cs.rwth-aachen.de
 *
 */

/*
 * JavaInterface.cpp
 *
 *  Created on: 13.10.2011
 *      Author: jens
 */

#include <iostream>
#include <fstream>
#include "comics_tools_counterexample_JniMethods.h"
#include "io/Configuration.h"
#include "io/IOWrapper.h"

using namespace std;
using namespace scc_cex;

/** Utilitiy functions, JNI initialization */

JavaVM *cached_jvm;

JNIEXPORT jint JNICALL
JNI_OnLoad(JavaVM *jvm, void *reserved)
{
	cached_jvm = jvm; /* cache the JavaVM pointer */
	return JNI_VERSION_1_2;
}

jvalue JNU_CallMethodByName(JNIEnv *env, jboolean *hasException, jobject obj, const char *name, const char *descriptor, ...) {
	va_list args;
	jclass clazz;
	jmethodID mid;
	jvalue result;
	if ((*env).EnsureLocalCapacity(2) == JNI_OK) {
		clazz = (*env).GetObjectClass(obj);
		mid = (*env).GetMethodID(clazz, name, descriptor);
		if (mid) {
			const char *p = descriptor;
			/* skip over argument types to find out the
			 return type */
			while (*p != ')')
				p++;
			/* skip ')' */
			p++;
			va_start(args, descriptor);
			switch (*p) {
			case 'V':
				(*env).CallVoidMethodV(obj, mid, args);
				break;
			case '[':
			case 'L':
				result.l = (*env).CallObjectMethodV(obj, mid, args);
				break;
			case 'Z':
				result.z = (*env).CallBooleanMethodV(obj, mid, args);
				break;
			case 'B':
				result.b = (*env).CallByteMethodV(obj, mid, args);
				break;
			case 'C':
				result.c = (*env).CallCharMethodV(obj, mid, args);
				break;
			case 'S':
				result.s = (*env).CallShortMethodV(obj, mid, args);
				break;
			case 'I':
				result.i = (*env).CallIntMethodV(obj, mid, args);
				break;
			case 'J':
				result.j = (*env).CallLongMethodV(obj, mid, args);
				break;
			case 'F':
				result.f = (*env).CallFloatMethodV(obj, mid, args);
				break;
			case 'D':
				result.d = (*env).CallDoubleMethodV(obj, mid, args);
				break;
			default:
				(*env).FatalError("illegal descriptor");
			}
			va_end(args);
		}
		(*env).DeleteLocalRef(clazz);
	}
	if (hasException) {
		*hasException = (*env).ExceptionCheck();
	}
	return result;
}

JNIEnv *JNU_GetEnv() {
	JNIEnv *env;
	(*cached_jvm).GetEnv((void **) &env, JNI_VERSION_1_2);
	return env;
}

/** End: Utilitiy functions, JNI initialization */

/** The wrapper class that handles that manages the state (i.e. the ModelChecker & CEx-Generator) */

class JniWrapper {
public:
	jobject back_ptr; //back pointer to peer instance
	IOWrapper* wrapper;
	Graph<PTYPE>* mcresult;
	CexGenerator* cexgen;
	CounterExampleParams* task;
	bool modelCheckBefore;

	~JniWrapper() {
		mcresult = NULL;
		cexgen = NULL;
		delete task;
		delete wrapper;
	}
};

/** End: Wrapper class */

/** JNI methods */

JNIEXPORT jlong JNICALL Java_comics_tools_counterexample_JniMethods_create(JNIEnv * env, jobject self) {
	JniWrapper *cpp_obj = new JniWrapper();
	cpp_obj->back_ptr = env->NewGlobalRef(self);

	return (jlong) cpp_obj;
}

JNIEXPORT void JNICALL Java_comics_tools_counterexample_JniMethods_destroy
(JNIEnv * env, jobject self, jlong peer) {
	JniWrapper* jniw = (JniWrapper*)peer;

	if (jniw->cexgen != NULL) {
		fstream outputFile("counter_example_summary.txt", fstream::out);
		outputFile << jniw->cexgen->getSummary();
		outputFile.close();
	}

	env->DeleteGlobalRef(jniw->back_ptr);
	delete jniw;

	return;
}

//jboolean initIOWrapper(JNIEnv * env, jlong peer, jstring input_string, const unsigned int input_flags) {
//	const char *str;
//	str = (*env).GetStringUTFChars(input_string, NULL);
//	if (str == NULL) {
//		return JNI_FALSE; /* OutOfMemoryError already thrown */
//	}
//
//	JniWrapper* jniw = (JniWrapper*) peer;
//	jniw->wrapper = new IOWrapper(str, input_flags);
//
//	(*env).ReleaseStringUTFChars(input_string, str);
//
//	jniw->modelCheckBefore = false;
//
//	return jniw->wrapper->readFiles() ? JNI_TRUE : JNI_FALSE;
//}

jboolean initIOWrapper(JNIEnv * env, jlong peer, jstring input_string) {
	const char *str;
	str = (*env).GetStringUTFChars(input_string, NULL);
	if (str == NULL) {
		return JNI_FALSE; /* OutOfMemoryError already thrown */
	}

	JniWrapper* jniw = (JniWrapper*) peer;
	//jniw->wrapper = new IOWrapper(str, input_flags);
	jniw->wrapper = new IOWrapper(str);

	(*env).ReleaseStringUTFChars(input_string, str);

	jniw->modelCheckBefore = false;

	return jniw->wrapper->readFiles() ? JNI_TRUE : JNI_FALSE;
}

JNIEXPORT jboolean JNICALL Java_comics_tools_counterexample_JniMethods_loadFile(JNIEnv * env, jobject self, jlong peer, jstring filename) {
	//unsigned int input_flags = IOFL_WAIT_FOR_CALLS | IOFL_EXTERNAL_CALLER | IOFL_INPUT_IS_FILE | IOFL_DTMC_ONLY;
	//unsigned int input_flags = 0;
	ConfigurationSingleton* conf = ConfigurationSingleton::getInstance();
	conf->setIoflWaitForCalls(true);
	conf->setIoflExternalCaller(true);
	conf->setIoflInputIsFile(true);
	conf->setIoflDtmcOnly(true);
	//return initIOWrapper(env, peer, filename, input_flags);
	return initIOWrapper(env, peer, filename);
}

JNIEXPORT jboolean JNICALL Java_comics_tools_counterexample_JniMethods_loadData(JNIEnv * env, jobject self, jlong peer, jstring data) {
	//unsigned int input_flags = IOFL_WAIT_FOR_CALLS | IOFL_EXTERNAL_CALLER | IOFL_INPUT_IS_DATA | IOFL_DTMC_ONLY;
	//unsigned int input_flags = 0;
	ConfigurationSingleton* conf = ConfigurationSingleton::getInstance();
	conf->setIoflWaitForCalls(true);
	conf->setIoflExternalCaller(true);
	conf->setIoflInputIsData(true);
	conf->setIoflDtmcOnly(true);
	//return initIOWrapper(env, peer, data, input_flags);
	return initIOWrapper(env, peer, data);
}

JNIEXPORT jstring JNICALL Java_comics_tools_counterexample_JniMethods_modelCheck(JNIEnv * env, jobject self, jlong peer) {
	JniWrapper* jniw = (JniWrapper*) peer;
	jniw->mcresult = jniw->wrapper->runModelChecking();
	jniw->modelCheckBefore = true;

	return (*env).NewStringUTF(jniw->wrapper->mcResultToString(jniw->mcresult).c_str());
}

JNIEXPORT jdouble JNICALL Java_comics_tools_counterexample_JniMethods_getResultProbability(JNIEnv * env, jobject self, jlong peer) {
	JniWrapper* jniw = (JniWrapper*) peer;
	return jniw->wrapper->getResultProbability();
}

JNIEXPORT jboolean JNICALL Java_comics_tools_counterexample_JniMethods_initCounterExampleSearch(JNIEnv * env, jobject self, jlong peer, jstring xmlGraph, jdouble probBound, jboolean abstractFlag, jboolean globalFlag, jboolean closureFlag) {
	JniWrapper* jniw = (JniWrapper*) peer;

	if (!jniw->modelCheckBefore) {
		//Init IOWrapper but do not model check
		ConfigurationSingleton* conf = ConfigurationSingleton::getInstance();
		conf->setIoflWaitForCalls(true);
		conf->setIoflExternalCaller(true);
		conf->setIoflInputIsData(true);
		conf->setIoflDtmcOnly(true);
		conf->setIoflReadFromXml(true);
		string str = "";
		jniw->wrapper = new IOWrapper(str);
	}

	if ((double) probBound >= 0 && (double) probBound <= 100) {
		jniw->wrapper->setProbBound((double) probBound);
	}

	const char *xmlString;
	xmlString = (*env).GetStringUTFChars(xmlGraph, NULL);
	if (xmlString == NULL) {
		return JNI_FALSE; /* OutOfMemoryError already thrown */
	}

	Graph<PTYPE>* graph;
	if (jniw->modelCheckBefore) {
		//Use model check result
		graph = jniw->mcresult;
	} else {
		//Probability from model checking is set while parsing
		graph = jniw->wrapper->getGraphFromXml(xmlString);
	}

	(*env).ReleaseStringUTFChars(xmlGraph, xmlString);

	pair<CexGenerator*, CounterExampleParams*> initres = jniw->wrapper->initCexSearch(graph, abstractFlag, globalFlag, closureFlag);
	jniw->cexgen = initres.first;
	jniw->cexgen->setModelCheckingProbability(jniw->wrapper->getResultProbability());
	jniw->cexgen->setStepLimit(0);
	jniw->task = initres.second;

	return (jniw->cexgen != NULL && jniw->task != NULL) ? JNI_TRUE : JNI_FALSE;
}

JNIEXPORT jstring JNICALL Java_comics_tools_counterexample_JniMethods_doCounterExampleSteps(JNIEnv * env, jobject self, jlong peer, jintArray concretizedGraphsJava, jboolean autoConcretize, jboolean stepsTillEnd, jint nrOfSteps, jint concretizeMode, jint heuristic) {
	if (!autoConcretize) {
		nrOfSteps = 1;
	} else if (stepsTillEnd) {
		nrOfSteps = UINT_MAX;
		autoConcretize = true;
	}

	int size = 0;
	int no = 0;
	jsize len = env->GetArrayLength(concretizedGraphsJava);
	vector<bool> concretizedGraphs;
	jint *body = env->GetIntArrayElements(concretizedGraphsJava, 0);

	//Set bool vector accordingly
	for (int i = 0; i < len; i++) {
		no = body[i];
		size = concretizedGraphs.size();
		if (no >= size) {
			size = no - size;
			for (int j = 1; j <= size; j++) {
				concretizedGraphs.push_back(false);
			}
			concretizedGraphs.push_back(true);
		} else {
			concretizedGraphs[no] = true;
		}
	}
	env->ReleaseIntArrayElements(concretizedGraphsJava, body, 0);

	JniWrapper* jniw = (JniWrapper*) peer;
	//Configuration::getInstance().userInputConcretized = concretizedGraphs;
	//Configuration::getInstance().autoConcretize = autoConcretize;
	ConfigurationSingleton::getInstance()->userInputConcretized = concretizedGraphs;
	ConfigurationSingleton::getInstance()->autoConcretize = autoConcretize;

	if (concretizedGraphs.size() > 0 || autoConcretize) {
		jniw->task->mFilterMode = (FilterMode) heuristic;
		jniw->task->mConcMode = (ConcretizationMode) concretizeMode;
		jniw->wrapper->executeCexSteps(jniw->cexgen, jniw->task, (unsigned int) nrOfSteps);
		return (*env).NewStringUTF(jniw->cexgen->counterExampleToXMLString().c_str());
	} else {
		//Otherwise we have done nothing in this step
		string s = "No step";
		return (*env).NewStringUTF(s.c_str());
	}
}

JNIEXPORT jboolean JNICALL Java_comics_tools_counterexample_JniMethods_isCounterExampleComplete(JNIEnv * env, jobject self, jlong peer) {
	JniWrapper* jniw = (JniWrapper*) peer;
	return jniw->cexgen->isCexComplete() ? JNI_TRUE : JNI_FALSE;
}

JNIEXPORT jint JNICALL Java_comics_tools_counterexample_JniMethods_getCounterExampleSteps(JNIEnv * env, jobject self, jlong peer) {
	JniWrapper* jniw = (JniWrapper*) peer;
	return jniw->cexgen->getStepCount();
}

JNIEXPORT jdouble JNICALL Java_comics_tools_counterexample_JniMethods_getCounterExampleProbability(JNIEnv * env, jobject self, jlong peer) {
	JniWrapper* jniw = (JniWrapper*) peer;
	return jniw->cexgen->getProbCEX();
}

/** End: JNI methods */

