/**
 *   COMICS - Computing Minimal Counterexamples for Discrete-time Markov Chains
 *
 *   COMICS is a stand-alone tool which performs model checking and the generation
 *   of counterexamples for discrete-time Markov Chains (DTMCs). *
 *
 *   Copyright (C) <2012> <RWTH Aachen University>
 *   Authors: Nils Jansen, Erika Abraham, Jens Katelaan, Maik Scheffler, Matthias Volk, Andreas Vorpahl
 *
 *   This program is free software: you can redistribute it and/or modify
 *   it under the terms of the GNU General Public License as published by
 *   the Free Software Foundation, either version 3 of the License, or
 *   (at your option) any later version.
 *
 *   This program is distributed in the hope that it will be useful,
 *   but WITHOUT ANY WARRANTY; without even the implied warranty of
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *   GNU General Public License for more details.
 *
 *   You should have received a copy of the GNU General Public License
 *   along with this program.  If not, see <http://www.gnu.org/licenses/>.
 *
 *   Main Contact:
 *
 *   Nils Jansen
 *   Theory of Hybrid Systems
 *   RWTH Aachen
 *   52056 Aachen
 *   Germany
 *   nils.jansen@cs.rwth-aachen.de
 *
 */

/*
 * Path.cpp
 *
 *  Created on: 08.03.2011
 *      Author: jens
 */

#include <sstream>
#include "GraphTypes.h"

using namespace std;

namespace scc_cex {

Path::Path(Path* _pred, Vertex<PTYPE>* _node, Edge<PTYPE>* _edge) {
	k = 0;
	L = 0;
	pred = _pred;
	node = _node;
	edge = _edge;
}

Path::Path(const unsigned int _k, PTYPE _L, Path* _pred, Vertex<PTYPE>* _node, Edge<PTYPE>* _edge) {
	k = _k;
	L = _L;
	pred = _pred;
	node = _node;
	edge = _edge;

	terminateHere = false;
}

string Path::toString() const {
	const Path* path = this;
	stringstream ss;
	ss << path->node->oindex;
	while (path->pred != NULL) {
		path = path->pred;
		if (path->node == NULL) {
			ss << "No node object referenced by path\n";
			return ss.str();
		}

		string jup = "<-";
		if (path->edge != NULL) {
			if (path->edge->state) {
				jup = "<=";
			}
		}
		ss << jup << path->node->oindex;
		//ss << "(#)";

//		if (path->edge != NULL) {
//			if(!path->edge->state) {
//				ss << "<-" << path->edge->source->oindex;
//			} else {
//				ss << "<=" << path->edge->source->oindex;
//			}
//		}

		if (path->terminateHere) {
			if (path->edge != NULL) {
				if (!path->edge->state) {
					ss << "<-" << path->edge->source->oindex;
				} else {
					ss << "<=" << path->edge->source->oindex;
				}
			}
			break;
		}
	}
	return ss.str();
}

string Path::recursiveOutput(const Path* path) const {
	stringstream ss;
	if (path->pred != NULL) {
		ss << recursiveOutput(path->pred);
		ss << "->";
	}
	ss << path->node->oindex;

	return ss.str();
}

string Path::toStringOutput() const {
	const Path* path = this;
	stringstream ss;
	ss << recursiveOutput(path);

	ss << " (Prob: " << L << ")";
	return ss.str();
}

string Path::toStringDebug() const {
	stringstream ss;
	ss << toString();
	ss << " (k = " << k << "; prob: " << L << ")";
	return ss.str();
}

Path* Path::deepCopy() {
	Path* res = new Path(k, L, NULL, node, edge);
	if (pred != NULL && !terminateHere)
		res->pred = pred->deepCopy();
	else if (pred != NULL) {
		res->pred = new Path(pred->k, pred->L, NULL, pred->node, pred->edge);
		//res->pred = NULL;
	}
	return res;
}

void Path::deepDelete() {
	if (pred != NULL && !terminateHere) {
		pred->deepDelete();
		delete pred;
	}
}

Path::~Path() {
	pred = NULL;
	node = NULL;
	edge = NULL;
}

std::ostream &operator<<(std::ostream &stream, Path obj) {
	stream << obj.toStringDebug();
	return stream;
}

}
