/**
 *   COMICS - Computing Minimal Counterexamples for Discrete-time Markov Chains
 *
 *   COMICS is a stand-alone tool which performs model checking and the generation
 *   of counterexamples for discrete-time Markov Chains (DTMCs). *
 *
 *   Copyright (C) <2012> <RWTH Aachen University>
 *   Authors: Nils Jansen, Erika Abraham, Jens Katelaan, Maik Scheffler, Matthias Volk, Andreas Vorpahl
 *
 *   This program is free software: you can redistribute it and/or modify
 *   it under the terms of the GNU General Public License as published by
 *   the Free Software Foundation, either version 3 of the License, or
 *   (at your option) any later version.
 *
 *   This program is distributed in the hope that it will be useful,
 *   but WITHOUT ANY WARRANTY; without even the implied warranty of
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *   GNU General Public License for more details.
 *
 *   You should have received a copy of the GNU General Public License
 *   along with this program.  If not, see <http://www.gnu.org/licenses/>.
 *
 *   Main Contact:
 *
 *   Nils Jansen
 *   Theory of Hybrid Systems
 *   RWTH Aachen
 *   52056 Aachen
 *   Germany
 *   nils.jansen@cs.rwth-aachen.de
 *
*/

/*
 * AbstractStateSelector.cpp
 *
 *  Created on: 18.03.2011
 *      Author: jens
 */

#include <math.h>
#ifdef USE_LOGLIB
#include <log4cplus/logger.h>
#endif

#include "AbstractStateSelector.h"

using namespace std;
using namespace log4cplus;

namespace scc_cex {

Logger sellogger = Logger::getInstance("CexGen.StateSelector");

AbstractStateSelector::AbstractStateSelector(Graph<PTYPE>* abstractGraph) {
	mAbstGraph = abstractGraph;

	sellogger.setLogLevel(DEFAULT_LOG_LEVEL);
}

AbstractStateSelector::~AbstractStateSelector() {
	mAbstGraph = NULL;
}

bool AbstractStateSelector::areThereAbstractStates() {
	return getAllUnconcretized().size() > 0;
}

graph_vector AbstractStateSelector::chooseAbstractStates(const ConcretizationMode concMode, const StateSelectionMode selectionMode, const FilterMode filterMode) {
	//Step 1: Select all candidates matching the desired criterion
	graph_vector candidates;
	switch (selectionMode) {
	case SELECT_HIGHEST:
		candidates = getHighestUnconcretized();
		break;
	case SELECT_DEEPEST:
		candidates = getDeepestUnconcretized();
		break;
	case SELECT_USER_INPUT:
		candidates = getUserInputUnconcretized();
		//We don't want to change anything, because this is the user input
		return candidates;
		break;
	default:
		candidates = getAllUnconcretized();
		break;
	}

	//Step 2: Invoke the desired filter
	switch (filterMode) {
	case FILTER_SCC_SIZE:
		candidates = sortByDecreasingSize(candidates, false);
		break;
	case FILTER_ABSTRACT_EDGE_PROBABILITY:
		candidates = sortByDecreasingProbability(candidates, false);
		break;
	case FILTER_INPUT_OUTPUT_DEGREE:
		candidates = sortByIncreasingDegree(candidates, false);
		break;
	case FILTER_RELATIVE_USAGE:
		candidates = sortByDecreasingRelativeUsage(candidates, false);
		break;
	default:
		//The none case -> don't do any sorting
		break;
	}

	double root;
	unsigned int desiredSize;

	switch (concMode) {
	case CONCRETIZE_ONE:
		//Concretize only one in each step
		if (candidates.size() > 0) candidates.resize(1);
		break;
	case CONCRETIZE_SQRT:
		//Square-root-criterion
		root = sqrt((double) candidates.size());
		desiredSize = (unsigned int) ceil(root);
		candidates.resize(desiredSize);
		break;
	case CONCRETIZE_ALL:
		//Concretize all in each step
		break;
	}

	//	if (candidates.size() > preferredNumber)
	//		candidates.resize(preferredNumber);

	return candidates;
}

graph_vector AbstractStateSelector::getHighestUnconcretized() const {
	LOG4CPLUS_DEBUG(sellogger, "Searching for HIGHEST unconcretized");

	graph_vector result;
	graph_vector candidates;
	graph_vector nextlevel;
	nextlevel.push_back(mAbstGraph);

	while (result.empty()) {
		for (graph_it it = nextlevel.begin(); it != nextlevel.end(); it++) {
			Graph<PTYPE>* state = *it;

			if (state->concretized) {
				//The block recursion flag signals that this whole subtree is unreachable, so don't add as candidate if flag is set
				if (!state->blockRecursion) {
					candidates.insert(candidates.end(), state->subgraphs.begin(), state->subgraphs.end());
					LOG4CPLUS_DEBUG(sellogger, "Added subgraphs of " << state->graph_id << " to abstract state search");
				}
			}
			else {
				//Subgraph wasn't concretized yet -> add to result if reachable
				//But need to check if it can actually be reached
				bool canUse = mAbstGraph->isReachableInMarking(state);
				if (canUse) {
					result.push_back(state);
					LOG4CPLUS_DEBUG(sellogger, "Added " << state->graph_id << " to result");
				}
				else {
					state->concretized = true; //So that we won't check again
					state->blockRecursion = true; //So that the whole subtree is excluded
					LOG4CPLUS_DEBUG(sellogger, "Will skip concretization of " << (*it)->graph_id << " because it is unreachable");
				}
			}
		}

		nextlevel = candidates;
		candidates.clear();
		if (nextlevel.empty()) break;
	}

	return result;
}

graph_vector AbstractStateSelector::getAllUnconcretized() const {
	LOG4CPLUS_DEBUG(sellogger, "Searching for ALL unconcretized");

	graph_vector result;
	graph_vector candidates;
	graph_vector nextlevel;
	nextlevel.push_back(mAbstGraph);

	while (!nextlevel.empty()) {
		for (graph_it it = nextlevel.begin(); it != nextlevel.end(); it++) {
			Graph<PTYPE>* state = *it;

			if (state->concretized) {
				//The block recursion flag signals that this whole subtree is unreachable, so don't add as candidate if flag is set
				if (!state->blockRecursion) {
					candidates.insert(candidates.end(), state->subgraphs.begin(), state->subgraphs.end());
					LOG4CPLUS_DEBUG(sellogger, "Added subgraphs of " << state->graph_id << " to abstract state search");
				}
			}
			else {
				//Subgraph wasn't concretized yet -> add to result if reachable
				//But need to check if it can actually be reached
				bool canUse = mAbstGraph->isReachableInMarking(state);
				if (canUse) {
					result.push_back(state);
					LOG4CPLUS_DEBUG(sellogger, "Added " << state->graph_id << " to result");
				}
				else {
					state->concretized = true; //So that we won't check again
					state->blockRecursion = true; //So that the whole subtree is excluded
					LOG4CPLUS_DEBUG(sellogger, "Will skip concretization of " << (*it)->graph_id << " because it is unreachable");
				}
			}
		}

		nextlevel = candidates;
		candidates.clear();
	}

	return result;
}

graph_vector AbstractStateSelector::getDeepestUnconcretized() const {
	LOG4CPLUS_DEBUG(sellogger, "Searching for DEEPEST unconcretized");

	graph_vector tmpresult;
	graph_vector result;
	graph_vector candidates;
	graph_vector nextlevel;
	nextlevel.push_back(mAbstGraph);

	while (!nextlevel.empty()) {
		for (graph_it it = nextlevel.begin(); it != nextlevel.end(); it++) {
			Graph<PTYPE>* state = *it;

			if (state->concretized) {
				//The block recursion flag signals that this whole subtree is unreachable, so don't add as candidate if flag is set
				if (!state->blockRecursion) {
					candidates.insert(candidates.end(), state->subgraphs.begin(), state->subgraphs.end());
					LOG4CPLUS_DEBUG(sellogger, "Added subgraphs of " << state->graph_id << " to abstract state search");
				}
			}
			else {
				//Subgraph wasn't concretized yet -> add to result if reachable
				//But need to check if it can actually be reached
				bool canUse = mAbstGraph->isReachableInMarking(state);
				if (canUse) {
					tmpresult.push_back(state);
				}
				else {
					state->concretized = true; //So that we won't check again
					state->blockRecursion = true; //So that the whole subtree is excluded
					LOG4CPLUS_DEBUG(sellogger, "Will skip concretization of " << state->graph_id << " because it is unreachable");
				}
			}
		}

		if (!tmpresult.empty()) result = tmpresult;
		nextlevel = candidates;
		candidates.clear();
		tmpresult.clear();
	}

	return result;
}

graph_vector AbstractStateSelector::getUserInputUnconcretized() const {
	LOG4CPLUS_DEBUG(sellogger, "Searching for USER INPUT unconcretized");

	vector<bool> userInput = ConfigurationSingleton::getInstance()->userInputConcretized;
	graph_vector result;
	graph_vector candidates;
	graph_vector nextlevel;
	nextlevel.push_back(mAbstGraph);

	stringstream s;
	s << "UserInput: ";
	for (int i = 0; i < userInput.size(); i++) {
		s << userInput[i];
	}
	LOG4CPLUS_DEBUG(sellogger, s.str());

	while (!nextlevel.empty()) {
		for (graph_it it = nextlevel.begin(); it != nextlevel.end(); it++) {
			Graph<PTYPE>* state = *it;

			if (state->concretized) {
				//The block recursion flag signals that this whole subtree is unreachable, so don't add as candidate if flag is set
				if (!state->blockRecursion) {
					candidates.insert(candidates.end(), state->subgraphs.begin(), state->subgraphs.end());
					LOG4CPLUS_DEBUG(sellogger, "Added subgraphs of " << state->graph_id << " to abstract state search");
				}
			} else {
				//Subgraph wasn't concretized yet -> add to result if reachable
				//But need to check if it can actually be reached
				bool canUse = mAbstGraph->isReachableInMarking(state);
				if (canUse) {

					if (state->graph_id < userInput.size() && userInput[state->graph_id]) {
						//Add graph and consider subgraphs
						result.push_back(state);
						LOG4CPLUS_DEBUG(sellogger, "Added " << state->graph_id << " to result");
					}
				} else {
					state->concretized = true; //So that we won't check again
					state->blockRecursion = true; //So that the whole subtree is excluded
					LOG4CPLUS_DEBUG(sellogger,
							"Will skip concretization of " << (*it)->graph_id << " because it is unreachable");
				}
			}
		}

		nextlevel = candidates;
		candidates.clear();
	}

	//Modify user input, because we will concretize some graphs
	for (graph_it it = result.begin(); it != result.end(); it++) {
		Graph<PTYPE>* graph = *it;
		userInput[graph->graph_id] = false;
	}
	ConfigurationSingleton::getInstance()->userInputConcretized = userInput;

	return result;
}


bool bySize (Graph<PTYPE>* g1, Graph<PTYPE>* g2) { return (g1->allVertices.size() > g2->allVertices.size()); }

bool byDegree (Graph<PTYPE>* g1, Graph<PTYPE>* g2) {
	unsigned int d1 = g1->inputVertices.size() * g1->outputVertices.size();
	unsigned int d2 = g2->inputVertices.size() * g2->outputVertices.size();
	return (d1 < d2);
}

double relativeUsage(vtx_vector vertices) {
	double denom = (double) vertices.size();
	double nom = 0;

	for (vtx_it it = vertices.begin(); it != vertices.end(); it++) {
		if ((*it)->isInClosure) nom++;
	}

	return nom / denom;
}

bool byRelativeUsage (Graph<PTYPE>* g1, Graph<PTYPE>* g2) {

	double relativeUsage1 = relativeUsage(g1->inputVertices) + relativeUsage(g1->outputVertices);
	double relativeUsage2 = relativeUsage(g2->inputVertices) + relativeUsage(g2->outputVertices);

	return (relativeUsage1 > relativeUsage2);
}

PTYPE probabilityMean(Graph<PTYPE>* g) {
	PTYPE probsum = 0;
	PTYPE count = 0;

	for (vtx_it it = g->inputVertices.begin(); it != g->inputVertices.end(); it++) {
		if ((*it)->isInClosure) {
			Vertex<PTYPE>* vtx = *it;

			for (edge_it e_it = vtx->outEdges.begin(); e_it != vtx->outEdges.end(); e_it++) {
				if ((*e_it)->target->isInClosure) {
					count++;
					probsum += (*e_it)->prob;
				}
			}
		}
	}

	return probsum / count;
}

bool byProbability (Graph<PTYPE>* g1, Graph<PTYPE>* g2) {

	PTYPE probMean1 = probabilityMean(g1);
	PTYPE probMean2 = probabilityMean(g2);

	return (probMean1 > probMean2);
}

graph_vector abstractFilter(graph_vector candidates, bool inverseSequence, bool (*func)(Graph<PTYPE>* g1, Graph<PTYPE>* g2)) {
	if (inverseSequence) {
		sort(candidates.rbegin(), candidates.rend(), func);
	}
	else {
		sort(candidates.begin(), candidates.end(), func);
	}

	return candidates;
}

graph_vector AbstractStateSelector::sortByDecreasingProbability(graph_vector candidates, const bool inverseSequence) const {
	candidates = abstractFilter(candidates, inverseSequence, byProbability);
	return candidates;
}

graph_vector AbstractStateSelector::sortByDecreasingSize(graph_vector candidates, const bool inverseSequence) const {
	candidates = abstractFilter(candidates, inverseSequence, bySize);
	return candidates;
}

graph_vector AbstractStateSelector::sortByIncreasingDegree(graph_vector candidates, const bool inverseSequence) const {
	candidates = abstractFilter(candidates, inverseSequence, byDegree);
	return candidates;
}

graph_vector AbstractStateSelector::sortByDecreasingRelativeUsage(graph_vector candidates, const bool inverseSequence) const {
	candidates = abstractFilter(candidates, inverseSequence, byRelativeUsage);
	return candidates;
}

}
