/**
 *   COMICS - Computing Minimal Counterexamples for Discrete-time Markov Chains
 *
 *   COMICS is a stand-alone tool which performs model checking and the generation
 *   of counterexamples for discrete-time Markov Chains (DTMCs). *
 *
 *   Copyright (C) <2012> <RWTH Aachen University>
 *   Authors: Nils Jansen, Erika Abraham, Jens Katelaan, Maik Scheffler, Matthias Volk, Andreas Vorpahl
 *
 *   This program is free software: you can redistribute it and/or modify
 *   it under the terms of the GNU General Public License as published by
 *   the Free Software Foundation, either version 3 of the License, or
 *   (at your option) any later version.
 *
 *   This program is distributed in the hope that it will be useful,
 *   but WITHOUT ANY WARRANTY; without even the implied warranty of
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *   GNU General Public License for more details.
 *
 *   You should have received a copy of the GNU General Public License
 *   along with this program.  If not, see <http://www.gnu.org/licenses/>.
 *
 *   Main Contact:
 *
 *   Nils Jansen
 *   Theory of Hybrid Systems
 *   RWTH Aachen
 *   52056 Aachen
 *   Germany
 *   nils.jansen@cs.rwth-aachen.de
 *
*/

/*
 * Closure.cpp
 *
 *  Created on: 09.03.2011
 *      Author: jens
 */

#include <vector>
#include <set>

#include "Closure.h"
#include "../io/Configuration.h"
#include "../mc/SCC_MC.h"

using namespace std;
using namespace log4cplus;

namespace scc_cex {

Logger cllogger = Logger::getInstance("CexGen.Closure");

bool resetClosure = false;

Closure::Closure(Vertex<PTYPE>* source, Vertex<PTYPE>* target, AbstractExtensionTracker* tracker) {
	mSource = source;
	mTarget = target;
	mTracker = tracker;

	closureGraph = NULL;

	cllogger.setLogLevel(DEFAULT_LOG_LEVEL);
}

PTYPE Closure::getProbMass() {
	computeByModelChecking();

	return mProbMass;
}

void Closure::computeByModelChecking() {

	LOG4CPLUS_TRACE(cllogger, "Checking closure prob mass of current graph...");

	SccMc subMc(false);

	LOG4CPLUS_TRACE(cllogger, "Starting closure computation");

	refineClosure();

	LOG4CPLUS_TRACE(cllogger, "Finished closure computation");

	//IOWrapper::writeResult(closureGraph, mSource->sisterVtx, "LOGGING");

	subMc.modelCheck(closureGraph);

	//IOWrapper::writeResult(closureGraph, mSource->sisterVtx, "LOGGING");

	mProbMass = subMc.getTargetResult(closureGraph->outputVertices[0]);
}

Vertex<PTYPE>* Closure::copyVertex(Vertex<PTYPE>* vtx) const {
	Vertex<PTYPE>* newVtx = new Vertex<PTYPE>();
	newVtx->oindex = vtx->oindex;

	vtx->sisterVtx = newVtx;
	return newVtx;
}

Edge<PTYPE>* Closure::copyEdge(Edge<PTYPE>* edge) const  {
	Edge<PTYPE>* newEdge = new Edge<PTYPE>(edge->source->sisterVtx, edge->target->sisterVtx, edge->prob);

	edge->sisterEdge = newEdge;
	return newEdge;
}

//Note: Not returning a reference on purpose
void Closure::refineClosure() {
	static Graph<PTYPE> staticGraph;

	if (resetClosure) {
		LOG4CPLUS_TRACE(cllogger, "Will reset closure");

		Graph<PTYPE> newG;
		staticGraph = newG;
		resetClosure = false;
	}

	closureGraph = &staticGraph;

	if (closureGraph->inputVertices.empty()) {
		//Haven't called this before, so initialization necessary
		mSource->isInClosure = true;
		Vertex<PTYPE>* src = copyVertex(mSource);
		src->isInputVertex = true;
		closureGraph->inputVertices.push_back(src);
		closureGraph->allVertices.push_back(src);

		LOG4CPLUS_TRACE(cllogger, "Have copied input " << (*src));

		mTarget->isInClosure = true;
		Vertex<PTYPE>* trg = copyVertex(mTarget);
		closureGraph->outputVertices.push_back(trg);

		LOG4CPLUS_TRACE(cllogger, "Have copied target " << (*trg));
	}
	else {
		closureGraph->state = CONCRETE;
		closureGraph->resetStructureAfterModelChecking();
	}

	//To get rid of previous model checking results
	for (vtx_it it = closureGraph->allVertices.begin(); it != closureGraph->allVertices.end(); it++) {
		(*it)->seen = false;
		(*it)->weight = -1.0;
		(*it)->dfs = 0;
		(*it)->lowlink=0;
		(*it)->isInputVertex = false;
		(*it)->outEdges.cleanUpEdges(false);
	}
	for (vtx_it it = closureGraph->allVertices.begin(); it != closureGraph->allVertices.end(); it++) {
		(*it)->inEdges.cleanUpEdges(true);
	}

	for (vtx_it it = closureGraph->outputVertices.begin(); it != closureGraph->outputVertices.end(); it++) {
		(*it)->seen = false;
		(*it)->weight = -1.0;
		(*it)->dfs = 0;
		(*it)->lowlink=0;
		(*it)->inEdges.cleanUpEdges(true);
	}
	closureGraph->inputVertices[0]->isInputVertex = true;

	//Insert new edges into the closure
	std::set<Edge<PTYPE>*>& newEdges = mTracker->getExtension();

	for (set<Edge<PTYPE>*>::iterator it = newEdges.begin(); it != newEdges.end(); it++) {
		Edge<PTYPE>* edge = *it;

		if (!edge->source->isInClosure) {
			//Copy source
			closureGraph->allVertices.push_back(copyVertex(edge->source));
			edge->source->isInClosure = true;

			LOG4CPLUS_TRACE(cllogger, "Have copied source of edge " << (*edge->source));
		}
		if (!edge->target->isInClosure) {
			//Copy target
			closureGraph->allVertices.push_back(copyVertex(edge->target));
			edge->target->isInClosure = true;

			LOG4CPLUS_TRACE(cllogger, "Have copied target of edge " << (*edge->target));
		}
		//Copy edge
		copyEdge(edge);

		LOG4CPLUS_TRACE(cllogger, "Have copied edge " << (*edge));
	}

	LOG4CPLUS_DEBUG(cllogger, "Starting preprocessing");


	Vertex<PTYPE>* vtxDummy;
	if (closureGraph->outputVertices.size() <= 1) {
		vtxDummy = new Vertex<PTYPE>();
		vtxDummy->oindex = UINT_MAX;
		closureGraph->outputVertices.push_back(vtxDummy);
	}
	else{
		vtxDummy = closureGraph->outputVertices[1];
	}

	for (vtx_it it = closureGraph->allVertices.begin(); it != closureGraph->allVertices.end(); it++) {
		//Must set graph reference for tarjan...
		(*it)->g = closureGraph;

		//...and create dummy edges to get a correct DTMC
		PTYPE sum = 0;
		for (edge_it e_it = (*it)->outEdges.begin(); e_it != (*it)->outEdges.end(); e_it++) {
			sum += (*e_it)->prob;
		}
		if (sum < 1) {
			new Edge<PTYPE>(*it, vtxDummy, 1-sum);
			LOG4CPLUS_TRACE(cllogger, "Dummy edge with prob " << (1-sum) << " added to node " << (**it));
		}
		else {
			LOG4CPLUS_TRACE(cllogger, "Won't add dummy edge to node " << (**it) << ", sum is already " << sum);
		}
	}

	LOG4CPLUS_DEBUG(cllogger, "Preprocessing done, will now modelcheck");
}

Closure::~Closure() {
	mGraph = NULL;
}

}
