/**
 *   COMICS - Computing Minimal Counterexamples for Discrete-time Markov Chains
 *
 *   COMICS is a stand-alone tool which performs model checking and the generation
 *   of counterexamples for discrete-time Markov Chains (DTMCs). *
 *
 *   Copyright (C) <2012> <RWTH Aachen University>
 *   Authors: Nils Jansen, Erika Abraham, Jens Katelaan, Maik Scheffler, Matthias Volk, Andreas Vorpahl
 *
 *   This program is free software: you can redistribute it and/or modify
 *   it under the terms of the GNU General Public License as published by
 *   the Free Software Foundation, either version 3 of the License, or
 *   (at your option) any later version.
 *
 *   This program is distributed in the hope that it will be useful,
 *   but WITHOUT ANY WARRANTY; without even the implied warranty of
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *   GNU General Public License for more details.
 *
 *   You should have received a copy of the GNU General Public License
 *   along with this program.  If not, see <http://www.gnu.org/licenses/>.
 *
 *   Main Contact:
 *
 *   Nils Jansen
 *   Theory of Hybrid Systems
 *   RWTH Aachen
 *   52056 Aachen
 *   Germany
 *   nils.jansen@cs.rwth-aachen.de
 *
*/

/*
 * CounterExampleParams.cpp
 *
 *  Created on: 22.03.2011
 *      Author: jens
 */

#include "CounterExampleParams.h"
#include <sstream>

using namespace std;

namespace scc_cex {

CounterExampleParams::CounterExampleParams(const bool doGlobalSearch, const bool doAbstractSearch,
		const ConcretizationMode concMode, const StateSelectionMode selectionMode, const FilterMode filterMode) {
	mUseGlobalSearch = doGlobalSearch;
	mUseAbstractSearch = doAbstractSearch;
	mConcMode = concMode;
	mSelectionMode = selectionMode;
	mFilterMode = filterMode;
}

ostream& operator<<(std::ostream& os, const CounterExampleParams& rhs) {
	os << "Counter Example [";
	os << "Global = " << rhs.mUseGlobalSearch;
	os << ", Abstract = " << rhs.mUseAbstractSearch;
	os << ", Concretization Mode = " << concModeToString(rhs.mConcMode);
	os << ", State Selection Mode = " << selectionModeToString(rhs.mSelectionMode);
	os << ", Filter Mode = " << filterModeToString(rhs.mFilterMode);
	os << "]";
	return os;
}

string CounterExampleParams::getParamHash() {
	stringstream ss;
	ss << (mUseGlobalSearch ? "G_" : "L_");
	ss << (mUseAbstractSearch ? "A_" : "C_");
	ss << mConcMode << "_";
	ss << mSelectionMode << "_";
	ss << mFilterMode;
	return ss.str();
}

std::string selectionModeToString(const StateSelectionMode m)  {
	switch (m) {
	case SELECT_HIGHEST: return "Select Top-Most";
	case SELECT_DEEPEST: return "Select Down-Most";
	default:            return "Select All Visible";
	}
}

std::string filterModeToString(const FilterMode m) {
	switch (m) {
	case FILTER_SCC_SIZE:
		return "Largest SCCs";
	case FILTER_ABSTRACT_EDGE_PROBABILITY:
		return "SCCs with highest abstract probabilities";
	case FILTER_INPUT_OUTPUT_DEGREE:
		return "SCCs with lowest input/output degrees";
	case FILTER_RELATIVE_USAGE:
		return "SCCs of highest relative closure membership";
	default:
		return "None";
	}
}

std::string concModeToString(const ConcretizationMode m) {
	switch (m) {
	case CONCRETIZE_SQRT: return "SQRT(#CANDIDATES)";
	case CONCRETIZE_ONE:  return "1";
	default:              return "#CANDIDATES";
	}
}

}
