/**
 *   COMICS - Computing Minimal Counterexamples for Discrete-time Markov Chains
 *
 *   COMICS is a stand-alone tool which performs model checking and the generation
 *   of counterexamples for discrete-time Markov Chains (DTMCs). *
 *
 *   Copyright (C) <2012> <RWTH Aachen University>
 *   Authors: Nils Jansen, Erika Abraham, Jens Katelaan, Maik Scheffler, Matthias Volk, Andreas Vorpahl
 *
 *   This program is free software: you can redistribute it and/or modify
 *   it under the terms of the GNU General Public License as published by
 *   the Free Software Foundation, either version 3 of the License, or
 *   (at your option) any later version.
 *
 *   This program is distributed in the hope that it will be useful,
 *   but WITHOUT ANY WARRANTY; without even the implied warranty of
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *   GNU General Public License for more details.
 *
 *   You should have received a copy of the GNU General Public License
 *   along with this program.  If not, see <http://www.gnu.org/licenses/>.
 *
 *   Main Contact:
 *
 *   Nils Jansen
 *   Theory of Hybrid Systems
 *   RWTH Aachen
 *   52056 Aachen
 *   Germany
 *   nils.jansen@cs.rwth-aachen.de
 *
*/

/*
 * CriticalSubSystem.cpp
 *
 *  Created on: 03.06.2011
 *      Author: jens
 */

#ifdef USE_LOGLIB
#include <log4cplus/logger.h>
#endif
#include "CriticalSubSystem.h"
#include "Closure.h"
#include "extrack/ContentExtensionTracker.h"
#include "extrack/PointerExtensionTracker.h"

using namespace std;
using namespace log4cplus;

namespace scc_cex {

Logger subsyslogger = Logger::getInstance("CriticalSubSystem");

CriticalSubSystem::CriticalSubSystem(Vertex<PTYPE>* cexInputNode, Vertex<PTYPE>* cexTargetNode) {
	mCexInputNode = cexInputNode;
	mCexTargetNode = cexTargetNode;
	subsyslogger.setLogLevel(DEFAULT_LOG_LEVEL);
	mTracker = new ContentExtensionTracker();
}

CriticalSubSystem::~CriticalSubSystem() {
	if (mTracker != NULL) delete mTracker;
}

AbstractExtensionTracker* CriticalSubSystem::getTracker() {
	return mTracker;
}

/**
 * If and only if you can guarantee that each edge is only created once
 * and the corresponding pointers always point to the same address,
 * you can enable pointer based edge tracking. This yields a speedup.
 */
void CriticalSubSystem::setPointerBasedEdgeTracking(const bool enable) {
	if (enable) mTracker = new PointerExtensionTracker();
	else mTracker = new ContentExtensionTracker();
}

/*
 * Generate the closure based on the current visibility. Return its prob mass
 */
PTYPE CriticalSubSystem::generateClosure() {
	LOG4CPLUS_DEBUG(subsyslogger, "Will generate closure now...");
	Closure closure(mCexInputNode, mCexTargetNode, mTracker);

	PTYPE mass = closure.getProbMass();

	if (mTracker == NULL) LOG4CPLUS_ERROR(subsyslogger, "No tracker set");
	mTracker->signalClosureExtensionFinished();

	return mass;
}

/*
 * Extending m_min is removing the not_m_min flag from edges in pi.
 * Returns true if there was a big enough extension that we need a new closure
 */
bool CriticalSubSystem::addPathToSubSystem(Path* pi) {
	if (mTracker == NULL) LOG4CPLUS_ERROR(subsyslogger, "No tracker set");

	mTracker->processPath(pi);
	return mTracker->isClosureExtensionPending();
}

/*
 * After big enough closure (with regard to probability) was found:
 * Discard all edges we didn't need to prepare for the next concretization step
 */
void CriticalSubSystem::removeUnusedEdges() {
	LOG4CPLUS_DEBUG(subsyslogger, "Setting m_min to m_max...");

	//That is: reset the tracking system
	//The tracking system will make sure that the edges won't be found again
	if (mTracker == NULL) LOG4CPLUS_ERROR(subsyslogger, "No tracker set");
	mTracker->signalResetTracking();

	LOG4CPLUS_DEBUG(subsyslogger, "m_min = m_max now");

	//We don't have to reset the not_m_min flag or anything:
	//the intersection of visible edges and not_m_min edges is empty at this point
}

void CriticalSubSystem::addEdgeToTrack(Edge<PTYPE>* e) {
	if (mTracker == NULL) LOG4CPLUS_ERROR(subsyslogger, "No tracker set");
	mTracker->addEdgeToTrack(e);
}

}
