/**
 *   COMICS - Computing Minimal Counterexamples for Discrete-time Markov Chains
 *
 *   COMICS is a stand-alone tool which performs model checking and the generation
 *   of counterexamples for discrete-time Markov Chains (DTMCs). *
 *
 *   Copyright (C) <2012> <RWTH Aachen University>
 *   Authors: Nils Jansen, Erika Abraham, Jens Katelaan, Maik Scheffler, Matthias Volk, Andreas Vorpahl
 *
 *   This program is free software: you can redistribute it and/or modify
 *   it under the terms of the GNU General Public License as published by
 *   the Free Software Foundation, either version 3 of the License, or
 *   (at your option) any later version.
 *
 *   This program is distributed in the hope that it will be useful,
 *   but WITHOUT ANY WARRANTY; without even the implied warranty of
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *   GNU General Public License for more details.
 *
 *   You should have received a copy of the GNU General Public License
 *   along with this program.  If not, see <http://www.gnu.org/licenses/>.
 *
 *   Main Contact:
 *
 *   Nils Jansen
 *   Theory of Hybrid Systems
 *   RWTH Aachen
 *   52056 Aachen
 *   Germany
 *   nils.jansen@cs.rwth-aachen.de
 *
*/

/*
 * Statistics.cpp
 *
 *  Created on: 17.03.2011
 *      Author: jens
 */

#include <cassert>
#include <sstream>
#include <iostream>
#include <ctime>
#include <time.h>
#include "Statistics.h"

using namespace std;

namespace scc_cex {

Statistics::Statistics() {
	mSearchParamsSet = false;
	mTaskSet = false;
	mConcretizationSet = false;

	mNrOfNodes = 0;
	mNrOfEdges = 0;
	mClosProb = 0;
}

Statistics::~Statistics() {
	// Nothing to do here right now
}

Statistics* Statistics::getInstance()
{
	  static Statistics instance;
	  return &instance;
}

string Statistics::generateSummary() const {
	//if (!mSearchParamsSet || mTaskSet || mConcretizationSet) {
	//	return "No data for counter example summary available.\nDid you actually generate a counter example?";
	//}

	unsigned int refinementSteps = getNumberOfSteps();

	stringstream ss;
	ss << "---------------------------------" << endl;
	ss << "----------- STATISTICS ----------" << endl;
	ss << "---------------------------------" << endl;
	ss << "Task: Counterexample for P(t)" << (mOp == OPERATOR_LEQ ? "<=" : "<") << mProbBound << endl;
	if (mNrOfNodes > 0) {
		ss << "Counter example size: " << mNrOfNodes << " states, " << mNrOfEdges << " transitions" << endl;
	}
	if (mMcResult > 0) {
		ss << "Model Checking result of original system: " << mMcResult << endl;
	}

	if (mClosProb > 0) {
		ss << "Closure probability: " << mClosProb << endl;
	}

	ss << "Time for model checking: " << (float)((float)(t_modelCheckerEnd - t_modelCheckerStart)/(float)CLOCKS_PER_SEC) << " secs" << endl;
	ss << "Time of counter example computation: " << (float)((float)(t_end-t_start)/(float)CLOCKS_PER_SEC) << " secs [Without Pre-/Post processing]" << endl;

	ss << "Search params: ";
	if (mDoAbstract) ss << "ABSTRACT, ";
	else ss << "CONCRETE, ";

	if (mDoGlobal) ss << "GLOBAL, ";
	else ss << "LOCAL, ";

	if (mDoClosures) ss << "CLOSURE-based";
	else ss << "PATH-based";
	ss << endl;

	ss << "Base node selection for concretization: " << selectionModeToString(mSelectionMode) << endl;
	ss << "Additional selection criterion: " << filterModeToString(mFilterMode) << endl;
	ss << "Number of concretizations per step: " << concModeToString(mConcMode) << endl;

	ss << "During the " << refinementSteps << " refinement steps the following number of shortest paths/closures were computed:" << endl;
	ss << "-----------------------------------------------" << endl;
	ss << "  step |      #paths |  #closures | #conc. scc" << endl;
	ss << "-----------------------------------------------" << endl;
	for (unsigned int i = 1; i <= refinementSteps; i++) {
		ss.width(6);
		ss << i;
		ss << " | ";

		ss.width(11);
		ss << getPathCountByStep(i);
		ss << " | ";

		ss.width(10);
		ss << getClosureCountByStep(i);
		ss << " | ";

		ss.width(11);
		ss << getConcretizedSccCountByStep(i);

		ss << endl;

	}
	ss << "-----------------------------------------------" << endl;

	unsigned int pathsum = 0;
	unsigned int closuresum = 0;
	unsigned int sccsum = 0;
	for (unsigned int i = 1; i <= refinementSteps; i++) {
		pathsum += getPathCountByStep(i);
		closuresum += getClosureCountByStep(i);
		sccsum += getConcretizedSccCountByStep(i);
	}

	ss << "Total #paths: " << pathsum << ", total #closures: " << closuresum << ", total #concretized SCCs: " << sccsum << endl;

	return ss.str();
}

void Statistics::addStep(const unsigned int nrOfPaths, const unsigned int nrOfClosures,
		const unsigned int concretizedScc)
{
	mPathCountByStep.push_back(nrOfPaths);
	mClosureCountByStep.push_back(nrOfClosures);
	mConcretizedSccByStep.push_back(concretizedScc);
}

void Statistics::startTime() {
	t_start = clock();
}

void Statistics::stopTime() {
	t_end = clock();
}

void Statistics::startModelCheckerTime() {
	t_modelCheckerStart = clock();
}

void Statistics::stopModelCheckerTime() {
	t_modelCheckerEnd = clock();
}

void Statistics::setSearchParams(const bool doGlobal, const bool doAbstract, const bool doClosures) {
	mSearchParamsSet = true;
	mDoGlobal = doGlobal;
	mDoAbstract = doAbstract;
	mDoClosures = doClosures;
}

void Statistics::setCounterExampleTask(const OperatorType op, const PTYPE probBound) {
	mTaskSet = true;
	mOp = op;
	mProbBound = probBound;
}

void Statistics::setCounterExampleSize(const unsigned int nrOfNodes, const unsigned int nrOfEdges) {
	mNrOfNodes = nrOfNodes;
	mNrOfEdges = nrOfEdges;
}

void Statistics::setConcretizationParams(const ConcretizationMode concMode,
		const StateSelectionMode selectionMode, const FilterMode filterMode)
{
	mConcretizationSet = true;
	mConcMode = concMode;
	mSelectionMode = selectionMode;
	mFilterMode = filterMode;
}

void Statistics::setProb(PTYPE prob) {
	mClosProb = prob;
}

PTYPE Statistics::getProb() {
	return mClosProb;
}

void Statistics::setMcResult(PTYPE prob) {
	mMcResult = prob;
}

unsigned int Statistics::getNumberOfSteps() const {
	assert(mClosureCountByStep.size() == mPathCountByStep.size());
	return mPathCountByStep.size();
}

unsigned int Statistics::getPathCountByStep(const unsigned int step) const {
	assert(mPathCountByStep.size() >= step);
	return mPathCountByStep[step-1];
}

unsigned int Statistics::getClosureCountByStep(const unsigned int step) const {
	assert(mClosureCountByStep.size() >= step);
	return mClosureCountByStep[step-1];
}

unsigned int Statistics::getConcretizedSccCountByStep(const unsigned int step) const {
	assert(mConcretizedSccByStep.size() >= step);
	return mConcretizedSccByStep[step-1];
}

}
