/**
 *   COMICS - Computing Minimal Counterexamples for Discrete-time Markov Chains
 *
 *   COMICS is a stand-alone tool which performs model checking and the generation
 *   of counterexamples for discrete-time Markov Chains (DTMCs). *
 *
 *   Copyright (C) <2012> <RWTH Aachen University>
 *   Authors: Nils Jansen, Erika Abraham, Jens Katelaan, Maik Scheffler, Matthias Volk, Andreas Vorpahl
 *
 *   This program is free software: you can redistribute it and/or modify
 *   it under the terms of the GNU General Public License as published by
 *   the Free Software Foundation, either version 3 of the License, or
 *   (at your option) any later version.
 *
 *   This program is distributed in the hope that it will be useful,
 *   but WITHOUT ANY WARRANTY; without even the implied warranty of
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *   GNU General Public License for more details.
 *
 *   You should have received a copy of the GNU General Public License
 *   along with this program.  If not, see <http://www.gnu.org/licenses/>.
 *
 *   Main Contact:
 *
 *   Nils Jansen
 *   Theory of Hybrid Systems
 *   RWTH Aachen
 *   52056 Aachen
 *   Germany
 *   nils.jansen@cs.rwth-aachen.de
 *
*/

/*
 * AbstractCexGen.cpp
 *
 *  Created on: 18.10.2011
 *      Author: jens
 */

#include "AbstractCexGen.h"

using namespace std;
using namespace log4cplus;

namespace scc_cex {

AbstractCexGen::~AbstractCexGen() {
	// Nothing to do here right now
}

/*
 * Search abstract counter example using global or local search, and closures
 */
//bool CexGenerator::searchAbstractCex(const bool doGlobalSearch) {
bool AbstractCexGen::searchCex() {
	LOG4CPLUS_DEBUG(cexlogger, "Entered abstract search...");
	bool userStep = false;
	//Check if step with user input is necessary
	vector<bool> userInput = ConfigurationSingleton::getInstance()->userInputConcretized;
	for (int i=0; i< userInput.size(); i++){
		if (userInput.at(i)){
			userStep = true;
			break;
		}
	}
	bool ready;

	while (true) {
		if (ConfigurationSingleton::getInstance()->isIoflExternalCaller() && userStep){
			//concretize SCCs which user has chosen from GUI
			ready = concretize(SELECT_USER_INPUT);
			//Increase step limit, because we only have done user input step
			if (!ready){
				mStepLimit++;
			}
		} else if (!ConfigurationSingleton::getInstance()->isIoflExternalCaller() || ConfigurationSingleton::getInstance()->autoConcretize){
			//concretize more, if on command line or user wants to
			ready = concretize(mCexParams->mSelectionMode);
		}
		//ready = concretize(mCexParams->mSelectionMode);
		if (ready)
			return false;
		else {
			//bool error = findCriticalSubsystem(doGlobalSearch, true);
			bool error = findCriticalSubsystem();

			if (error) return true;

			mSubSys->removeUnusedEdges();
		}

		if (Statistics::getInstance()->getNumberOfSteps() >= mStepLimit) {
			LOG4CPLUS_DEBUG(cexlogger, "Aborting counter example search -- reached maximum number of steps (" << mStepLimit << ")");
			return false;
		}
	}

	//Should be unreachble...
	LOG4CPLUS_ERROR(cexlogger, "Reached supposedly unreachable code in searchAbstractCex()");
	return true;
}

/*
 * Concretizes an unspecified number of abstract nodes using heuristics
 * Returns true iff concretization is complete after this function call
 */
bool AbstractCexGen::concretize(const StateSelectionMode selectionMode) {
	LOG4CPLUS_DEBUG(cexlogger, "Next concretization step");

	concretizedInCurrentStep.clear();

	AbstractStateSelector selector(mAbstGraph);
	graph_vector states = selector.chooseAbstractStates(mCexParams->mConcMode, selectionMode, mCexParams->mFilterMode);

	stringstream s1;
	s1 << "Resulting Graphs: ";
	for (graph_it it = states.begin(); it != states.end(); it++){
		s1 << (*it)->graph_id << ", ";
	}
	LOG4CPLUS_DEBUG(cexlogger, s1.str());


	if (states.empty()) {
		LOG4CPLUS_DEBUG(cexlogger, "Nothing left to concretize - falling back to findCounterExample()");
		return true;
	}
	else {
		LOG4CPLUS_DEBUG(cexlogger, "Will now concretize " << states.size() << " states");

		for (graph_it it = states.begin(); it != states.end(); it++) {
			Graph<PTYPE>* state = *it;

			LOG4CPLUS_DEBUG(cexlogger, "Starting concretization of " << state->graph_id);

			//generate m_min by removing the abstract edges from foundState
			LOG4CPLUS_DEBUG(cexlogger, "generate m_min...");
			removeAbstractEdges(state);

			//generate m_max by setting the edges in the SCC to visible
			LOG4CPLUS_DEBUG(cexlogger, "generate m_max \\ m_min...");
			markEdgesInScc(state);
			state->concretized = true;

			LOG4CPLUS_DEBUG(cexlogger, "Finished concretization of " << state->graph_id);

			concretizedInCurrentStep.insert(state);
		}

		return false;
	}
}

}

