/**
 *   COMICS - Computing Minimal Counterexamples for Discrete-time Markov Chains
 *
 *   COMICS is a stand-alone tool which performs model checking and the generation
 *   of counterexamples for discrete-time Markov Chains (DTMCs). *
 *
 *   Copyright (C) <2012> <RWTH Aachen University>
 *   Authors: Nils Jansen, Erika Abraham, Jens Katelaan, Maik Scheffler, Matthias Volk, Andreas Vorpahl
 *
 *   This program is free software: you can redistribute it and/or modify
 *   it under the terms of the GNU General Public License as published by
 *   the Free Software Foundation, either version 3 of the License, or
 *   (at your option) any later version.
 *
 *   This program is distributed in the hope that it will be useful,
 *   but WITHOUT ANY WARRANTY; without even the implied warranty of
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *   GNU General Public License for more details.
 *
 *   You should have received a copy of the GNU General Public License
 *   along with this program.  If not, see <http://www.gnu.org/licenses/>.
 *
 *   Main Contact:
 *
 *   Nils Jansen
 *   Theory of Hybrid Systems
 *   RWTH Aachen
 *   52056 Aachen
 *   Germany
 *   nils.jansen@cs.rwth-aachen.de
 *
 */

/*
 * CexGenerator.cpp
 *
 *  Created on: 08.03.2011
 *      Author: jens
 */

#include <set>
#include <map>
#include <queue>
#include <fstream>

#include "CexGenerator.h"
#include "../Closure.h"
#include "../../io/Configuration.h"
#include "../../io/OutputFormatters.h"
#include "../AbstractStateSelector.h"
#include "../pathfinder/LocalPathFinder.h"
#include "../pathfinder/GlobalPathFinder.h"

using namespace std;
using namespace log4cplus;

namespace scc_cex {

Logger cexlogger = Logger::getInstance("CexGen");

CexGenerator::CexGenerator(Graph<PTYPE>* abstGraph) {
	mAbstGraph = abstGraph;

	mStepLimit = UINT_MAX;
	mStartedSearch = false;
	mStats = Statistics::getInstance();

	cexlogger.setLogLevel(DEFAULT_LOG_LEVEL);
}

CexGenerator::~CexGenerator() {
	mAbstGraph = NULL;
	mCexTargetNode = NULL;
	mCexInputNode = NULL;
	delete mSubSys;
}

void CexGenerator::findCounterExample(PTYPE probBound, OperatorType op, unsigned int targetIndex, CounterExampleParams* cexParams) {
	LOG4CPLUS_INFO(cexlogger, "Will now generate counter example...");

	ConfigurationSingleton *conf = ConfigurationSingleton::getInstance();

	if (!mStartedSearch) {
		mBreakProb = probBound;
		mOperator = op;
		mCexParams = cexParams;

		//Initializations:
		if (!mCexParams->mUseAbstractSearch) {
			LOG4CPLUS_INFO(cexlogger, "Will search on concrete system");
			reinitializeEdgesForConcreteSearch(mAbstGraph->allVertices);
		} else {
			LOG4CPLUS_INFO(cexlogger, "Will search on abstract system");
			reinitializeEdges(mAbstGraph->allVertices);
		}

		if (mCexParams->mUseGlobalSearch) {
			LOG4CPLUS_INFO(cexlogger, "Will use *global* search to generate counter example");
			mPathFinder = new GlobalPathFinder(this);
		} else {
			LOG4CPLUS_INFO(cexlogger, "Will use *local* search to generate counter example");
			mPathFinder = new LocalPathFinder(this);
		}

		mStats->setSearchParams(mCexParams->mUseGlobalSearch, mCexParams->mUseAbstractSearch, !conf->isIoflBoundPathsum());
		mStats->setCounterExampleTask(op, probBound);
		mStats->setConcretizationParams(cexParams->mConcMode, cexParams->mSelectionMode, cexParams->mFilterMode);
		mStats->setMcResult(mcResult);

		//Find the input and target node, mark edge from input to target visible
		mCexInputNode = mAbstGraph->inputVertices[0];
		//Path* initialPath = new Path(1, 0, NULL, mCexInputNode, NULL);
		//mCexInputNode->setkthShortestPath(1, initialPath);
		LOG4CPLUS_TRACE(cexlogger, "Inputnode is: " << mCexInputNode->oindex);
		for (edge_it it = mCexInputNode->outEdges.begin(); it != mCexInputNode->outEdges.end(); it++) {
			LOG4CPLUS_TRACE(cexlogger, "processing edge from " << (*it)->source->oindex << " to " << (*it)->target->oindex);
			LOG4CPLUS_TRACE(cexlogger, "targetIndex: " << targetIndex);
			if ((*it)->target->oindex == targetIndex) {
				mCexTargetNode = (*it)->target;
				LOG4CPLUS_INFO(cexlogger, "Will search counter example to " << (*mCexTargetNode));

				if (mCexParams->mUseAbstractSearch)
					(*it)->setMark(true);

				break;
			}
		}

		if(ConfigurationSingleton::getInstance()->isIoflNoModelChecking()) {
			mCexTargetNode = mAbstGraph->outputVertices[0];
		}

		//Initialize the critical sub system (-to-be)
		mSubSys = new CriticalSubSystem(mCexInputNode, mCexTargetNode);
		mSubSys->setPointerBasedEdgeTracking(true);

		LOG4CPLUS_TRACE(cexlogger, "Initializations of counter example graph complete");
	}
	mStartedSearch = true;
	mCexParams = cexParams;
	mStats->setConcretizationParams(cexParams->mConcMode, cexParams->mSelectionMode, cexParams->mFilterMode);
	LOG4CPLUS_DEBUG(cexlogger, "Concretization Mode: " << concModeToString(mCexParams->mConcMode));
	LOG4CPLUS_DEBUG(cexlogger, "Selection Mode: " << selectionModeToString(mCexParams->mSelectionMode));
	LOG4CPLUS_DEBUG(cexlogger, "Filter Mode: " << filterModeToString(mCexParams->mFilterMode));

	LOG4CPLUS_DEBUG(cexlogger, "C works in mysterious ways...");
	mStats->startTime();
	bool error = searchCex();
	mStats->stopTime();

	if (!error && !conf->isIoflBoundPathsum()) {
		LOG4CPLUS_INFO(cexlogger, "The critical subsystem forms a counter example!");

		//Get & save closure size for stats
		computeClosureSize();
	} else if (!error) {
		LOG4CPLUS_INFO(cexlogger, "The paths form a counter example");

		//Compute number of nodes and transitions. Okay, not efficient, that's for sure, but we don't care about that (in postprocessing)
		set<Vertex<PTYPE>*> involvedVertices;
		set<Edge<PTYPE>*> involvedEdges;

		for (vector<Path*>::iterator it = mCexTargetNode->mShortestPaths.begin(); it != mCexTargetNode->mShortestPaths.end(); it++) {
			Path* curr = *it;
			while (curr != NULL && !curr->terminateHere) {
				if (curr->node != NULL)
					involvedVertices.insert(curr->node);
				if (curr->edge != NULL)
					involvedEdges.insert(curr->edge);
				curr = curr->pred;
			}
		}

		LOG4CPLUS_INFO( cexlogger, "The paths induce a set of " << involvedVertices.size() << " states, a set of " << involvedEdges.size() << " transitions");
	} else {
		LOG4CPLUS_ERROR(cexlogger, "Couldn't find counter example: A fatal error occurred");
	}
}

string CexGenerator::getSummary() const {
	return mStats->generateSummary();
}

PTYPE CexGenerator::getProbCEX() {
	return mStats->getProb();
}

/* ------------ private functions ---------------- */

Graph<PTYPE>* CexGenerator::getAbstractGraph() const {
	return mAbstGraph;
}

//Generate XML string for current counter example
string CexGenerator::counterExampleToXMLString() const {
	if (ConfigurationSingleton::getInstance()->isIoflBoundPathsum()) {
		return pathString.str();
	}

	bool concreteSearch = !mCexParams->mUseAbstractSearch;

	std::vector<bool> mTargetNodes;
	for (int i = 1; i <= mCexTargetNode->oindex; i++) {
		mTargetNodes.push_back(false);
	}
	mTargetNodes.push_back(true);
	XMLOutputFormatter<PTYPE> *formatter = new XMLOutputFormatter<PTYPE>(mCexInputNode, mTargetNodes, false, concreteSearch);

	//Check which graphs should be contained in xml
	queue<Graph<PTYPE>*> graphQueue;
	graphQueue.push(mAbstGraph);
	while (!graphQueue.empty()) {
		Graph<PTYPE>* g = graphQueue.front();
		graphQueue.pop();
		if (concreteSearch) {
			g->inXML = true;
		} else if (g->concretized) {
			g->inXML = true;
		} else if (g->parent != 0 && !g->parent->concretized) {
			g->inXML = g->parent->inXML;
		} else {
			//Check reachability from concretized parent
			bool reachable = false;
			for (unsigned int j = 0; (j < g->inputVertices.size() && !reachable); j++) {
				for (edge_it itEdge = g->inputVertices[j]->inEdges.begin(); (itEdge != g->inputVertices[j]->inEdges.end() && !reachable); itEdge++) {
					if ((*itEdge)->getMark()) {
						reachable = true;
					}
				}
			}
			g->inXML = reachable;
		}

		//Insert subgraphs
		for (unsigned int i = 0; i < g->subgraphs.size(); i++) {
			graphQueue.push(g->subgraphs[i]);
		}
	}

	//Check which states should be contained in xml
	for (unsigned int i = 0; i < mAbstGraph->allVertices.size(); i++) {
		Vertex<PTYPE>* v = mAbstGraph->allVertices[i];
		if (concreteSearch) {
			v->inXML = v->isInClosure;
		} else {
			v->inXML = v->isInClosure || (!v->g->concretized && v->g->inXML);
		}
	}

	for (unsigned int i = 0; i < mAbstGraph->outputVertices.size(); i++) {
		Vertex<PTYPE>* v = mAbstGraph->outputVertices[i];
		if (concreteSearch) {
			v->inXML = v->isInClosure;
		} else if (v->isInClosure) {
			v->inXML = true;
		} else {
			//Search for edge in XML
			bool reachable = false;
			for (edge_it itEdge = v->inEdges.begin(); (itEdge != v->inEdges.end() && !reachable); itEdge++) {
				if ((*itEdge)->getMark()) {
					reachable = true;
				} else {
					Vertex<PTYPE>* v = (*itEdge)->source;
					if ((*itEdge)->isAbstract() && v->inXML && v->g != 0 && !v->g->concretized) {
						reachable = true;
					}
				}
			}
			v->inXML = reachable;
		}
	}

	string s = formatter->generateOutput(*mAbstGraph);
	LOG4CPLUS_DEBUG(cexlogger, s);
	return s;

}

string CexGenerator::counterExampleToString() const {
	if (ConfigurationSingleton::getInstance()->isIoflBoundPathsum()) {
		return pathString.str();
	}
	counterExampleToXMLString();

	queue<Vertex<PTYPE>*> nodeQueue;
	nodeQueue.push(mCexInputNode);
	map<int, int> idRenaming;

	map<pair<int, int>, PTYPE> edges;

	unsigned int nodeCount = 0;
	unsigned int edgeCount = 0;
	unsigned int maxID = 0;

	for (vtx_it it = mAbstGraph->allVertices.begin(); it != mAbstGraph->allVertices.end(); it++) {
		(*it)->seen = false;
		if ((*it)->oindex > maxID) {
			maxID = (*it)->oindex;
		}
	}
	for (vtx_it it = mAbstGraph->outputVertices.begin(); it != mAbstGraph->outputVertices.end(); it++) {
		(*it)->seen = false;
		if ((*it)->oindex > maxID) {
			maxID = (*it)->oindex;
		}
	}

	//At this point, m_min = m_max, so every visible edge is in the cex
	while (!nodeQueue.empty()) {
		Vertex<PTYPE>* node = nodeQueue.front();
		nodeQueue.pop();
		nodeCount++;

		//Don't rename!
		idRenaming[node->oindex] = node->oindex;
		//idRenaming[node->oindex] = nodeCount;
		if (node->oindex > maxID) {
			maxID = node->oindex;
		}

		for (edge_it it = node->outEdges.begin(); it != node->outEdges.endMarked(); it++) {
			if ((*it)->not_m_min) {
				continue;
			}

			edges[make_pair(node->oindex, (*it)->target->oindex)] = (*it)->prob;

			if (!(*it)->target->seen) {
				nodeQueue.push((*it)->target);
				(*it)->target->seen = true;
			}

			edgeCount++;
		}

	}

	stringstream ss;
	ss << "STATES " << maxID << endl;
	ss << "TRANSITIONS " << edgeCount << endl;
	ss << "INITIAL " << idRenaming[mCexInputNode->oindex] << endl;
	ss << "TARGET " << idRenaming[mCexTargetNode->oindex] << endl;

	for (map<pair<int, int>, PTYPE>::iterator it = edges.begin(); it != edges.end(); it++) {
		ss << idRenaming[(*it).first.first] << " " << idRenaming[(*it).first.second] << " " << (*it).second << endl;
	}

	return ss.str();
}

/*
 * In Concretization: Remove all abstract edges from the input nodes of the given scc
 * Call this when scc is to be concretized -> after this m_min and nothing else is visible
 * "Remove" means set invisible - this is more efficient than actually remove the edge from its adjacent vertices
 * But it means that the "sister" edge in the closure is deleted
 */
void CexGenerator::removeAbstractEdges(Graph<PTYPE>* scc) {
	LOG4CPLUS_DEBUG(cexlogger, "Now getting rid of the abstract edges in concretized SCC");

	unsigned int removedCount = 0;

	for (vtx_it it = scc->inputVertices.begin(); it != scc->inputVertices.end(); it++) {
		for (edge_it it2 = (*it)->outEdges.begin(); it2 != (*it)->outEdges.end(); it2++) {
			Edge<PTYPE>* e = *it2;
			if (e->isAbstract() && e->getMark()) {
				e->setMark(false);

				//Delete sister
				if (e->sisterEdge != NULL) {
					LOG4CPLUS_TRACE(cexlogger, "Will remove " << (* e->sisterEdge) << " because of concretization");

					Vertex<PTYPE>* src = e->sisterEdge->source;
					Vertex<PTYPE>* trg = e->sisterEdge->target;
					src->outEdges.remove(e->sisterEdge);
					trg->inEdges.remove(e->sisterEdge);
					delete e->sisterEdge;
					e->sisterEdge = NULL;
				}

				removedCount++;
			}
		}
	}

	LOG4CPLUS_DEBUG(cexlogger, "A total of " << removedCount << " edges were removed");
}

/*
 * In Concretization: Search through the given SCC
 * and mark the contained input nodes concretized, the edges visible
 */
void CexGenerator::markEdgesInScc(Graph<PTYPE>* scc) {

	queue<Vertex<PTYPE>*> waiting;

	for (vtx_it it = scc->allVertices.begin(); it != scc->allVertices.end(); it++) {
		(*it)->tmpMarking = false;
	}

	LOG4CPLUS_DEBUG(cexlogger, "Starting the marking");

	for (vtx_it it = scc->inputVertices.begin(); it != scc->inputVertices.end(); it++) {
		(*it)->isConcretized = true;
		(*it)->tmpMarking = true;
		LOG4CPLUS_TRACE(cexlogger, "has input vertices " << (*it)->oindex << "\n");
		waiting.push(*it);
	}

	while (!waiting.empty()) {
		LOG4CPLUS_TRACE(cexlogger, "" << waiting.size() << " nodes left in the queue");

		Vertex<PTYPE>* next = waiting.front();
		waiting.pop();

		assert(next->tmpMarking);

		LOG4CPLUS_TRACE(cexlogger, "Now processing " << (*next));

		bool followAbstract = false;

		if (next->g != NULL && next->g == scc) {
			//in scc -> follow & mark concrete edges
			followAbstract = false;
		} else if (next->g != NULL && next->g->isChildOf(scc)) {
			//sub scc -> follow & mark abstract edges
			followAbstract = true;
		} else {
			//outside scc -> don't follow any edges
			LOG4CPLUS_TRACE(cexlogger, "Outside SCC, don't follow edges");
			continue;
		}

		if (followAbstract)
			LOG4CPLUS_TRACE(cexlogger, "Follow the abstract edges");
		else
			LOG4CPLUS_TRACE(cexlogger, "Follow the concrete edges");

		for (edge_it it = next->outEdges.begin(); it != next->outEdges.end(); it++) {
			if ((*it)->isAbstract() == followAbstract) {
				LOG4CPLUS_TRACE(cexlogger, "Adding: " << (**it));

				mSubSys->addEdgeToTrack(*it);

				//See if we have to look at the edge's target as well
				Vertex<PTYPE>* candidate = (*it)->target;
				if (!candidate->tmpMarking) {
					candidate->tmpMarking = true;
					waiting.push(candidate);
					LOG4CPLUS_TRACE(cexlogger, "Added " << (*candidate) << " to queue");
				} else {
					LOG4CPLUS_TRACE(cexlogger, (*candidate) << " already dealt with");
				}
			}
		}

	}

	for (vtx_it it = scc->allVertices.begin(); it != scc->allVertices.end(); it++) {
		(*it)->seen = false;
	}

	LOG4CPLUS_DEBUG(cexlogger, "m_max \\ m_min now ready");
}

/*
 * This is the common wrapper for all the search algorithms,
 * i.e. abstract global, abstract local, concrete closure-based, concrete path-based
 * TODO: One could experiment with not recomputing sums/closures for every new path (for performance gains)
 */
bool CexGenerator::findCriticalSubsystem() {
	//bool CexGenerator::findCriticalSubsystem(const bool doGlobalSearch, const bool doClosureComputations) {
	PTYPE currentProbMass = 0;
	unsigned int k = 0;
	unsigned int closureCount = 0;

	bool doClosureComputations = !ConfigurationSingleton::getInstance()->isIoflBoundPathsum();

	//LOG4CPLUS_TRACE(cexlogger, "Global search: " << doGlobalSearch << ", Use closures: " << doClosureComputations);

	bool error = false;

	do {
		k++;

		LOG4CPLUS_TRACE(cexlogger, "Looping in kth shortest search, k being " << k);

		Path* pi = mPathFinder->nextShortestPath(k);
		LOG4CPLUS_TRACE(cexlogger, "---------New path: " << pi->toString());

		if (pi == NULL) {
			LOG4CPLUS_ERROR(cexlogger, "Couldn't find any path for k = " << k << ", will break search");
			error = true;
			break;
		}

		if (doClosureComputations) {
			LOG4CPLUS_TRACE(cexlogger, "The " << k << "th shortest path to target is: " << pi->toStringDebug());
		} else {
			//In this case the paths *are* the counter examples, so we should print 'em
			//TODO: maybe we should have a separate counter example file
			pathString << pi->toStringOutput() << "\n";
		}

		//If we are in closure-based search, we have to...
		if (doClosureComputations) {
			//1.) determine, if we have to recompute the closure
			bool wasExtended = mSubSys->addPathToSubSystem(pi);

			//2.) might have to do some cleanup:
			if (mPathFinder->pathDeletionNecessary()) {
				//The local search returns a copy of a shortest path, not a shortest path!
				//So we have to avoid memory leaks
				pi->deepDelete();
				delete pi;
			}

			//3.) might have to recompute the closure
			if (wasExtended) {
				LOG4CPLUS_DEBUG(cexlogger, "m_min was extended (k = " << k << ") -> generate closure");
				currentProbMass = mSubSys->generateClosure();
				closureCount++;
			} else {
				LOG4CPLUS_TRACE(cexlogger, "m_min was not extended -> generate next path");
			}

			LOG4CPLUS_TRACE( cexlogger, "Current prob mass of the closure is " << currentProbMass << " meaning we still need " << (mBreakProb - currentProbMass));
		}
		//If we don't compute closures, we just add pathsums
		else {
			currentProbMass += pi->L;

			LOG4CPLUS_TRACE( cexlogger, "Current prob mass (for k=" << k << ")of the paths is " << currentProbMass << " meaning we still need " << (mBreakProb - currentProbMass));
		}

	} while ((mOperator == OPERATOR_LEQ && currentProbMass <= mBreakProb) || (mOperator == OPERATOR_LESS && currentProbMass < mBreakProb));

	if (!error) {
		LOG4CPLUS_DEBUG(cexlogger, "Probability mass for this step suffices");

		//For benchmarks etc: How many paths were necessary to reach the critical mass?
		mStats->addStep(k, closureCount, concretizedInCurrentStep.size());
		mStats->setProb(currentProbMass);

		//Print the abstract cex (Might remove this later on)
		//printCounterExample();
		pathString << "Total prob: " << currentProbMass;
		LOG4CPLUS_DEBUG( cexlogger, "Probability mass of " << (doClosureComputations ? "closure: " : "all paths: ") << currentProbMass);

		if (mCexParams->mUseAbstractSearch)
			LOG4CPLUS_DEBUG(cexlogger, "Will start next concretization step, if any left");
	}

	return error;
}

void CexGenerator::computeClosureSize() {
	queue<Vertex<PTYPE>*> nodeQueue;
	nodeQueue.push(mCexInputNode);

	unsigned int nodeCount = 0;
	unsigned int edgeCount = 0;

	for (vtx_it it = mAbstGraph->allVertices.begin(); it != mAbstGraph->allVertices.end(); it++) {
		(*it)->seen = false;
	}
	for (vtx_it it = mAbstGraph->outputVertices.begin(); it != mAbstGraph->outputVertices.end(); it++) {
		(*it)->seen = false;
	}

	//At this point, m_min = m_max, so every visible edge is in the cex
	while (!nodeQueue.empty()) {
		Vertex<PTYPE>* node = nodeQueue.front();
		nodeQueue.pop();

		bool first = true;
		for (edge_it it = node->outEdges.begin(); it != node->outEdges.endMarked(); it++) {
			if ((*it)->not_m_min)
				continue;

			if (first)
				first = false;

			if (!(*it)->target->seen) {
				nodeQueue.push((*it)->target);
				(*it)->target->seen = true;
			}

			edgeCount++;
		}
		nodeCount++;
	}

	mStats->setCounterExampleSize(nodeCount, edgeCount);
}

}
