/**
 *   COMICS - Computing Minimal Counterexamples for Discrete-time Markov Chains
 *
 *   COMICS is a stand-alone tool which performs model checking and the generation
 *   of counterexamples for discrete-time Markov Chains (DTMCs). *
 *
 *   Copyright (C) <2012> <RWTH Aachen University>
 *   Authors: Nils Jansen, Erika Abraham, Jens Katelaan, Maik Scheffler, Matthias Volk, Andreas Vorpahl
 *
 *   This program is free software: you can redistribute it and/or modify
 *   it under the terms of the GNU General Public License as published by
 *   the Free Software Foundation, either version 3 of the License, or
 *   (at your option) any later version.
 *
 *   This program is distributed in the hope that it will be useful,
 *   but WITHOUT ANY WARRANTY; without even the implied warranty of
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *   GNU General Public License for more details.
 *
 *   You should have received a copy of the GNU General Public License
 *   along with this program.  If not, see <http://www.gnu.org/licenses/>.
 *
 *   Main Contact:
 *
 *   Nils Jansen
 *   Theory of Hybrid Systems
 *   RWTH Aachen
 *   52056 Aachen
 *   Germany
 *   nils.jansen@cs.rwth-aachen.de
 *
*/

/*
 * CexGenerator.h
 *
 *  Created on: 08.03.2011
 *      Author: jens
 */

#ifndef CEXGENERATOR_H_
#define CEXGENERATOR_H_

#include <set>
#ifdef USE_LOGLIB
#include <log4cplus/logger.h>
#endif

#include "../../GraphTypes.h"
#include "../Statistics.h"
#include "../AbstractStateSelector.h"
#include "../CounterExampleParams.h"
#include "../CriticalSubSystem.h"
#include <sstream>

namespace scc_cex {

extern log4cplus::Logger cexlogger;

class PathFinder;
class GlobalPathFinder;
class LocalPathFinder;

/**
 * The (abstract) base class of the counter example generators.
 */
class CexGenerator {
public:
	CexGenerator(Graph<PTYPE>* abstGraph);
	virtual ~CexGenerator();

	void findCounterExample(PTYPE mProbBound, OperatorType mOperator, unsigned int targetIndex, CounterExampleParams* cexParams);
	std::string getSummary() const;
	Graph<PTYPE>* getAbstractGraph() const;
	std::string counterExampleToString() const;
	std::string counterExampleToXMLString() const;
	PTYPE getProbCEX();

	void setStepLimit(const unsigned int numberOfSteps) {
		mStepLimit = numberOfSteps;
	}

	void setModelCheckingProbability(PTYPE prob) {
		mcResult = prob;
	}

	unsigned int getStepLimit() const {
		return mStepLimit;
	}

	unsigned int getStepCount() const{
		return Statistics::getInstance()->getNumberOfSteps();
	}

	bool isCexComplete() {
		if (!mStartedSearch) return false;

		AbstractStateSelector selector(mAbstGraph);
		return !selector.areThereAbstractStates();
	}

	friend class PathFinder;
	friend class GlobalPathFinder;
	friend class LocalPathFinder;

protected:
	/* counter example functions */
	virtual bool searchCex() = 0;

	void removeAbstractEdges(Graph<PTYPE>* scc);
	void markEdgesInScc(Graph<PTYPE>* scc);

	bool findCriticalSubsystem();

	void computeClosureSize();

	/* attributes */
	Graph<PTYPE>* mAbstGraph;
    Vertex<PTYPE>* mCexInputNode;
    Vertex<PTYPE>* mCexTargetNode;

    PathFinder* mPathFinder;
    CriticalSubSystem* mSubSys;
    //Model Checking Result
    PTYPE mcResult;

    /* cex properties */
	PTYPE mBreakProb;
	OperatorType mOperator;
	CounterExampleParams* mCexParams;
    unsigned int mStepLimit;
    bool mStartedSearch;
    std::stringstream pathString;

	/* cex attributes */
	Statistics *mStats;

	graph_set concretizedInCurrentStep;
};

/*
 * To compare the distance of two shortest paths
 */
class CompareDistance
{
public:
	bool operator() (Vertex<PTYPE>* n1,Vertex<PTYPE>* n2) const {
		PTYPE dist1 = n1->getkthShortestPath(1)->L;
		PTYPE dist2 = n2->getkthShortestPath(1)->L;
		return dist1 < dist2; //We are actually looking for the longest path
	}
};

}

#endif /* CEXGENERATOR_H_ */
