/**
 *   COMICS - Computing Minimal Counterexamples for Discrete-time Markov Chains
 *
 *   COMICS is a stand-alone tool which performs model checking and the generation
 *   of counterexamples for discrete-time Markov Chains (DTMCs). *
 *
 *   Copyright (C) <2012> <RWTH Aachen University>
 *   Authors: Nils Jansen, Erika Abraham, Jens Katelaan, Maik Scheffler, Matthias Volk, Andreas Vorpahl
 *
 *   This program is free software: you can redistribute it and/or modify
 *   it under the terms of the GNU General Public License as published by
 *   the Free Software Foundation, either version 3 of the License, or
 *   (at your option) any later version.
 *
 *   This program is distributed in the hope that it will be useful,
 *   but WITHOUT ANY WARRANTY; without even the implied warranty of
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *   GNU General Public License for more details.
 *
 *   You should have received a copy of the GNU General Public License
 *   along with this program.  If not, see <http://www.gnu.org/licenses/>.
 *
 *   Main Contact:
 *
 *   Nils Jansen
 *   Theory of Hybrid Systems
 *   RWTH Aachen
 *   52056 Aachen
 *   Germany
 *   nils.jansen@cs.rwth-aachen.de
 *
*/

/*
 * LocalPathFinder.cpp
 *
 *  Created on: 18.10.2011
 *      Author: jens
 */

#ifdef USE_LOGLIB
#include <log4cplus/logger.h>
#endif
#include "LocalPathFinder.h"

using namespace std;
using namespace log4cplus;

namespace scc_cex {

#define USE_AUX_NODE 1

#include <stdio.h>
#include <time.h>


LocalPathFinder::LocalPathFinder(CexGenerator* cexGen) : PathFinder(cexGen) {
	this->execTime = 0;
}

LocalPathFinder::~LocalPathFinder() {
	// Nothing to do here right now
}

/**
 * The local path search creates path objects itself that are not referred to
 * or taken care of anywhere else in the code. The caller must therefore delete
 * the objects after he has used the path.
 */
bool LocalPathFinder::pathDeletionNecessary() {
	return true;
}

/*
 * The local path search restricts the shortest path search to the scc we currently concretize
 * While this greatly reduces the cost of the path search, it probably makes some copying
 * around necessary - can't do this in-place in a straight-forward way without altering
 * the structure of the graph
 */
Path* LocalPathFinder::nextShortestPath(const unsigned int k) {
	//clock_t temptime = clock();
	assert(k>=1);


	//What we do keep is an ever-growing set of nodes with weight 1 (starting nodes)
	//and a likewise growing set of nodes we are trying to reach
	//Therefore only reset those sets for k=1
	if (k == 1) {
		nodesToStartFrom.clear();
		nodesToSearchTo.clear();

		//k=1 means that we only know the input and output nodes of the concretized SCCs so far
		for (set<Graph<PTYPE>*>::iterator it = mCexGen->concretizedInCurrentStep.begin();it != mCexGen->concretizedInCurrentStep.end(); it++) {
			for (vtx_it inp_it = (*it)->inputVertices.begin();inp_it != (*it)->inputVertices.end(); inp_it++) {
				nodesToStartFrom.insert(*inp_it);
			}

			//Only add the reachable output vertices
			for (vtx_it out_it = (*it)->outputVertices.begin();out_it != (*it)->outputVertices.end(); out_it++) {
				if ((*out_it)->isInClosure || (*out_it) == mCexGen->mCexTargetNode) {
					nodesToSearchTo.insert(*out_it);
				}

			}
		}


		LOG4CPLUS_DEBUG(pathlogger,"k = 1, therefore initialized s_start and s_goal with data from the " << mCexGen->concretizedInCurrentStep.size() << " concretized node(s)");
	}

	assert(!nodesToStartFrom.empty());
	assert(!nodesToSearchTo.empty());



	for (vtx_set::iterator from_it = nodesToStartFrom.begin(); from_it != nodesToStartFrom.end(); from_it++) {
		Vertex<PTYPE>* temp = *from_it;
		LOG4CPLUS_TRACE(pathlogger,"from Set Contains (before loop): " << temp->oindex);
	}
	for (vtx_set::iterator to_it = nodesToSearchTo.begin(); to_it != nodesToSearchTo.end(); to_it++) {
		Vertex<PTYPE>* temp = *to_it;
		LOG4CPLUS_TRACE(pathlogger,"To Set Contains (before loop): " << temp->oindex);
	}
#ifdef USE_AUX_NODE
	/*
	 * insert auxiliary node with an edge to each node in nodestoStartFrom with prob 1
	 */
	Vertex<PTYPE> *mAuxiliaryNode = new Vertex<PTYPE>();
	mAuxiliaryNode->oindex = -1;
	mAuxiliaryNode->cleanShortestPaths();
	mAuxiliaryNode->setkthShortestPath(1, new Path(1, 0, NULL, mAuxiliaryNode, NULL));
	int counter = 0;
	for (vtx_set::iterator from_it = nodesToStartFrom.begin(); from_it != nodesToStartFrom.end(); from_it++) {
		Vertex<PTYPE>* temp = *from_it;
		new Edge<PTYPE>(mAuxiliaryNode, temp, 1.0);
		mAuxiliaryNode->outEdges[counter]->setMark(true);
		mAuxiliaryNode->outEdges[counter]->not_m_min = true;
		counter++;
	}

#endif

	//TODO: this is a naive n^3 implementation, there should be some possible improvements. E.g. doing only one Dijkstra from an auxiliary predecessor of all input nodes.
	//Now we'll do some searchin using Dijkstra restricted to the nodes in our scc...
	Path* result = NULL;

	LOG4CPLUS_DEBUG(pathlogger,"In this step we have " << nodesToStartFrom.size() << " sources and " << nodesToSearchTo.size() << " targets");

#ifndef USE_AUX_NODE
	for (vtx_set::iterator global_it = nodesToStartFrom.begin();global_it != nodesToStartFrom.end(); global_it++) {
		//...separately for each input node, of course;
		Vertex<PTYPE>* magicNode = *global_it;
#else
		Vertex<PTYPE>* magicNode = mAuxiliaryNode;
#endif
		LOG4CPLUS_DEBUG(pathlogger,"Local Dijkstra with source: " << magicNode->oindex);

		//mAbstGraph->cleanShortestPaths();
		for (graph_set::iterator it = mCexGen->concretizedInCurrentStep.begin();it != mCexGen->concretizedInCurrentStep.end(); it++) {
			(*it)->cleanShortestPaths();
		}


		LOG4CPLUS_TRACE(pathlogger,"Reset previously computed shortest paths for the concretized SCCs.");

		//Initialize the set of nodes that have to be visited
		//(Initially all nodes have to be visited)
		priority_queue<Vertex<PTYPE>*, vector<Vertex<PTYPE>*>, CompareDistance> remainingNodes;


		remainingNodes.push(magicNode);

		while (remainingNodes.size() > 0) {
			//Get 1st element, i.e. the element with the lowest distance
			Vertex<PTYPE>* u = remainingNodes.top();


			PTYPE uWeight = u->getkthShortestPath(1)->L;

#ifdef USE_AUX_NODE
			if(nodesToStartFrom.find(u) != nodesToStartFrom.end()) {
				uWeight = 1;
			}
#endif

			if (u == magicNode)
				uWeight = 1; //This dirty trick is done to rule out zero-edge-paths when input and output sets overlap
			if (uWeight == 0) {
				//All remaining nodes are unreachable --> shouldn't happen!
				LOG4CPLUS_ERROR(pathlogger,"found unreachable nodes in Dijkstra's algorithm");
				break;
			}



			//Remove u from remainingNodes
			remainingNodes.pop();
#ifdef USE_AUX_NODE
			if((k>1) && (u->oindex == mAuxiliaryNode->oindex)) {
				for (vtx_set::iterator from_it = nodesToStartFrom.begin(); from_it != nodesToStartFrom.end(); from_it++) {
					Vertex<PTYPE>* temp = *from_it;
					remainingNodes.push(temp);
				}
			}
#endif

			//PTYPE uWeight = u->getkthShortestPath(1)->L;
			LOG4CPLUS_TRACE(pathlogger,"Node's weight is " << uWeight << " (but really: " << u->getkthShortestPath(1)->L << ")");


			if (u->outEdges.size() < 1) {
				//LOG4CPLUS_TRACE(pathlogger,u->getName() << " does not have any outgoing edges");
				continue;
			}

			//Change distance of all adjacent nodes
			//Here the edges need not only be visible, their targets must also be in our SCC or be one of the targets
			for (edge_it it = u->outEdges.begin();it != u->outEdges.endMarked(); it++) {

				Edge<PTYPE>* edge = *it;
				Vertex<PTYPE>* trgNode = edge->target;

				//LOG4CPLUS_DEBUG(pathlogger,"Processing edge from " << u->oindex << " to node " << trgNode->oindex);


				//LOG4CPLUS_DEBUG(pathlogger,"edges properties: prob: " << edge->prob << " | state: " << edge->state << " | notmmin: " << edge->not_m_min << " | visited: " << edge->visited << " | mark: " << edge->getMark());

				if (!edge->not_m_min) {
					//In the local search, we are only interested in truly new path segments
					//We never need to use an edge twice, because all sources of already used edges are in s_goal
					LOG4CPLUS_TRACE(pathlogger,"Will skip edge " << (*edge) << ", it's not new");
					continue;
				}


				bool skip = true;
				bool addTarget = true;

				if (nodesToSearchTo.find(trgNode) != nodesToSearchTo.end()) {
					//Target is one of our targets -> don't skip the edge, but also don't add the target to the queue, our search should stop there
					skip = false;
					addTarget = false;
					//LOG4CPLUS_TRACE(pathlogger,"I will not add " << trgNode->oindex << " because its one of our targets");
				} else if (!mCexGen->mCexParams->mUseAbstractSearch) {
					//If we are searching on the concrete system, we are looking at every vertex
					skip = false;
				} else if (trgNode->g != NULL && mCexGen->concretizedInCurrentStep.find(trgNode->g) != mCexGen->concretizedInCurrentStep.end()) {
					//Target is in one of the SCCs we are concretizing -> don't skip edge
					skip = false;
				} else {
					//See if the target SCC is child of one of our concretized SCCs
					//Expensive when many nodes are concretized. TODO: algorithmic improvements possible?
					if (trgNode->g != NULL) {
						for (set<Graph<PTYPE>*>::iterator it = mCexGen->concretizedInCurrentStep.begin();it != mCexGen->concretizedInCurrentStep.end();it++) {
							if (trgNode->g->isChildOf(*it)) {
								skip = false;
								break;
							}
						}
					}
				}

				if (skip) {
					LOG4CPLUS_TRACE(pathlogger,"Will skip edge " << (*edge) << " for some reason");
					//LOG4CPLUS_DEBUG(pathlogger, "Skipping edge");
					continue;
				}

				LOG4CPLUS_TRACE(pathlogger, "Target: " << trgNode->getName());
				PTYPE newDist = edge->prob * uWeight;
				PTYPE currDist = trgNode->getkthShortestPath(1)->L;

				//If we've found a more probable path, save new prob & predecessor
#ifndef USE_AUX_NODE
				if (newDist > currDist) {
					LOG4CPLUS_TRACE(pathlogger,"Changing distance of " << trgNode->getName() << " from " << currDist << " to " << newDist);
					Path* shortestPath = trgNode->getkthShortestPath(1);

					if ((nodesToStartFrom.find(trgNode) == nodesToStartFrom.end()) || (nodesToSearchTo.find(trgNode) != nodesToSearchTo.end())) {
						shortestPath->L = newDist;
						shortestPath->pred = u->getkthShortestPath(1);
						shortestPath->edge = edge;
					}
					//Successor info for assembling the RegEx

//					if (nodesToStartFrom.find(trgNode) != nodesToStartFrom.end()) {
//						shortestPath->terminateHere = true;
//					}



					if (u == magicNode)
						shortestPath->terminateHere = true; //Avoid loops

					//Priority queue filled here because priority updates are not noted
					if (addTarget) {
						remainingNodes.push(trgNode);
						//LOG4CPLUS_TRACE(pathlogger,"Adding " << trgNode->oindex << " to the list of remaining nodes");
					}
				}
#endif
#ifdef USE_AUX_NODE
				if(newDist > currDist) {
					if(u->oindex != mAuxiliaryNode->oindex) {
						LOG4CPLUS_TRACE(pathlogger,"Changing distance of " << trgNode->getName() << " from " << currDist << " to " << newDist);
						Path* shortestPath = trgNode->getkthShortestPath(1);

						if ((nodesToStartFrom.find(trgNode) == nodesToStartFrom.end()) || (nodesToSearchTo.find(trgNode) != nodesToSearchTo.end())) {
							shortestPath->L = newDist;
							shortestPath->pred = u->getkthShortestPath(1);
							shortestPath->edge = edge;
						}
						//shortestPath->L = newDist;
						//shortestPath->pred = u->getkthShortestPath(1);
						//Successor info for assembling the RegEx

						//shortestPath->edge = edge;

						// it might be possible, that we enlarge a path that already has the terminateHere Flag set, then we have to remove it in order to set the flag at the new node again
						shortestPath->terminateHere = false;

						//since we dont compute the dijkstra for each nodesTostartfrom node explicitly we have to adjust the condition to find loops in the paths
						if((vertexAlreadyInPath(trgNode, shortestPath)) && (nodesToStartFrom.find(u) != nodesToStartFrom.end())) {
							shortestPath->terminateHere = true; //Avoid loops
						}

						//Priority queue filled here because priority updates are not noted
						if (addTarget) {
							remainingNodes.push(trgNode);
							//LOG4CPLUS_TRACE(pathlogger,"Adding " << trgNode->oindex << " to the list of remaining nodes");
						}
					} else {
						/* trgNode is one of our searchfrom nodes, hence we don't want to change the distance to 1 */
						if (addTarget) {
								remainingNodes.push(trgNode);
								//LOG4CPLUS_TRACE(pathlogger,"Adding " << trgNode->oindex << " to the list of remaining nodes");
							}
					}
				}
#endif
			} //edge iterator end
		} //dijkstra, while end

		//up here we computed the most propable path to each element of nodestosearch to from one! element of nodestostartfrom
		//Now we just look for the most probable path to any of the targets
		set<Vertex<PTYPE>*>::iterator it = nodesToSearchTo.begin();
		Path* localResult = (*it)->getkthShortestPath(1);
		//PTYPE jopp = localResult->L;
		//myPathToString(localResult);
		it++;
		for (; it != nodesToSearchTo.end(); it++) {
			Path* candidate = (*it)->getkthShortestPath(1);
			if (candidate->L > localResult->L) {
				localResult = candidate;
				LOG4CPLUS_TRACE(pathlogger,"New candidate for current source (" << magicNode->oindex << "): " << localResult->toStringDebug());
				//LOG4CPLUS_TRACE(pathlogger,"New candidate for current source (" << magicNode->oindex << "): " << localResult->toStringDebug());
			}
		}

		//localResult contains now the path with the highest prob to one element of nodestosearchto from one nodetostartfrom

		LOG4CPLUS_DEBUG(pathlogger,"Found shortest path fragment for current source: " << localResult->toStringDebug());
		//LOG4CPLUS_DEBUG(pathlogger,"Found shortest path fragment for current source: " << localResult->toStringDebug());

		if (result == NULL || localResult->L > result->L) {
			if (result != NULL)
				LOG4CPLUS_DEBUG(pathlogger,"Overwriting result (" << result->L << ") with new path fragment (" << localResult->L << ")");
			else {
				LOG4CPLUS_DEBUG(pathlogger,"New path fragment is first possible result (" << localResult->L << ")");
			}

			//This means we need to deep-copy the result so that it isn't reset!
			if (result != NULL) {
				result->deepDelete();
				delete result;
			}
			result = localResult->deepCopy();
		}
#ifndef USE_AUX_NODE
	} //end dijkstra for each input node
#endif

	//result contains the most probable path from nodestostartfrom to one node of nodetosearchto

	LOG4CPLUS_DEBUG(pathlogger,"Shortest path fragment for all sources: " << result->toStringDebug());
	//LOG4CPLUS_DEBUG(pathlogger,"Shortest path fragment for all sources: " << result->toStringDebug());

	if (result->L == 0) {
		//This happens when we have have already used every single edge in the SCCs,
		//i.e. when the prob bound cannot be reached (counter example for property that isn't refuted)
		LOG4CPLUS_ERROR(pathlogger,"There are no edges in m_max \\ m_min left, can't find any more paths");
		return NULL;
	}

	/* check whether any path contains node from the input set or output set at any other position than start resp. end */
//	Path *tempres = result;
//	int ku = 0;
//	while(tempres != NULL) {
//		ku = ku + 1;
//		if( ((nodesToSearchTo.find(tempres->node) != nodesToSearchTo.end())) && (ku != 1) ) {
//		}
//		if( ((nodesToStartFrom.find(tempres->node) != nodesToStartFrom.end())) && (!((tempres->pred==NULL) || (tempres->terminateHere))) ) {
//		}
//		if(!tempres->terminateHere)
//			tempres = tempres->pred;
//	}


	//Before we return the result path, we have to change the start from / search to sets for the next k
	//I hope I follow the paper algorithm in that I add
	//a) the target of each and every edge in the path to the "start from"-set
	//b) the source of each and every edge in the path to the "search to"-set
	Path* pi = result;
	while (pi != NULL) {
		if (pi->pred != NULL) {
			nodesToStartFrom.insert(pi->edge->target);
			//if (nodesToStartFrom.find(pi->edge->target) == nodesToStartFrom.end()) {
				nodesToSearchTo.insert(pi->edge->source);
			//}
			//nodesToStartFrom.insert(pi->node);
			//nodesToSearchTo.insert(pi->node);
			//LOG4CPLUS_TRACE(pathlogger,"Added " << (* pi->edge->source) << " to s_goal, " << (* pi->edge->target) << " to s_start");
		}
		pi = pi->pred;
	}

	//LOG4CPLUS_DEBUG(pathlogger,"Returned path: " << result->toString());
	//clock_t temptime2 = clock();
	//this->execTime += (temptime2 - temptime);
	//printf("Clocks elapsed: %f\n", (double)(this->execTime));
	return result;
}

bool LocalPathFinder::vertexAlreadyInPath(Vertex<PTYPE>* vtx, Path* pa) {
	Path* tmp = pa;
	bool found = false;
	while((tmp != NULL) && (!found)) {
		if(tmp->node->oindex == vtx->oindex) {
			found = true;
		}
		tmp = tmp->pred;
	}
	return found;
}

}
