/**
 *   COMICS - Computing Minimal Counterexamples for Discrete-time Markov Chains
 *
 *   COMICS is a stand-alone tool which performs model checking and the generation
 *   of counterexamples for discrete-time Markov Chains (DTMCs). *
 *
 *   Copyright (C) <2012> <RWTH Aachen University>
 *   Authors: Nils Jansen, Erika Abraham, Jens Katelaan, Maik Scheffler, Matthias Volk, Andreas Vorpahl
 *
 *   This program is free software: you can redistribute it and/or modify
 *   it under the terms of the GNU General Public License as published by
 *   the Free Software Foundation, either version 3 of the License, or
 *   (at your option) any later version.
 *
 *   This program is distributed in the hope that it will be useful,
 *   but WITHOUT ANY WARRANTY; without even the implied warranty of
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *   GNU General Public License for more details.
 *
 *   You should have received a copy of the GNU General Public License
 *   along with this program.  If not, see <http://www.gnu.org/licenses/>.
 *
 *   Main Contact:
 *
 *   Nils Jansen
 *   Theory of Hybrid Systems
 *   RWTH Aachen
 *   52056 Aachen
 *   Germany
 *   nils.jansen@cs.rwth-aachen.de
 *
 */

/*
 * PathFinder.cpp
 *
 *  Created on: 18.10.2011
 *      Author: jens
 */

#ifdef USE_LOGLIB
#include <log4cplus/logger.h>
#endif
#include "PathFinder.h"

using namespace std;
using namespace log4cplus;

namespace scc_cex {

Logger pathlogger = Logger::getInstance("CexGen.PathFinder");
//Logger pathlogger = Logger::getInstance("CexGen");

PathFinder::PathFinder(CexGenerator* cexGen) {
	mCexGen = cexGen;
	pathlogger.setLogLevel(DEFAULT_LOG_LEVEL);
}

PathFinder::~PathFinder() {
	mCexGen = NULL;
}

/*
 * Dijkstra's algorithm modified to use multiplication and search most expensive
 * path. We don't have to care about loops in this steps, because the paths we
 * find are the very shortest -> can't contain loops
 */
void PathFinder::modifiedDijkstra() {
	//First we have to reset all shortest path info, because we are starting a
	//new shortest path search
	//Cleaning resets the 1st shortest paths to probability 0 and deletes all
	//paths for k>=2
	mCexGen->mAbstGraph->cleanShortestPaths();
	LOG4CPLUS_TRACE(pathlogger, "Reset previously computed shortest paths.");


	//Set probability from input to input to 1

	Path* pathToInput = mCexGen->mCexInputNode->getkthShortestPath(1);
	pathToInput->L = 1;
	LOG4CPLUS_TRACE(pathlogger, "Set prob to reach input (" << mCexGen->mCexInputNode->oindex << ") to 1");



	//Initialize the set of nodes that have to be visited
	//(Initially all nodes have to be visited)
	priority_queue<Vertex<PTYPE>*, vector<Vertex<PTYPE>*>, CompareDistance> remainingNodes;

	remainingNodes.push(mCexGen->mCexInputNode);


	while (remainingNodes.size() > 0) {
		LOG4CPLUS_TRACE(pathlogger, "--- remaining nodes loop: " << remainingNodes.size() << " nodes left ---");
		//Get 1st element, i.e. the element with the lowest distance
		Vertex<PTYPE>* u = remainingNodes.top();
		LOG4CPLUS_TRACE(pathlogger, "Processing node: " << u->getName());

		PTYPE uWeight = u->getkthShortestPath(1)->L;
		if (uWeight == 0) {
			//All remaining nodes are unreachable --> shouldn't happen!
			LOG4CPLUS_ERROR(pathlogger, "found unreachable nodes in Dijkstra's algorithm");
			break;
		}

		//Remove u from remainingNodes
		remainingNodes.pop();

		LOG4CPLUS_TRACE(pathlogger, "Node's weight is " << uWeight);

		if (u->outEdges.size() < 1) {
			LOG4CPLUS_TRACE(pathlogger, u->getName() << " does not have any outgoing edges");
			continue;
		}

		//Change distance of all adjacent nodes
		for (edge_it it = u->outEdges.begin(); it != u->outEdges.endMarked(); it++) {
			Edge<PTYPE>* edge = *it;
			Vertex<PTYPE>* trgNode = edge->target;
			LOG4CPLUS_TRACE(pathlogger, "Target: " << trgNode->getName());
			PTYPE newDist = edge->prob * uWeight;
			PTYPE currDist = trgNode->getkthShortestPath(1)->L;

			//If we've found a more probable path, save new prob & predesccor
			if (newDist > currDist) {
				LOG4CPLUS_TRACE(pathlogger, "Changing distance of " << trgNode->getName() << " from " << currDist << " to " << newDist);
				Path* shortestPath = trgNode->getkthShortestPath(1);
				shortestPath->L = newDist;
				shortestPath->pred = u->getkthShortestPath(1);
				//Successor info for assembling the RegEx

				shortestPath->edge = edge;

				//Priority queue filled here because priority updates are not noted
				remainingNodes.push(trgNode);
			}
		}
	}
}

/*
 * This function corresponds to the NextPath function as described in the paper by Jimenez & Marzal
 */
void PathFinder::nextPath(Vertex<PTYPE>* targetNode, unsigned int k) {
	assert(k >= 2);


	LOG4CPLUS_TRACE(pathlogger, "Computing: NextPath(" << targetNode->getName() << "," << k << ")");

	if (k == 2) {
		LOG4CPLUS_TRACE(pathlogger, "k = 2 => do step B1");
		//B.1: The shortest paths to the direct predecessors of our target node
		//concatenated with the edge to our target node are the candidates for
		//the next path

		EdgeContainer& edges = targetNode->inEdges;
		for (edge_it it = edges.begin(); it != edges.endMarked(); it++) {

			Edge<PTYPE>* edge = *it;

			bool followAbstract = !edge->source->isConcretized;
			if (!followAbstract && edge->isAbstract()) {
				LOG4CPLUS_TRACE(pathlogger, "Can't use " << (*edge) << ", has wrong state (" << edge->isAbstract() << ") and followAbstract is " << followAbstract);
				continue;
			} else {
				LOG4CPLUS_TRACE(pathlogger, "Considering: " << (*edge));
			}

			//Don't insert the very shortest path into the list of candidates,
			//we have obviously already used that for k = 1
			//if (pred == targetNode->mShortestPaths[0]->pred) continue;
			if (edge == targetNode->getkthShortestPath(1)->edge) {
				LOG4CPLUS_TRACE(pathlogger, "Edge used in 1st shortest path -> skip");
				continue;
			}

			LOG4CPLUS_TRACE(pathlogger, "Will add edge as candidate.");

			Path* pred = (edge)->source->getkthShortestPath(1);
			Path *peter = new Path(k, edge->prob * pred->L, pred, targetNode, edge);
			targetNode->mCandidates.push(Path(k, edge->prob * pred->L, pred, targetNode, edge));

			LOG4CPLUS_TRACE(pathlogger, targetNode->getName() << ":B1: Added 1st shortest path or 2nd edge from " << (*it)->source->getName() << ", prob: " << (*it)->prob * pred->L);
		}
		//If there is a loop on the target node, we might also take the shortest
		//path to the target node and loop there
		//12/22: Not necessary any more, have separate edges for cumulated loop prob

		if (edges.size() > 0) {
			LOG4CPLUS_TRACE(pathlogger, targetNode->getName() << ":B1: Computed " << (edges.size()-1) << " candidates for k=2");
		}
		else
			LOG4CPLUS_TRACE(pathlogger, targetNode->getName() << ":B1: Recursion stops, no candidates found for k=2");
	}

	//B.2 [the very shortest path to the input node has length 0, so we skip B.3-B.5 if the condition holds]
	if (!(k == 2 && targetNode == mCexGen->mCexInputNode)) {
		LOG4CPLUS_TRACE(pathlogger, targetNode->getName() << ":B2: Conditions hold, execute B3 to B5");
		//In NextPath(v, k-1) a path "using" predecessor u to get to v was computed
		//We therefore have to compute the next shortest path to u (if existent)
		//and use it to generate a new candidate. All other candidates remain unchanged

		//B.3
		Path* pathKminus1 = targetNode->getkthShortestPath(k - 1)->pred; //Note: looking at the path for k-1
		unsigned int kprime = pathKminus1->k;
		Vertex<PTYPE>* u = pathKminus1->node;
		LOG4CPLUS_TRACE(pathlogger, targetNode->getName() << ":B3: u = " << u->getName() << ", k' = " << kprime);

		//B.4: Compute the next shortest path to u if necessary
		if (u->mShortestPaths.size() < kprime + 1 || u->mShortestPaths[kprime] == NULL) {
			LOG4CPLUS_TRACE(pathlogger, targetNode->getName() << ":B4: have to compute next shortest path for u");
			nextPath(u, kprime + 1);
		} else {
			LOG4CPLUS_TRACE(pathlogger, targetNode->getName() << ":B4: path for u (k = " << (kprime+1) << ") already known: " << u->mShortestPaths[kprime]->toStringDebug());
		}

		//B.5: If that path exists, we have a new candidate
		if (u->mShortestPaths.size() >= kprime + 1 && u->mShortestPaths[kprime] != NULL) {
			LOG4CPLUS_TRACE(pathlogger, targetNode->getName() << ":B5: path exists, maybe add candidate");
			Path* pred = u->mShortestPaths[kprime];

			Edge<PTYPE>* edge = targetNode->getkthShortestPath(k - 1)->edge;

			targetNode->mCandidates.push(Path(k, edge->prob * pred->L, pred, targetNode, edge));
			LOG4CPLUS_TRACE(pathlogger, targetNode->getName() << ":B5: Added candidate has prob: " << edge->prob * pred->L);
		} else {
			LOG4CPLUS_TRACE(pathlogger, targetNode->getName() << ":B5: no more paths from u exist, no new candidate");
		}
	}

	//B.6
	if (targetNode->mCandidates.empty()) {
		LOG4CPLUS_TRACE(pathlogger, targetNode->getName() << ":B6: No more paths for target node");
	} else {
		//Evil tracing function of totally crappy performance, don't use in normal circumstances
		if (DEFAULT_LOG_LEVEL <= TRACE_LOG_LEVEL && !targetNode->mCandidates.empty()) {
			LOG4CPLUS_TRACE(pathlogger, "Currently known candidates:");
			vector<Path> currentCandidates;
			while (!targetNode->mCandidates.empty()) {
				currentCandidates.push_back(targetNode->mCandidates.top());
				targetNode->mCandidates.pop();
			}
			for (unsigned int i = 0; i < currentCandidates.size(); i++) {
				LOG4CPLUS_TRACE(pathlogger, "- " << currentCandidates[i].toStringDebug());
				targetNode->mCandidates.push(currentCandidates[i]);
			}
			currentCandidates.clear();
		}

		const Path& newCand = targetNode->mCandidates.top();

		Path* kthShortestPath = new Path(k, newCand.L, newCand.pred, newCand.node, newCand.edge);

		targetNode->setkthShortestPath(k, kthShortestPath);
		LOG4CPLUS_TRACE(pathlogger, targetNode->getName() << ":B6: have found kth shortest path from " << newCand.pred->node->getName() << " (L: " << newCand.L << ")");

		targetNode->mCandidates.pop();
	}
}

}

