/**
 *   COMICS - Computing Minimal Counterexamples for Discrete-time Markov Chains
 *
 *   COMICS is a stand-alone tool which performs model checking and the generation
 *   of counterexamples for discrete-time Markov Chains (DTMCs). *
 *
 *   Copyright (C) <2012> <RWTH Aachen University>
 *   Authors: Nils Jansen, Erika Abraham, Jens Katelaan, Maik Scheffler, Matthias Volk, Andreas Vorpahl
 *
 *   This program is free software: you can redistribute it and/or modify
 *   it under the terms of the GNU General Public License as published by
 *   the Free Software Foundation, either version 3 of the License, or
 *   (at your option) any later version.
 *
 *   This program is distributed in the hope that it will be useful,
 *   but WITHOUT ANY WARRANTY; without even the implied warranty of
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *   GNU General Public License for more details.
 *
 *   You should have received a copy of the GNU General Public License
 *   along with this program.  If not, see <http://www.gnu.org/licenses/>.
 *
 *   Main Contact:
 *
 *   Nils Jansen
 *   Theory of Hybrid Systems
 *   RWTH Aachen
 *   52056 Aachen
 *   Germany
 *   nils.jansen@cs.rwth-aachen.de
 *
 */

/*
 * IOWrapper.cpp
 *
 * TODO: Fix: Several code segments are based on continuous node numbering in the original file (auxiliary node ids, target node flags). This is not exactly ideal. It could be fixed with relative ease by using a one-to-one mapping of given node ids to the numbers 0,1,...
 *
 *  Created on: 01.12.2010
 *      Author: jens
 */

#include <iostream>
#include <fstream>
#include <sstream>
#include <stdlib.h>
#include <utility>
#include <algorithm>
#include <cctype>
#include <string>

#include <sys/types.h>
#include <sys/stat.h>
#include <dirent.h>

#include "IOWrapper.h"
#include "InputParsers.h"
#include "OutputFormatters.h"
#include "Configuration.h"
#include "../cex/Closure.h"
#include "../cex/AbstractStateSelector.h"
#include "../cex/gens/ConcreteCexGen.h"
#include "../cex/gens/AbstractCexGen.h"

#ifdef USE_LOGLIB
#include <log4cplus/logger.h>
#include <log4cplus/configurator.h>
#endif

#ifdef JNI
#include "../SCCInterface.h"
#endif

#ifdef _WIN32 || _WIN64
#define SLASH '\\'
#else
#define SLASH '/'
#endif

using namespace std;
using namespace log4cplus;

const char* JAVA = "RETURN_TO_JAVA";
const char* DEFAULT_RESULT_FILE = "result.dtmc";

namespace scc_cex {

Logger iologger = Logger::getInstance("IOWrapper");

CexGenerator* make_cexgen(Graph<PTYPE>* g, CounterExampleParams* task) {
	if (task->mUseAbstractSearch)
		return new AbstractCexGen(g);
	else
		return new ConcreteCexGen(g);
}

IOWrapper::IOWrapper(string str, unsigned int input_flags, unsigned int heuristic_flags) {
	initLogging();

	mStepSize = UINT_MAX;
	mIterationCount = 1;
	mProbBound = 0;
	mProbModelCheck = 0;
	mResultFileName = "";
	mGenerateCEx = false;

	ConfigurationSingleton *conf = ConfigurationSingleton::getInstance();

	//Configuration::setFlags(input_flags);
	mHeuristicsBenchmark = conf->isIoflBenchmarkHeuristic();
	mSearchBenchmark = conf->isIoflBenchmarkSearch();
	mSelectionBenchmark = conf->isIoflBenchmarkSelection();
	mConcBenchmark = conf->isIoflBenchmarkConc();
	mDoAbstractSearch = !conf->isIoflCexConcrete();
	mDoGlobalSearch = !conf->isIoflCexLocal();
	mWaitForExternalCalls = conf->isIoflWaitForCalls();

	//Configuration::setHeuristicFlags(heuristic_flags);
	mChooseByDegree = conf->isIoflChooseDegree();
	mChooseByProbability = conf->isIoflChooseProbability();
	mChooseByMembership = conf->isIoflChooseMembership();

	mModelChecker = NULL;

	if (conf->isIoflNoModelChecking()) {
		if (mDoAbstractSearch) {
			LOG4CPLUS_WARN(iologger, "When skipping the modelcheker, you can't use the abstract search. Switch to concrete search");
			mDoAbstractSearch = false;
			conf->setIoflCexConcrete(true);
			conf->setIoflForceAbstract(false);
		}
	}

	istream* strm;
	if (conf->isIoflInputIsFile()) {

		if (!scc_cex::doesFileExist(str)) {
			LOG4CPLUS_ERROR(iologger, "The given file (" << str << ") doesn't exist");
			return;
		}

		inputFile = str;

		//create input stream from file
		strm = new ifstream(str.c_str(), ifstream::in);

		//We need to remember the directory of the .conf file to interpret the file names given there!
		mPath = "";
		string::size_type lastSlashPos = str.find_last_of(SLASH);
		if (lastSlashPos != std::string::npos) {
			mPath = str.substr(0, lastSlashPos);
		}

		LOG4CPLUS_INFO(iologger, "Will parse " << str);
	} else {
		//create input stream from string
		strm = new istringstream(str, ifstream::in);
	}
	init(strm);
}

IOWrapper::IOWrapper(string str, unsigned int input_flags) {
	initLogging();

	mStepSize = UINT_MAX;
	mIterationCount = 1;
	mProbBound = 0;
	mProbModelCheck = 0;
	mGenerateCEx = false;

	ConfigurationSingleton *conf = ConfigurationSingleton::getInstance();

//	Configuration::setFlags(input_flags);
	mHeuristicsBenchmark = conf->isIoflBenchmarkHeuristic();
	mSearchBenchmark = conf->isIoflBenchmarkSearch();
	mSelectionBenchmark = conf->isIoflBenchmarkSelection();
	mConcBenchmark = conf->isIoflBenchmarkConc();
	mDoAbstractSearch = !conf->isIoflCexConcrete();
	mDoGlobalSearch = !conf->isIoflCexLocal();
	mWaitForExternalCalls = conf->isIoflWaitForCalls();

	//set variables for heuristic selection to false
	mChooseByDegree = false;
	mChooseByMembership = false;
	mChooseByProbability = false;

	mModelChecker = NULL;

	if (conf->isIoflNoModelChecking()) {
		if (mDoAbstractSearch) {
			LOG4CPLUS_WARN(iologger, "When skipping the modelcheker, you can't use the abstract search. Switch to concrete search");
			mDoAbstractSearch = false;
			conf->setIoflCexConcrete(true);
			conf->setIoflForceAbstract(false);
		}
	}

	istream* strm;
	if (conf->isIoflInputIsFile()) {

		if (!scc_cex::doesFileExist(str)) {
			LOG4CPLUS_ERROR(iologger, "The given file (" << str << ") doesn't exist");
			return;
		}

		inputFile = str;

		//create input stream from file
		strm = new ifstream(str.c_str(), ifstream::in);

		//We need to remember the directory of the .conf file to interpret the file names given there!
		mPath = "";
		string::size_type lastSlashPos = str.find_last_of(SLASH);
		if (lastSlashPos != std::string::npos) {
			mPath = str.substr(0, lastSlashPos);
		}

		LOG4CPLUS_INFO(iologger, "Will parse " << str);
	} else {
		//create input stream from string
		strm = new istringstream(str, ifstream::in);
	}
	init(strm);
}

IOWrapper::IOWrapper(string str) {
	initLogging();

	mStepSize = UINT_MAX;
	mIterationCount = 1;
	mProbBound = 0;
	mProbModelCheck = 0;
	mGenerateCEx = false;

	ConfigurationSingleton *conf = ConfigurationSingleton::getInstance();

//	Configuration::setFlags(input_flags);
	mHeuristicsBenchmark = conf->isIoflBenchmarkHeuristic();
	mSearchBenchmark = conf->isIoflBenchmarkSearch();
	mSelectionBenchmark = conf->isIoflBenchmarkSelection();
	mConcBenchmark = conf->isIoflBenchmarkConc();
	mDoAbstractSearch = !conf->isIoflCexConcrete();
	mDoGlobalSearch = !conf->isIoflCexLocal();
	mWaitForExternalCalls = conf->isIoflWaitForCalls();

	//set variables for heuristic selection to false
	mChooseByDegree = false;
	mChooseByMembership = false;
	mChooseByProbability = false;

	mModelChecker = NULL;

	istream* strm;
	if (conf->isIoflInputIsFile()) {

		if (!scc_cex::doesFileExist(str)) {
			LOG4CPLUS_ERROR(iologger, "The given file (" << str << ") doesn't exist");
			return;
		}

		inputFile = str;

		//create input stream from file
		strm = new ifstream(str.c_str(), ifstream::in);

		//We need to remember the directory of the .conf file to interpret the file names given there!
		mPath = "";
		string::size_type lastSlashPos = str.find_last_of(SLASH);
		if (lastSlashPos != std::string::npos) {
			mPath = str.substr(0, lastSlashPos);
		}

		LOG4CPLUS_INFO(iologger, "Will parse " << str);
	} else {
		//create input stream from string
		strm = new istringstream(str, ifstream::in);
	}
	init(strm);
}

IOWrapper::~IOWrapper() {
	delete mModelChecker;

	for (vtx_it it = mVertexList.begin(); it != mVertexList.end(); it++)
		delete *it;
	mVertexList.clear();
	mInitialNode = NULL;
}

void IOWrapper::init(istream *strm) {
	if (ConfigurationSingleton::getInstance()->isIoflExternalCaller()) {
		mResultFile = JAVA;
	} else {
		mResultFile = DEFAULT_RESULT_FILE;
	}
	mStrm = strm;
	mAtPropsGiven = false;
	mOperator = OPERATOR_LESS;
	mInitialNode = NULL;
	mTargetNodes.clear();
	mTargetNodes.push_back(false);

	//Reset several values to avoid conflicts
	resetClosure = true;
}

void IOWrapper::initLogging() {
#ifdef USE_LOGLIB
	BasicConfigurator config;
	config.configure();
#endif
	iologger.setLogLevel(DEFAULT_LOG_LEVEL);
}

bool IOWrapper::readFiles() {
	bool couldRead = readFiles(mStrm);
	if (!couldRead) {
		LOG4CPLUS_ERROR(iologger, "The input file(s) couldn't be read - terminating.");
		return false;
	}
	return true;
}

Graph<PTYPE>* IOWrapper::runModelChecking() {
	//All input read -> now do the model checking
	mModelChecker = new SccMc(false);

	//In case we (might) want to run a counter example search,
	//disable the preprocessing that we do in MC-only mode
	if (mGenerateCEx || mWaitForExternalCalls) {
		mModelChecker->disablePreprocessing(true);
	}

	mModelChecker->setAuxInputNode(mInitialNode);
	Graph<PTYPE>* g = mModelChecker->modelCheck(mVertexList, mInitialNodeID, mTargetNodes, mVtxToPropsMap);
	mResultFileName = writeMCResult(g);
	return g;
}

pair<CexGenerator*, CounterExampleParams*> IOWrapper::initCexSearch(Graph<PTYPE>* g, bool abstractFlag, bool globalFlag, bool closureFlag) {
	if (!probabilityIsRefuted()) {
		LOG4CPLUS_INFO(iologger, "Won't generate counter example: The property is not refuted");
		return make_pair<CexGenerator*, CounterExampleParams*>(NULL, NULL);
	}

	ConfigurationSingleton *conf = ConfigurationSingleton::getInstance();

	//Set parameter
	LOG4CPLUS_INFO(iologger, "Flags: abstract: " << abstractFlag << ", global: " << globalFlag << ", closure: " << closureFlag);
	CounterExampleParams* params = new CounterExampleParams(globalFlag, abstractFlag, CONCRETIZE_ONE, SELECT_ALL, FILTER_NONE);
	//unsigned int flags = Configuration::getInstance().getFlags();
	if (!abstractFlag) {
		conf->setIoflCexConcrete(true);
	}
	if (!globalFlag) {
		conf->setIoflCexLocal(true);
	}
	if (!closureFlag) {
		conf->setIoflBoundPathsum(true);
	}

	CexGenerator* cexGenerator = make_cexgen(g, params);

	return make_pair<CexGenerator*, CounterExampleParams*>(cexGenerator, params);
}

void IOWrapper::executeCexSteps(CexGenerator* cexGenerator, CounterExampleParams* params, unsigned int steps) {
	if (UINT_MAX - steps > cexGenerator->getStepLimit()) {
		cexGenerator->setStepLimit(cexGenerator->getStepLimit() + steps);
		LOG4CPLUS_DEBUG(iologger, "Will do up to " << steps << " concretization steps");
	} else {
		cexGenerator->setStepLimit(UINT_MAX);
		LOG4CPLUS_DEBUG(iologger, "Will finish counter example search.");
	}
	LOG4CPLUS_DEBUG(iologger, "FILTERMODE: " + filterModeToString(params->mFilterMode));
	;
	cexGenerator->findCounterExample(mProbBound, mOperator, mTargetIndex, params);
	LOG4CPLUS_DEBUG(iologger, cexGenerator->counterExampleToString());
}

/**
 * A complete run of the MC/Cex Generator, i.e.
 * - Read files
 * - Run MC
 * - Possibly run counter example search
 * - Possibly run counter example search again if in benchmark mode
 * - Write the results to files
 */
void IOWrapper::run() {
	bool couldRead = readFiles();
	if (!couldRead) {
		return;
	}

	//SccMc::reachabilityAnalysis(mVertexList, mInitialNodeID, mTargetNodes, )

	Graph<PTYPE>* g;

	if (!ConfigurationSingleton::getInstance()->isIoflReadFromXml()) {
		if (!ConfigurationSingleton::getInstance()->isIoflNoModelChecking()) {
			Statistics::getInstance()->startModelCheckerTime();
			g = runModelChecking();
			setProbModelCheck(getResultProbability());
			Statistics::getInstance()->stopModelCheckerTime();
		} else {
			g = new Graph<PTYPE>();
			g->inputVertices.push_back(mVertexList[mInitialNodeID]);
			g->outputVertices.push_back(mVertexList[mTargetIndex]);
			mVertexList[mInitialNodeID]->isInputVertex = true;
			mVertexList[mInitialNodeID]->isConcretized = false;
			mVertexList[mTargetIndex]->g = NULL;
			for (int i = 1; i < mVertexList.size(); i++) {
				if (i != mTargetIndex) {
					mVertexList[i]->g = g;
					g->allVertices.push_back(mVertexList[i]);
				}
			}
		}
	} else {
		std::ifstream stream(inputFile.c_str());
		g = getGraphFromXml(stream);
		if (mProbModelCheck == 0.0) {
			SccMc* modelChecker = new SccMc(false);
			Statistics::getInstance()->startModelCheckerTime();
			g = runModelChecking();
			Statistics::getInstance()->stopModelCheckerTime();
			setProbModelCheck(getResultProbability());

		}
	}

	if (ConfigurationSingleton::getInstance()->isIoflCexConcrete()) {
		mDoAbstractSearch = false;
	}

	//Display graph read from the input
//	AbstractOutputFormatter<PTYPE> *formatter;
//	if (ConfigurationSingleton::getInstance()->isIoflExternalCaller()) {
//		formatter = new XMLOutputFormatter<PTYPE>(mInitialNode, mTargetNodes, true, false);
//	} else {
//		formatter = new XMLOutputFormatter<PTYPE>(mInitialNode, mTargetNodes, true, false);
//	}

//	LOG4CPLUS_TRACE(iologger, formatter->generateOutput(*g));

	//Generate counter example(s), if applicable
	if (!mGenerateCEx) {
		LOG4CPLUS_INFO(iologger, "Model Checking result: " << getResultProbability());
		return;
	}

	if ((!probabilityIsRefuted()) && (!ConfigurationSingleton::getInstance()->isIoflNoModelChecking())) {
		LOG4CPLUS_INFO(iologger, "Won't generate counter example: The property is not refuted");
		return;
	}

	if (ConfigurationSingleton::getInstance()->isIoflOnlyModelChecking()) {
		LOG4CPLUS_INFO(iologger, "Model Checking result: " << getResultProbability());
		return;
	}

	//TODO: Should make the filename of the counter example summary customizable
	fstream outputFile("counter_example_summary.txt", fstream::out);

	//Generate counter example task sequence
	vector<CounterExampleParams*> taskSequence = generateTaskSequence();
	for (vector<CounterExampleParams*>::iterator task_it = taskSequence.begin(); task_it != taskSequence.end(); task_it++) {
		LOG4CPLUS_INFO(iologger, "Next task: " << **task_it);

		//CexGenerator cexGenerator(g);
		CexGenerator* cexGenerator = make_cexgen(g, *task_it);
		cexGenerator->setModelCheckingProbability(getResultProbability());
		//unsigned int flags = Configuration::getInstance().getFlags();
		bool writeCex = false;
		if (ConfigurationSingleton::getInstance()->isIoflIoflSaveIterations())
			writeCex = true;

		//if (mStepLimit != UINT_MAX) cexGenerator->setStepLimit(mStepLimit);

		//cexGenerator->findCounterExample(mProbBound, mOperator, mTargetIndex, *task_it);
		cexGenerator->setStepLimit(0);

		//Execute <mIterationCount> many steps with stepsize <mStepSize>
		LOG4CPLUS_INFO( iologger, "Iterations: " << mIterationCount << ", Stepsize: " << mStepSize);
		unsigned int iteration = 1;

		while (!cexGenerator->isCexComplete() && iteration <= mIterationCount) {
			LOG4CPLUS_DEBUG(iologger, "Iteration no: " << iteration);
			executeCexSteps(cexGenerator, *task_it, mStepSize);

			//Output
			//Nils: Output in other file? User defined?
			if (writeCex) {
				stringstream filename;
				filename << "counter_example_" << (*task_it)->getParamHash() << "-" << iteration << ".dtmc";
				fstream cexFile(filename.str().c_str(), fstream::out);
				cexFile << cexGenerator->counterExampleToString();
				flush(cexFile);
				cexFile.close();
			}
			iteration++;
		}

		outputFile << cexGenerator->getSummary();
		flush(outputFile);

		//Write counter example to string
		//TODO make user definied
		string filename = "counter_example";
		if (ConfigurationSingleton::getInstance()->isIoflBoundPathsum()){
			filename += ".path";
		} else {
			filename += ".dtmc";
		}
		fstream cexFile(filename.c_str(), fstream::out);
		cexFile << cexGenerator->counterExampleToString();
		flush(cexFile);
		cexFile.close();
		LOG4CPLUS_INFO(iologger, "Counter example can be found in: " << filename);

		//Undo the changes to the graph so that we can use it again
		if (*task_it != taskSequence.back()) {
			g->deepUnconcretize();

			for (vtx_it it = g->allVertices.begin(); it != g->allVertices.end(); it++) {
				(*it)->seen = false;
				(*it)->isInClosure = false;
				if ((*it)->sisterVtx != NULL) {
					delete (*it)->sisterVtx;
					(*it)->sisterVtx = NULL;
				}
			}

			for (vtx_it it = g->outputVertices.begin(); it != g->outputVertices.end(); it++) {
				(*it)->seen = false;
				(*it)->isInClosure = false;
				if ((*it)->sisterVtx != NULL) {
					delete (*it)->sisterVtx;
					(*it)->sisterVtx = NULL;
				}
			}

			resetClosure = true;

			LOG4CPLUS_INFO( iologger, "Reset counter example data, will now start search with new params");
		}

		delete cexGenerator;

	}

	//LOG4CPLUS_INFO(iologger, "Result can be found in: " << mResultFileName);
	outputFile.close();
}

vector<CounterExampleParams*> IOWrapper::generateTaskSequence() const {
	vector<CounterExampleParams*> taskSequence;

	//Lots of benchmark variants...
	unsigned int searchCount = (mSearchBenchmark ? 4 : 1);
	unsigned int highestFilter = (mHeuristicsBenchmark ? FILTER_RELATIVE_USAGE : FILTER_NONE);
	bool allSelectionModes = mSelectionBenchmark;

	for (unsigned int searchnr = 1; searchnr <= searchCount; searchnr++) {
		bool global = mDoGlobalSearch;
		bool abstract = mDoAbstractSearch;

		if (searchnr == 2 || searchnr == 4)
			global = !global;

		if (searchnr == 3 || searchnr == 4)
			abstract = !abstract;

		unsigned int concModeMin = (abstract ? CONCRETIZE_SQRT : CONCRETIZE_ONE);
		unsigned int concModeMax = (abstract ? (mConcBenchmark ? CONCRETIZE_ALL : CONCRETIZE_SQRT) : CONCRETIZE_ONE);

		unsigned int chosenFilter = FILTER_NONE;
		if (mChooseByDegree) {
			chosenFilter = FILTER_INPUT_OUTPUT_DEGREE;
		}
		if (mChooseByMembership) {
			chosenFilter = FILTER_RELATIVE_USAGE;
		}
		if (mChooseByProbability) {
			chosenFilter = FILTER_ABSTRACT_EDGE_PROBABILITY;
		}

		for (unsigned int concMode = concModeMin; concMode <= concModeMax; concMode++) {

			if (concMode == CONCRETIZE_ALL)
				highestFilter = FILTER_NONE; //When we take all, filtering has of course no effect

//			if (mHeuristicsBenchmark) {
			for (unsigned int filter = FILTER_NONE; filter <= highestFilter; filter++) {
				if (!highestFilter == FILTER_NONE) {
					chosenFilter = filter;
				}
				//vorpahl: do _not_ comment the next command. I dont know why it has to be here twice, but global search on xml is not working with the next line commented
				taskSequence.push_back(new CounterExampleParams(global, abstract, (ConcretizationMode) (concMode), SELECT_ALL, (FilterMode) (chosenFilter)));
				//taskSequence.push_back(new CounterExampleParams(global, abstract, (ConcretizationMode) (concMode), SELECT_ALL, (FilterMode) (chosenFilter)));
				//If all or none are to be selected, ignore allSelectionModes
				if (allSelectionModes && !(concMode == CONCRETIZE_ALL) && abstract) {
					taskSequence.push_back(new CounterExampleParams(global, abstract, (ConcretizationMode) (concMode), SELECT_HIGHEST, (FilterMode) (chosenFilter)));
					taskSequence.push_back(new CounterExampleParams(global, abstract, (ConcretizationMode) (concMode), SELECT_DEEPEST, (FilterMode) (chosenFilter)));
				}
				//It doesn't make sense to try concretization heuristics on a concrete system...
				if (!abstract)
					break;
			}
//			}
			//It doesn't make sense to switch between concretization modes on a concrete system...
			if (!abstract)
				break;
		}
	}

	return taskSequence;
}

PTYPE IOWrapper::getResultProbability() const {
	if (mModelChecker != NULL) {
		return mModelChecker->getTargetResult(mTargetIndex);
	} else if (ConfigurationSingleton::getInstance()->isIoflExternalCaller() || ConfigurationSingleton::getInstance()->isIoflReadFromXml()) {
		return mProbModelCheck;
	} else {
		return 0;
	}
}

bool IOWrapper::probabilityIsRefuted() const {
	if (!ConfigurationSingleton::getInstance()->isIoflNoModelChecking()) {
		PTYPE mcResult = getResultProbability();
		if (mcResult < mProbBound) {
			LOG4CPLUS_ERROR( iologger, "Can't find counter example because the probability isn't refuted" << " (MC result: " << mcResult << "; Property: " << mProbBound << ")");
			return false;
		}
		return true;
	}

}

void IOWrapper::readSCCInXML(Graph<PTYPE>* res, pugi::xml_node node) {
	//pugi::xml_attribute sccId = node.attribute("id");
	//res->graph_id = sccId.as_int();

	//unsigned int value;

	pugi::xml_node vtx = node.child("vtx");
	pugi::xml_node inp = node.child("inp");
	pugi::xml_node out = node.child("out");
	pugi::xml_node edg = node.child("edg");

	while (inp) {
		unsigned int nodeId = atoi(inp.child_value());
		nodeId--;
		for (unsigned int i = mVertexList.size(); i <= nodeId; i++) {
			mVertexList.push_back(NULL);
		}
		if (mVertexList[nodeId] != NULL) {
			mVertexList[nodeId]->g = res;
		}
		res->inputVertices.push_back(mVertexList[nodeId]);
		res->allVertices.push_back(mVertexList[nodeId]);
		inp = inp.next_sibling("inp");
	}

	while (out) {
		unsigned int nodeId = atoi(out.child_value());
		nodeId--;
		for (unsigned int i = mVertexList.size(); i <= nodeId; i++) {
			mVertexList.push_back(NULL);
		}
		res->outputVertices.push_back(mVertexList[nodeId]);
		out = out.next_sibling("out");
	}

	while (vtx) {
		unsigned int nodeId = atoi(vtx.child_value());
		nodeId--;
		for (unsigned int i = mVertexList.size(); i <= nodeId; i++) {
			mVertexList.push_back(NULL);
		}
		if (mVertexList[nodeId] != NULL) {
			mVertexList[nodeId]->g = res;
		}
		res->allVertices.push_back(mVertexList[nodeId]);
		vtx = vtx.next_sibling("vtx");
	}

	//TODO: read prob as PTYPE, not as double
	Edge<PTYPE> *tempedge;
	bool sccContainsAbsEdge = false;
	while (edg) {
		bool abstract1 = edg.attribute("abs").as_bool();
		double prob = edg.attribute("prb").as_double();
		//PTYPE prob = probParser.parseProb(read_prob);
		unsigned int source = edg.attribute("src").as_uint();
		unsigned int target = edg.attribute("trg").as_uint();
		sccContainsAbsEdge = abstract1;
		if (abstract1) {
			ConfigurationSingleton::getInstance()->setIoflCexConcrete(false);
		}
		source--;
		target--;

		assert(prob>0);
		tempedge = new Edge<PTYPE>(mVertexList[source], mVertexList[target], prob);
		tempedge->state = abstract1;
		tempedge->not_m_min = true;

		edg = edg.next_sibling("edg");
	}

	res->state = CONCRETE;
	res->concretized = true;

	//check if the scc is abstract, i.e. has at least one outgoing abstract edge
	for (vtx_it it = res->allVertices.begin(); it != res->allVertices.end(); it++) {
		for (edge_it eit = (*it)->outEdges.begin(); eit != (*it)->outEdges.end(); eit++) {
			if ((*eit)->state == ABSTRACT) {
				res->state = ABSTRACT;
				res->concretized = false;
			}
		}
	}

	for (vtx_it it = res->inputVertices.begin(); it != res->inputVertices.end(); it++) {
		for (edge_it eit = (*it)->outEdges.begin(); eit != (*it)->outEdges.end(); eit++) {
			if ((*eit)->state == ABSTRACT) {
				res->state = ABSTRACT;
				res->concretized = false;
			}
		}
	}

	int counter = 0;
	pugi::xml_node childScc = node.child("scc");
	while (childScc) {
		counter++;
		//pugi::xml_attribute sccId = childScc.attribute("id");
		//Graph<PTYPE> *temp = res->addSubgraph(sccId.as_int());
		Graph<PTYPE> *temp = res->addSubgraph();
		readSCCInXML(temp, childScc);
		//res->addSubgraph(temp);
		childScc = childScc.next_sibling("scc");
	}

	if (counter == 0) {
		//res->concretized = true;
	}

}

Graph<PTYPE>* IOWrapper::getGraphFromXml(std::string xmlString) {
	stringstream ss;
	ss << xmlString;
	return getGraphFromXml(ss);
}

Graph<PTYPE>* IOWrapper::getGraphFromXml(std::istream& stream) {
	fstream f;
	stringstream so;
	so << stream;
	f.open("test.xml", ios::out);
	f << so.str() << endl;
	f.close();

	pugi::xml_document doc;
	//pugi::xml_parse_result result = doc.load_file(xmlPath.c_str());
	pugi::xml_parse_result result = doc.load(stream);
	LOG4CPLUS_INFO(iologger, "Load result: " << result.description());

	ConfigurationSingleton::getInstance()->setIoflCexConcrete(true);

	Graph<PTYPE> *resG = NULL; //new Graph<PTYPE>();
	Graph<PTYPE> *auxTargetGraph = NULL;

	/* The first scc child contains all nodes, so we initialize the nodelist */

	pugi::xml_node firstScc = doc.child("dtmc").child("scc");

	pugi::xml_node vtx = firstScc.child("vtx");
	pugi::xml_node out = firstScc.child("out");
	pugi::xml_node edg = firstScc.child("edg");
	pugi::xml_node inp = firstScc.child("inp");

	//read all vertices
	while (vtx) {
		unsigned int nodeId = atoi(vtx.child_value());
		nodeId--;
		for (unsigned int i = mVertexList.size(); i <= nodeId; i++) {
			mVertexList.push_back(NULL);
		}
		if (mVertexList[nodeId] == NULL) {
			mVertexList[nodeId] = new Vertex<PTYPE>();
			mVertexList[nodeId]->oindex = nodeId;
		}
		vtx = vtx.next_sibling("vtx");
	}
	while (inp) {
		unsigned int nodeId = atoi(inp.child_value());
		nodeId--;
		for (unsigned int i = mVertexList.size(); i <= nodeId; i++) {
			mVertexList.push_back(NULL);
		}
		if (mVertexList[nodeId] == NULL) {
			mVertexList[nodeId] = new Vertex<PTYPE>();
			mVertexList[nodeId]->oindex = nodeId;
		}
		inp = inp.next_sibling("out");
	}
	while (out) {
		unsigned int nodeId = atoi(out.child_value());
		nodeId--;
		for (unsigned int i = mVertexList.size(); i <= nodeId; i++) {
			mVertexList.push_back(NULL);
		}
		if (mVertexList[nodeId] == NULL) {
			mVertexList[nodeId] = new Vertex<PTYPE>();
			mVertexList[nodeId]->oindex = nodeId;
		}
		out = out.next_sibling("out");
	}

	//all vertices have been read, since input states also belong to the allVertices

	//read the target states
	pugi::xml_node tar = doc.child("dtmc").child("target");
	int numTargets = 0;
	while (tar) {
		numTargets++;
		unsigned int nodeId = atoi(tar.child_value());
		nodeId--;
		for (unsigned int i = mTargetNodes.size(); i <= nodeId; i++) {
			mTargetNodes.push_back(false);
		}
		mTargetNodes[nodeId] = true;
		mTargetIndex = nodeId;
		tar = tar.next_sibling("target");
	}

	//read the model checking result
	pugi::xml_node probMass = doc.child("dtmc").child("prob");
	if (probMass) {
		PTYPE mcResult = atof(string(probMass.child_value()).c_str());
		setProbModelCheck(mcResult);
	} else {
		setProbModelCheck(0.0);
	}

	if (inp) {
		unsigned int id = atoi(inp.child_value());
		id--;
		mInitialNode = mVertexList[id];
		mInitialNodeID = id;
	}

	//TODO read as PTYPE
	Edge<PTYPE> *tempedge;
	while (edg) {
		bool abstract = edg.attribute("abs").as_bool();
		PTYPE prob = edg.attribute("prb").as_double();
		//PTYPE prob = probParser.parseProb(read_prob);
		unsigned int source = edg.attribute("src").as_uint();
		unsigned int target = edg.attribute("trg").as_uint();
		source--;
		target--;
		if (abstract) {
			ConfigurationSingleton::getInstance()->setIoflCexConcrete(false);
		}
		if (!mTargetNodes[source]) {
			assert(prob>0);
			tempedge = new Edge<PTYPE>(mVertexList[source], mVertexList[target], prob);
			tempedge->state = abstract;
			tempedge->not_m_min = true;

		}

		edg = edg.next_sibling("edg");
	}

	if ((numTargets > 1) && (mProbModelCheck == 0.0)) {

		//auxTargetGraph = new Graph<PTYPE>();
		//auxTargetGraph = new Graph<PTYPE>(2450);
		//mark new scc as abstract
		//auxTargetGraph->state = true;
		//add all nodes to the graphs allVertices list
		//for (unsigned int i = 0; i < mVertexList.size(); i++) {
		//	auxTargetGraph->allVertices.push_back(mVertexList[i]);
		//	mVertexList[i]->g = auxTargetGraph;
		//}

		//now generate the new auxiliary target node
		Vertex<PTYPE>* auxTargetNode = new Vertex<PTYPE>();
		auxTargetNode->oindex = mVertexList.size();
		mVertexList.push_back(auxTargetNode);
		//auxTargetGraph->outputVertices.push_back(auxTargetNode);

		//we will have to add a new input node for our new scc as well
		//Vertex<PTYPE>* newInputNode = new Vertex<PTYPE>();
		//newInputNode->oindex = mVertexList.size();
		//newInputNode->g = auxTargetGraph;
		//mVertexList.push_back(newInputNode);
		//auxTargetGraph->allVertices.push_back(newInputNode);
		//auxTargetGraph->inputVertices.push_back(newInputNode);

		//the graph containing the actual xml input is a child of the new generated scc
		//pugi::xml_attribute sccId = firstScc.attribute("id");
		//resG = auxTargetGraph->addSubgraph(sccId.as_int());
		//resG = auxTargetGraph->addSubgraph();

		mTargetIndex = auxTargetNode->oindex;

		//add an abstract edge from the new input node to the auxiliary target node
		//Edge<PTYPE>* e1 = new Edge<PTYPE>(newInputNode, auxTargetNode, 1.0);
		//e1->state = true;
		//e1->not_m_min = true;

		//... and an edge from the newInput node to the old one, not abstract with prob 1
		//e1 = new Edge<PTYPE>(newInputNode, mInitialNode, 1.0);
		//e1->state = false;
		//e1->not_m_min = true;

		//...and edges from the old target states to the new one
		for (unsigned int i = 0; i <= mTargetNodes.size(); i++) {
			if (mTargetNodes[i]) {
				Edge<PTYPE> *e1 = new Edge<PTYPE>(mVertexList[i], auxTargetNode, 1.0);
				e1->not_m_min = true;
				e1->state = CONCRETE;
				//now this is not a target state any more
				mTargetNodes[i] = false;
			}
		}

		//Finally mark new node as target node
		for (unsigned int i = mTargetNodes.size(); i <= auxTargetNode->oindex; i++) {
			mTargetNodes.push_back(false);
		}
		mTargetNodes[auxTargetNode->oindex] = true;
		auxTargetNode->g = NULL;

		LOG4CPLUS_INFO(iologger, "Added auxiliary target node with id " << auxTargetNode->oindex);

		//set the global input node to the input node of our new scc
		//mInitialNode = newInputNode;
		//mInitialNodeID = newInputNode->oindex;
	}

	if (resG == NULL) {
		resG = new Graph<PTYPE>();
		//pugi::xml_attribute sccId = firstScc.attribute("id");
		//resG->graph_id = sccId.as_int();
	}

	out = firstScc.child("out");
	while (out) {
		unsigned int nodeId = atoi(out.child_value());
		nodeId--;
//		for (int i = mVertexList.size(); i <= nodeId; i++) {
//			mVertexList.push_back(NULL);
//		}
		if (mVertexList[nodeId] != NULL) {
			//mVertexList[nodeId] = new Vertex<PTYPE>();
			mVertexList[nodeId]->oindex = nodeId;
			//mVertexList[nodeId]->g = NULL;
		}
		resG->outputVertices.push_back(mVertexList[nodeId]);
		out = out.next_sibling("out");
	}

	vtx = firstScc.child("vtx");
	while (vtx) {
		unsigned int nodeId = atoi(vtx.child_value());
		nodeId--;
//		for (int i = mVertexList.size(); i <= nodeId; i++) {
//			mVertexList.push_back(NULL);
//		}
		if (mVertexList[nodeId] != NULL) {
			//mVertexList[nodeId] = new Vertex<PTYPE>();
			mVertexList[nodeId]->oindex = nodeId;
			mVertexList[nodeId]->g = resG;
		}
		resG->allVertices.push_back(mVertexList[nodeId]);
		vtx = vtx.next_sibling("vtx");
	}

	inp = firstScc.child("inp");

	while (inp) {
		unsigned int nodeId = atoi(inp.child_value());
		nodeId--;
		if (mVertexList[nodeId] != NULL) {
			mVertexList[nodeId]->isInputVertex = true;
			resG->inputVertices.push_back(mVertexList[nodeId]);
			mVertexList[nodeId]->g = resG;
		}
		inp = inp.next_sibling("inp");
		resG->allVertices.push_back(mVertexList[nodeId]);
	}

	pugi::xml_node childScc = firstScc.child("scc");

	//the most inner scc, i.e which have no child scc are the conretized ones
	if (!childScc) {
		resG->concretized = true;
	}

	resG->state = CONCRETE;
	resG->concretized = true;

	//check if the scc is abstract, i.e. has at least one outgoing abstract edge
	for (vtx_it it = resG->allVertices.begin(); it != resG->allVertices.end(); it++) {
		for (edge_it eit = (*it)->outEdges.begin(); eit != (*it)->outEdges.end(); eit++) {
			if ((*eit)->state == true) {
				resG->state = ABSTRACT;
				resG->concretized = false;
			}
		}
	}

	for (vtx_it it = resG->inputVertices.begin(); it != resG->inputVertices.end(); it++) {
		for (edge_it eit = (*it)->outEdges.begin(); eit != (*it)->outEdges.end(); eit++) {
			if ((*eit)->state == true) {
				resG->state = ABSTRACT;
				resG->concretized = false;
			}
		}
	}

	while (childScc) {
		//pugi::xml_attribute sccId = childScc.attribute("id");
		//Graph<PTYPE> *temp = resG->addSubgraph(sccId.as_int());
		Graph<PTYPE> *temp = resG->addSubgraph();
		readSCCInXML(temp, childScc);
		childScc = childScc.next_sibling("scc");
	}

	if (auxTargetGraph != NULL) {
		return auxTargetGraph;
	}

	AbstractOutputFormatter<PTYPE> *formatter;
	formatter = new XMLOutputFormatter<PTYPE>(mInitialNode, mTargetNodes, true, false);
	LOG4CPLUS_INFO(iologger, formatter->generateOutput(*resG));

	return resG;

}

bool IOWrapper::readFiles(std::istream *strm) {
	//Either we are provided with only a single DTMC file,
	//(this is how SCC_MC was invoked originally and for compatibility and
	// convenience we keep this possibility)
	//or an XML file with the location of DTMC, AP etc. is given

	//if(!Configuration::getInstance().readFromXml()) {

	string dtmcFile;

	ConfigurationSingleton *conf = ConfigurationSingleton::getInstance();

	if (!conf->isIoflDtmcOnly()) {
		LOG4CPLUS_INFO(iologger, "Expecting multi file input...");

		bool mustReadPropFile = false;
		bool isPropFileSet = false;
		bool isDtmcFileSet = false;
		bool isResultFileSet = false;
		bool isXmlFileSet = false;
		string propFile;
		string xmlFile;

		//By default, we assume that no counter example is desired
		//mGenerateCEx = false;

		//The input is the configuration file
		string str;
		while (strm->good()) {
			(*strm) >> str;

			if(str.at(0) == '#') {
				continue;
			}

			std::transform(str.begin(), str.end(), str.begin(), ::toupper);

			if (str.compare("TASK") == 0) {
				string task;
				(*strm) >> task;
				std::transform(task.begin(), task.end(), task.begin(), ::tolower);
				if (!task.compare("counterexample")) {
					if (ConfigurationSingleton::getInstance()->isIoflOnlyModelChecking()) {
						LOG4CPLUS_WARN( iologger, "Won't search counterexample because it was restricted by command line");
					} else {
						mGenerateCEx = true;
					}
				} else if (!task.compare("RestrictedCounterExample")) {
					mustReadPropFile = true;
					mGenerateCEx = true;
				} else if (!task.compare("OnlyModelChecking")) {
					mGenerateCEx = false;
					ConfigurationSingleton::getInstance()->setIoflOnlyModelChecking(true);
				}
				mGenerateCEx = true;
				//Etc.
			} else if (str.compare("PROBABILITY_BOUND") == 0) {
				//(*strm) >> mProbBound;
				if (!ConfigurationSingleton::getInstance()->isIoflForceProbBound()) {
					DefaultInputParser<PTYPE> probParser;
					string read_prob;
					(*strm) >> read_prob;
					mProbBound = probParser.parseProb(read_prob);
				}
			} else if (str.compare("PROBABILITY_OPERATOR") == 0) {
				(*strm) >> str;
				if (str.compare("<=") == 0)
					mOperator = OPERATOR_LEQ;
				else if (str.compare("<") == 0)
					mOperator = OPERATOR_LESS;
				else
					mOperator = OPERATOR_LEQ; //Default: <=
			} else if (str.compare("DTMC_FILE") == 0 && !isDtmcFileSet) {
				isDtmcFileSet = true;
				(*strm) >> dtmcFile;
				if (dtmcFile.find(".") != 0) {
					dtmcFile = mPath + "/" + dtmcFile;
				}
				LOG4CPLUS_INFO(iologger, "Expecting DTMC in " << dtmcFile);
				if (!doesFileExist(dtmcFile)) {
					LOG4CPLUS_ERROR( iologger, "The DTMC file apparently doesn't exist. Have you provided a file path relative to the .conf file?");
				}
			} else if (str.compare("AP_FILE") == 0 && !isPropFileSet) {
				isPropFileSet = true;
				(*strm) >> propFile;
				if (propFile.find(".") != 0) {
					propFile = mPath + "/" + propFile;
				}
				LOG4CPLUS_INFO(iologger, "Expecting Atomic Propositions in " << propFile);
			} else if (str.compare("RESULT_FILE") == 0 && !isResultFileSet) {
				isResultFileSet = true;
				(*strm) >> mResultFile;
				if (mResultFile.find(".") != 0) {
					mResultFile = mPath + "/" + mResultFile;
				}
			} else if (str.compare("XML_FILE") == 0) {
				ConfigurationSingleton::getInstance()->setIoflReadFromXml(true);
				isXmlFileSet = true;
				(*strm) >> xmlFile;
				if (xmlFile.find(".") != 0) {
					inputFile = mPath + "/" + xmlFile;
				} else {
					inputFile = xmlFile;
				}
				ConfigurationSingleton::getInstance()->setIoflReadFromXml(true);
				return true;
			} else if (str.compare("REPRESENTATION") == 0) {
				//unsigned int flags = Configuration::getInstance().getFlags();
				string representation;
				(*strm) >> representation;
				std::transform(representation.begin(), representation.end(), representation.begin(), ::tolower);
				if (representation == "pathset") {
					if (conf->isIoflForceClosure())
						LOG4CPLUS_WARN( iologger, "I can't use the pathset representation, because closure representation was forced by command line");
					else if (conf->isIoflCexLocal()) {
						LOG4CPLUS_WARN( iologger, "Can't use pathsum criterion for local search, because only path fragments are computed. Will use closures instead.");
					} else {
						conf->setIoflBoundPathsum(true);
					}
				} else if (representation == "subsystem") {
					if (conf->isIoflBoundPathsum())
						LOG4CPLUS_WARN( iologger, "I can't use the closure representation, because pathset representation was forced by command line");
					else {
					}
				}
			} else if (str.compare("SEARCH_ALGORITHM") == 0) {
				//unsigned int flags = Configuration::getInstance().getFlags();
				string searchAlgorithm;
				(*strm) >> searchAlgorithm;
				std::transform(searchAlgorithm.begin(), searchAlgorithm.end(), searchAlgorithm.begin(), ::tolower);
				if (searchAlgorithm == "local") {
					if (conf->isIoflForceGlobal())
						LOG4CPLUS_WARN(iologger, "I can't use the local search, because global search was forced by command line");
					else if (conf->isIoflBoundPathsum()) {
						LOG4CPLUS_WARN(iologger, "Can't use local search when pathset is set, will change to global search");
					} else {
						conf->setIoflCexLocal(true);
						mDoGlobalSearch = false;
					}
				} else if (searchAlgorithm == "global") {
					if (conf->isIoflCexLocal())
						LOG4CPLUS_WARN(iologger, "I can't use the global search, because local search was forced by command line");
					else {
						mDoGlobalSearch = true;
					}
				}
			} else if (str.compare("ABSTRACTION") == 0) {
				//unsigned int flags = Configuration::getInstance().getFlags();
				string abstractionMethod;
				(*strm) >> abstractionMethod;
				std::transform(abstractionMethod.begin(), abstractionMethod.end(), abstractionMethod.begin(), ::tolower);
				if (abstractionMethod == "abstract") {
					//check whether concre was set
					if (conf->isIoflCexConcrete() && !conf->isIoflNoModelChecking())
						LOG4CPLUS_WARN(iologger, "I can't use the abstract search, because concrete search was forced by command line");
					else if (conf->isIoflNoModelChecking()) {
						//LOG4CPLUS_WARN(iologger, "When skipping the modelcheker, you can't use the abstract search. Switch to concrete search");
						mDoAbstractSearch = false;
						conf->setIoflCexConcrete(true);
						conf->setIoflForceAbstract(false);
					} else {
						mDoAbstractSearch = true;
					}
				} else if (abstractionMethod.compare("concrete") == 0) {
					//check whether abstract was forced by command line call
					if (conf->isIoflForceAbstract()) {
						LOG4CPLUS_WARN(iologger, "I can't use the concrete search, because abstract search was forced by command line");
					} else {
						mDoAbstractSearch = false;
						conf->setIoflCexConcrete(true);
					}
				}
			} else if (str.compare("AP_BLACKLIST") == 0) {
				//Do something
			} else if (str.compare("AP_WHITELIST") == 0) {
				//Do something
			}
		}

		if ((mGenerateCEx) && (!ConfigurationSingleton::getInstance()->isIoflOnlyModelChecking())) {
			LOG4CPLUS_INFO( iologger, "Task: Compute counter example for probability bound " << mProbBound);
		}

		if (!isDtmcFileSet) {
			LOG4CPLUS_ERROR(iologger, "Configuration file does not contain name of DTMC file");
			delete strm;
			return false;
		}

		if (mustReadPropFile && !isPropFileSet) {
			LOG4CPLUS_WARN(iologger, "Configuration file does not contain name of AP file");
		}

		//If path to prop file was given, parse prop file
		//It is a good idea to do this *before* we read the DTMC file, because
		//we can then assign the APs to the vertices at the time they are
		//discovered
		if (isPropFileSet) {
			delete strm;
			strm = new ifstream(propFile.c_str(), ifstream::in);
			bool couldReadAPs = parsePropositions(strm);
			if (couldReadAPs) {

			} else {
				if (!mustReadPropFile)
					LOG4CPLUS_WARN( iologger, "Could *not* read atomic propositions from given file!");
				else
					LOG4CPLUS_ERROR( iologger, "Could *not* read atomic propositions from given file!");
			}
		}

		//Now we can finally read the DTMC file
		delete strm;
		strm = new ifstream(dtmcFile.c_str(), ifstream::in);
	} else {
		//mGenerateCEx = false;
	}

	//read the input
	LOG4CPLUS_TRACE(iologger, "will now read DTMC from input...");
	bool ok = true;
	if (dtmcFile.find(".xml") != string::npos) {
		LOG4CPLUS_TRACE(iologger, "will parse xml file ...");
		getGraphFromXml(*strm);
	} else {
		LOG4CPLUS_TRACE(iologger, "will parse tra file ...");
		ok = readDTMC(strm);
	}
	if (ok) {
		LOG4CPLUS_TRACE(iologger, "Finished reading the DTMC");
		return true;
	} else {
		LOG4CPLUS_ERROR(iologger, "Could *not* read the DTMC");
		return false;
	}
}

string IOWrapper::mcResultToString(Graph<PTYPE>* g) const {
	//Choose output formatter depending on destination parameter:
	AbstractOutputFormatter<PTYPE> *formatter;
	if (ConfigurationSingleton::getInstance()->isIoflExternalCaller()) {
		//formatter = new DefaultOutputFormatter<PTYPE>(mInitialNode);
		formatter = new XMLOutputFormatter<PTYPE>(mInitialNode, mTargetNodes, true, false);
	} else {
		formatter = new XMLOutputFormatter<PTYPE>(mInitialNode, mTargetNodes, true, false);
	}
	return formatter->generateOutput(*g);
}

std::string IOWrapper::writeMCResult(Graph<PTYPE>* g) const {
	//unsigned int flags = Configuration::getInstance().getFlags();
	string output = "";
	AbstractOutputFormatter<PTYPE> *formatter;
	string filepath = mResultFile;
	if (mResultFileName != "") {
		filepath = mResultFileName;
	}
	string name = filepath.substr(0, filepath.find_first_of("."));
	//if(flags & IOFL_OUTPUT_AS_DTMC) {
	formatter = new DefaultOutputFormatter<PTYPE>(mInitialNode, false);
	output = formatter->generateOutput(*g);
	name = name + ".dtmc";
	//} else if (flags & IOFL_OUTPUT_AS_XML) {
	if (ConfigurationSingleton::getInstance()->isIoflOutputAsXml()) {
		name = filepath.substr(0, filepath.find_first_of("."));
		formatter = new XMLOutputFormatter<PTYPE>(mInitialNode, mTargetNodes, true, ConfigurationSingleton::getInstance()->isIoflCexConcrete());
		output = formatter->generateOutput(*g);
		name = name + ".xml";
	}

	if (mResultFile.compare(JAVA) == 0) {
		//No output currently
		/*fstream outputFile("graph.tra", fstream::out);
		 outputFile << output;
		 outputFile.close();

		 LOG4CPLUS_INFO(iologger, "Prepared TRA data");*/
	} else if (mResultFile.compare("LOGGING") == 0) {
		LOG4CPLUS_INFO(iologger, output);
	} else {
		//Ordinary command line invocation -> write result file
		//fstream outputFile(name.c_str(), fstream::out);
		//outputFile << output;
		//outputFile.close();
		//LOG4CPLUS_INFO(iologger, "Result can be found in: " << name);
	}

	return name;
}

void IOWrapper::writeCounterExample(const CexGenerator& cexGenerator) const {
	string cex = cexGenerator.counterExampleToString();

	LOG4CPLUS_DEBUG(iologger, cex);

	//Write result file

	string strPath = "results";
	if (stat(strPath.c_str(), 0) != 0) {
#ifdef _WIN32 || _WIN64
		//_mkdir(strPath.c_str());
#else
		mkdir(strPath.c_str(), 777);
#endif
	}

	//Generate unique file name
	const time_t stamp = time(NULL);
	stringstream fileName;
	fileName << strPath << "/counter_example_" << stamp << ".txt";

	//Try to write the file
	fstream outputFile(fileName.str().c_str(), fstream::out);
	if (outputFile.fail())
		LOG4CPLUS_ERROR(iologger, "Couldn't open file: " << fileName.str());
	else {
		outputFile << cex;
		LOG4CPLUS_INFO(iologger, "Counter example written to: " << fileName.str());
	}
	outputFile.close();
}

bool IOWrapper::parsePropositions(std::istream *strm) {
	if (strm->fail())
		return false;

	LOG4CPLUS_TRACE(iologger, "Will now parse atomic propositions...");

	string str;
	int vtx = -1;
	string propList;

	bool inHeader = true;
	//We assume that each line has correct structure,
	//so this loop is not exactly robust!
	while (inHeader && strm->good()) {
		//Read "header", i.e. list of propsitions
		(*strm) >> str;

		if (str.compare("AP") == 0) {
			(*strm) >> str;
			//We just insert the propositions into our prop list in the sequence
			//they are in the file. This ordering is then used later to map the
			//vertex/AP maps to bit vectors
			mAtPropList.push_back(str);
		} else {
			inHeader = false; //Break out of loop after this iteration is completed

			//Process 1st vertex/AP data
			vtx = atoi(str.c_str());
			(*strm) >> propList;

			AtPropType props = propStringToAP(propList);
			LOG4CPLUS_TRACE(iologger, "Props for " << vtx << ": " << printProps(props));
			mVtxToPropsMap.insert(pair<int, AtPropType>(vtx, props));
		}
	}
	//Read vertex / atomic proposition maps
	while (strm->good()) {
		(*strm) >> str;
		vtx = atoi(str.c_str());
		(*strm) >> propList;
		//TODO: More intelligent map insert, assuming that the input is ordered by vertex id
		AtPropType props = propStringToAP(propList);
		LOG4CPLUS_TRACE(iologger, "Props for " << vtx << ": " << printProps(props));
		mVtxToPropsMap.insert(pair<int, AtPropType>(vtx, props));
	}

	return true;
}

string IOWrapper::printProps(AtPropType props) const {
	stringstream ss;
	for (unsigned int i = 0; i < props.size(); i++) {
		if (props[i])
			ss << mAtPropList[i] << ",";
	}
	return ss.str();
}

AtPropType IOWrapper::propStringToAP(std::string str) {
	AtPropType props;
	for (unsigned int i = 0; i < mAtPropList.size(); i++)
		props.push_back(0);

	stringstream in(str);

	string token;
	//Split rest of line into tokens...
	while (getline(in, token, ',')) {
		//...and set the corresponding bit vector entries to 1
		//(We use the same sequence as the AP-vector for indexing)
		for (unsigned int id = 0; id < mAtPropList.size(); id++) {
			if (mAtPropList[id].compare(token) == 0) {
				props[id] = 1;
				break;
			}
		}
	}

	return props;
}

bool IOWrapper::readDTMC(istream *strm) {
	if (strm->fail())
		return false;

	int source = -1;
	int target = -1;
	string read_prob = "-1.0";
	PTYPE prob = -1.0;

	string str;
	mInitialNodeID = -1;
	int numStates = -1;
	int numTransitions = -1;

	unsigned int numTargets = 0;

	vector<Vertex<PTYPE>*> selfLoopVertices;

	DefaultInputParser<PTYPE> probParser;

	//While loop for extracting header info (number of states etc.)
	//We assume that each line has correct structure,
	//so this loop is not exactly robust!
	while (strm->good()) {

		(*strm) >> str;

		if (str.compare("STATES") == 0) {

			if (numStates != -1) {
				LOG4CPLUS_ERROR(iologger, "Multiply defined number of states");
				goto Error;
			}
			(*strm) >> numStates;
			LOG4CPLUS_TRACE(iologger, str << " " << numStates);
			if (numStates <= 0) {
				LOG4CPLUS_ERROR(iologger, "Number of states must be larger than 0");
				goto Error;
			}
			for (int i = mVertexList.size(); i <= numStates; i++) {
				mVertexList.push_back(NULL);
			}

		} else if (str.compare("INITIAL") == 0) {

			if (mInitialNodeID != -1) {
				LOG4CPLUS_TRACE(iologger, "Multiply defined initial node");
				goto Error;
			}
			(*strm) >> mInitialNodeID;
			LOG4CPLUS_TRACE(iologger, str << " " << mInitialNodeID);
			if (mInitialNodeID < 0) {
				LOG4CPLUS_ERROR(iologger, "Node ids must be non-negative");
				goto Error;
			}
			for (unsigned int i = mVertexList.size(); i <= (unsigned int) mInitialNodeID; i++) {
				mVertexList.push_back(NULL);
			}
			if (mVertexList[mInitialNodeID] == NULL) {
				mVertexList[mInitialNodeID] = new Vertex<PTYPE>();
				mVertexList[mInitialNodeID]->oindex = mInitialNodeID;

				map<int, AtPropType>::iterator it = mVtxToPropsMap.find(mInitialNodeID);
				if (it != mVtxToPropsMap.end())
					mVertexList[mInitialNodeID]->ap = it->second;
			} else {
				assert(mVertexList[mInitialNodeID]->oindex == (unsigned int)mInitialNodeID);
				LOG4CPLUS_ERROR(iologger, "Probability : 1");
				exit(0);
			}

		} else if (str.compare("TARGET") == 0) {
			numTargets++;

			unsigned int id;
			(*strm) >> id;

			LOG4CPLUS_TRACE(iologger, str << " " << id);
			if (id < 0) {
				LOG4CPLUS_ERROR(iologger, "Node ids must be non-negative");
				goto Error;
			}
			for (unsigned int i = mVertexList.size(); i <= id; i++) {
				mVertexList.push_back(NULL);
			}
			if (mVertexList[id] == NULL) {
				mVertexList[id] = new Vertex<PTYPE>();
				mVertexList[id]->oindex = id;
				map<int, AtPropType>::iterator it = mVtxToPropsMap.find(id);
				if (it != mVtxToPropsMap.end())
					mVertexList[id]->ap = it->second;
			}
			for (unsigned int i = mTargetNodes.size(); i <= id; i++)
				mTargetNodes.push_back(false);
			mTargetNodes[id] = true;

			mTargetIndex = id;

		} else if (str.compare("TRANSITIONS") == 0) {

			if (numTransitions != -1) {
				LOG4CPLUS_ERROR(iologger, "Multiply defined number of transitions");
				goto Error;
			}
			(*strm) >> numTransitions;
			LOG4CPLUS_TRACE(iologger, str << " " << numTransitions);

			if (numTransitions < 0) {
				LOG4CPLUS_ERROR(iologger, "Number of transitions must be non-negative");
				goto Error;
			}

		}
		//No more Header info --> next comes a transition
		else {

			source = atoi(str.c_str());
			(*strm) >> target >> read_prob;
			prob = probParser.parseProb(read_prob);
			LOG4CPLUS_TRACE(iologger, source << " " << target << " " << prob);

			if (!mTargetNodes[source]) {
				if (mVertexList[source] == NULL) {
					mVertexList[source] = new Vertex<PTYPE>();
					mVertexList[source]->oindex = source;
					map<int, AtPropType>::iterator it = mVtxToPropsMap.find(source);
					if (it != mVtxToPropsMap.end())
						mVertexList[source]->ap = it->second;
				}
				if (mVertexList[target] == NULL) {
					mVertexList[target] = new Vertex<PTYPE>();
					mVertexList[target]->oindex = target;
					map<int, AtPropType>::iterator it = mVtxToPropsMap.find(target);
					if (it != mVtxToPropsMap.end())
						mVertexList[target]->ap = it->second;
				}
				if (source == target)
					selfLoopVertices.push_back(mVertexList[source]);
				else {
					assert(prob > 0);
					new Edge<PTYPE>(mVertexList[source], mVertexList[target], prob);
				}
			}

			//We break out of the loop here, i.e. only the header info and the first
			//transition is read in this while loop
			break;
		}
	}

	if (!strm->good())
		return false;

	//Read all (remaining) transitions
	//We assume that each line has correct structure,
	//so this loop is not exactly robust!
	while (strm->good()) {

		source = -1;
		target = -1;
		read_prob = "-1.0";
		//prob = -1.0;

		(*strm) >> source;
		if (!strm->good())
			break;
		(*strm) >> target;
		if (!strm->good())
			break;
		(*strm) >> read_prob;
		prob = probParser.parseProb(read_prob);

		LOG4CPLUS_TRACE(iologger, source << " " << target << " " << prob);

		if (mTargetNodes.size() <= (unsigned int) source || !mTargetNodes[source]) {
			if (mVertexList[source] == NULL) {
				mVertexList[source] = new Vertex<PTYPE>();
				mVertexList[source]->oindex = source;
			}
			if (mVertexList[target] == NULL) {
				mVertexList[target] = new Vertex<PTYPE>();
				mVertexList[target]->oindex = target;
			}
			if (source == target)
				selfLoopVertices.push_back(mVertexList[source]);
			else {
				assert(prob > 0);
				new Edge<PTYPE>(mVertexList[source], mVertexList[target], prob);
			}
		}
	}

	assert(mInitialNodeID >= 0);

	assert(mVertexList[mInitialNodeID] != NULL);
	assert(mVertexList[mInitialNodeID]->oindex == (unsigned int)mInitialNodeID);

	LOG4CPLUS_TRACE(iologger, "Initial node is " << mInitialNodeID);

	//We don't want edges leading into the initial state, so if that's the case,
	//we add an auxiliary initial node
	if (mVertexList[mInitialNodeID]->inEdges.size() > 0) {
		mInitialNode = new Vertex<PTYPE>();
		mInitialNode->oindex = mVertexList.size();
		mVertexList.push_back(mInitialNode);
		new Edge<PTYPE>(mInitialNode, mVertexList[mInitialNodeID], 1.0);
		mInitialNodeID = mInitialNode->oindex;

		LOG4CPLUS_DEBUG(iologger, "Added auxiliary initial node with id " << mInitialNodeID);
	} else {
		if (mInitialNode != NULL) {
			mInitialNode = NULL;
		}
	}

	SccMc::scaleProbabilities(selfLoopVertices);

	if (debuggingMode >= TRACING) {
		stringstream ss;
		ss << "The following nodes are target nodes: ";
		for (unsigned int i = 0; i <= mTargetNodes.size(); i++)
			if (mTargetNodes[i])
				ss << i << ", ";

		LOG4CPLUS_DEBUG(iologger, ss.str());
	}

	//If we are in counter example mode, or we might later be there (external caller!),
	//we only want a single target node...
	if (((mGenerateCEx || ConfigurationSingleton::getInstance()->isIoflExternalCaller()) && numTargets > 1) || (ConfigurationSingleton::getInstance()->isIoflOnlyModelChecking())) {
		LOG4CPLUS_DEBUG( iologger, "For counter example mode we need a single target, will introduce auxiliary target.");

		//...therefore create new target node...
		Vertex<PTYPE>* auxTargetNode = new Vertex<PTYPE>();
		auxTargetNode->oindex = mVertexList.size();
		mVertexList.push_back(auxTargetNode);
		mTargetIndex = auxTargetNode->oindex;

		//...and edges to that target node
		for (unsigned int i = 0; i <= mTargetNodes.size(); i++) {
			if (mTargetNodes[i]) {
				new Edge<PTYPE>(mVertexList[i], auxTargetNode, 1.0);

				//now this is not a target state any more
				mTargetNodes[i] = false;
			}
		}

		//Finally mark new node as target node
		for (unsigned int i = mTargetNodes.size(); i <= auxTargetNode->oindex; i++)
			mTargetNodes.push_back(false);
		mTargetNodes[auxTargetNode->oindex] = true;

		LOG4CPLUS_DEBUG( iologger, "Added auxiliary target node with id " << auxTargetNode->oindex);
	}

	return true;

	Error:

	return false;
}

}
