/**
 *   COMICS - Computing Minimal Counterexamples for Discrete-time Markov Chains
 *
 *   COMICS is a stand-alone tool which performs model checking and the generation
 *   of counterexamples for discrete-time Markov Chains (DTMCs). *
 *
 *   Copyright (C) <2012> <RWTH Aachen University>
 *   Authors: Nils Jansen, Erika Abraham, Jens Katelaan, Maik Scheffler, Matthias Volk, Andreas Vorpahl
 *
 *   This program is free software: you can redistribute it and/or modify
 *   it under the terms of the GNU General Public License as published by
 *   the Free Software Foundation, either version 3 of the License, or
 *   (at your option) any later version.
 *
 *   This program is distributed in the hope that it will be useful,
 *   but WITHOUT ANY WARRANTY; without even the implied warranty of
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *   GNU General Public License for more details.
 *
 *   You should have received a copy of the GNU General Public License
 *   along with this program.  If not, see <http://www.gnu.org/licenses/>.
 *
 *   Main Contact:
 *
 *   Nils Jansen
 *   Theory of Hybrid Systems
 *   RWTH Aachen
 *   52056 Aachen
 *   Germany
 *   nils.jansen@cs.rwth-aachen.de
 *
*/

/*
 * IOWrapper.h
 *
 *  Created on: 01.12.2010
 *      Author: jens
 */

#ifndef IOCONTROLLER_H_
#define IOCONTROLLER_H_

#include "../defines.h"
#include "../GraphTypes.h"
#include "Configuration.h"
#include "./pugixml/pugixml.hpp"
#include "../mc/SCC_MC.h"
#include "../cex/gens/CexGenerator.h"
#include "../cex/CounterExampleParams.h"
#include <vector>
#include <map>
#include <string>

namespace scc_cex {


class IOWrapper {
public:
	IOWrapper(std::string str);
	IOWrapper(std::string str, unsigned int input_flags);
	IOWrapper(std::string str, unsigned int input_flags, unsigned int heuristic_flags);
	virtual ~IOWrapper();

	void run();

	void readSCCInXML(Graph<PTYPE>* res, pugi::xml_node node);
	Graph<PTYPE>* getGraphFromXml(std::istream& stream);
	Graph<PTYPE>* getGraphFromXml(std::string xmlString);

	bool readFiles();
	Graph<PTYPE>* runModelChecking();
	std::string mcResultToString(Graph<PTYPE>* g) const;
	PTYPE getResultProbability() const;
	std::pair<CexGenerator*,CounterExampleParams*> initCexSearch(Graph<PTYPE>* g, bool abstractFlag, bool globalFlag, bool closureFlag);
	void executeCexSteps(CexGenerator* cexGenerator, CounterExampleParams* params, unsigned int steps);

	void setProbBound(PTYPE bound) {
		mProbBound = bound;
	}

	void setGenerateCex(bool val) {
		mGenerateCEx = val;
	}

	void setProbModelCheck(PTYPE probability) {
		mProbModelCheck = probability;
	}

	void setIterationCount(const unsigned int numberOfIterations) {
		mIterationCount = numberOfIterations;
	}

	void setStepSize(const unsigned int stepSize) {
		mStepSize = stepSize;
	}

	std::string mResultFileName;

private:
	std::istream* mStrm;
    std::string mResultFile;
    std::string inputFile; //stores the path given to the application by the first command line argument

    /** Input flags */
    bool mHeuristicsBenchmark;
    bool mSelectionBenchmark;
    bool mSearchBenchmark;
    bool mConcBenchmark;
    bool mDoAbstractSearch;
    bool mDoGlobalSearch;
    bool mAtPropsGiven;
    bool mWaitForExternalCalls;

    //Input flags for filter heuristic
    bool mChooseByProbability;
    bool mChooseByDegree;
    bool mChooseByMembership;

    unsigned int mStepSize;
    unsigned int mIterationCount;

    std::map<int,AtPropType> mVtxToPropsMap;
    std::vector<std::string> mAtPropList;
    std::vector<Vertex<PTYPE> *> mVertexList;
    Graph<PTYPE>* xmlGraph;

    int mInitialNodeID;
    Vertex<PTYPE> *mInitialNode;
    std::vector<bool> mTargetNodes;
    unsigned int mTargetIndex;

    bool mGenerateCEx;
    PTYPE mProbModelCheck;
    PTYPE mProbBound;
    OperatorType mOperator;
    std::string mPath;

    SccMc* mModelChecker;

	void init(std::istream *strm);
	void initLogging();

	bool readFiles(std::istream *strm);

	bool parsePropositions(std::istream *strm);
	AtPropType propStringToAP(std::string);
	std::string printProps(AtPropType props) const;

	std::vector<CounterExampleParams*> generateTaskSequence() const;

	bool readDTMC(std::istream *strm);

	std::string writeMCResult(Graph<PTYPE>* g) const;
	void writeCounterExample(const CexGenerator& cexGenerator) const;

	bool probabilityIsRefuted() const;
};

}

#endif /* IOCONTROLLER_H_ */
