/**
 *   COMICS - Computing Minimal Counterexamples for Discrete-time Markov Chains
 *
 *   COMICS is a stand-alone tool which performs model checking and the generation
 *   of counterexamples for discrete-time Markov Chains (DTMCs). *
 *
 *   Copyright (C) <2012> <RWTH Aachen University>
 *   Authors: Nils Jansen, Erika Abraham, Jens Katelaan, Maik Scheffler, Matthias Volk, Andreas Vorpahl
 *
 *   This program is free software: you can redistribute it and/or modify
 *   it under the terms of the GNU General Public License as published by
 *   the Free Software Foundation, either version 3 of the License, or
 *   (at your option) any later version.
 *
 *   This program is distributed in the hope that it will be useful,
 *   but WITHOUT ANY WARRANTY; without even the implied warranty of
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *   GNU General Public License for more details.
 *
 *   You should have received a copy of the GNU General Public License
 *   along with this program.  If not, see <http://www.gnu.org/licenses/>.
 *
 *   Main Contact:
 *
 *   Nils Jansen
 *   Theory of Hybrid Systems
 *   RWTH Aachen
 *   52056 Aachen
 *   Germany
 *   nils.jansen@cs.rwth-aachen.de
 *
*/

/*
 * InputParsers.h
 *
 *  Created on: 03.11.2010
 *      Author: jens
 */

#ifndef INPUTPARSERS_H_
#define INPUTPARSERS_H_

#include "../defines.h"
#include <string>
#ifdef USE_EXACT
#include <gmpxx.h>
#endif
#include <sstream>

namespace scc_cex {

template <class T>
class AbstractInputParser {
public:
	virtual T parseProb(const std::string readValue) const {
		return 0;
	}

	virtual ~AbstractInputParser() {

	}
};

/*
 * "Default Default" Parser, relying solely on the parsing mechanism
 * of the >> operator. This suffices for the non-parametric-double case
 */
template <class T>
class DefaultInputParser: public AbstractInputParser<T> {
public:
	T parseProb(const std::string readValue) const {
		T ret;
		std::istringstream stm;
		stm.str(readValue);
		stm >> ret;
		return ret;
	}
};

/*
 * Default implementation for non-parametric-exact-arithmetic probabilities
 */
#ifdef USE_EXACT
template <>
class DefaultInputParser <mpq_class>: public AbstractInputParser<mpq_class> {
public:
	mpq_class parseProb(const std::string readValue) const {
		mpq_class prob;
		if (readValue[0] == '1')
			prob = 1;
		else {
			//Prob is of form 0.x -> convert to fraction so that it can be parsed
			std::string nom = readValue.substr(2);
			std::string denom = "1";
			denom.append(nom.length(), '0');
			while (nom[0] == '0') nom = nom.substr(1);
			prob = nom + '/' + denom;

			//common factors are not cancelled automatically!
			prob.canonicalize();
		}
		return prob;
	}
};
#endif

template <class T>
class ParametricInputParser: public AbstractInputParser<T> {
public:
	T parseProb(const std::string readValue) const {
		return 0;
	}
};

}

#endif /* INPUTPARSERS_H_ */
