/**
 *   COMICS - Computing Minimal Counterexamples for Discrete-time Markov Chains
 *
 *   COMICS is a stand-alone tool which performs model checking and the generation
 *   of counterexamples for discrete-time Markov Chains (DTMCs). *
 *
 *   Copyright (C) <2012> <RWTH Aachen University>
 *   Authors: Nils Jansen, Erika Abraham, Jens Katelaan, Maik Scheffler, Matthias Volk, Andreas Vorpahl
 *
 *   This program is free software: you can redistribute it and/or modify
 *   it under the terms of the GNU General Public License as published by
 *   the Free Software Foundation, either version 3 of the License, or
 *   (at your option) any later version.
 *
 *   This program is distributed in the hope that it will be useful,
 *   but WITHOUT ANY WARRANTY; without even the implied warranty of
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *   GNU General Public License for more details.
 *
 *   You should have received a copy of the GNU General Public License
 *   along with this program.  If not, see <http://www.gnu.org/licenses/>.
 *
 *   Main Contact:
 *
 *   Nils Jansen
 *   Theory of Hybrid Systems
 *   RWTH Aachen
 *   52056 Aachen
 *   Germany
 *   nils.jansen@cs.rwth-aachen.de
 *
*/


#include "defines.h"
#include "IOWrapper.h"
#include "InputParsers.h"
#include "Configuration.h"

using namespace std;
using scc_cex::DefaultInputParser;

void printHelp(string name) {
	cout << "COMICS - Computing Minimal Subsystems " << endl;
	cout << "Copyright (c) RWTH Aachen University 2012" << endl;
	cout << "Authors: Nils Jansen, Erika Abraham, Jens Katelaan, Maik Scheffler," << endl;
	cout <<	"   Matthias Volk, Andreas Vorpahl" << endl;
	cout << endl;
	cout << "This is COMICS 1.0. Possible ways to call are: " << endl;
	cout << "comics.sh filename.conf [options]" << endl;
	cout << "NOTICE: filepaths have to be relative to the executable!" << endl;
	cout << "  with options:" << endl;
	cout << "--outputdtmc <filename>    (Save result as <filename.dtmc>)" << endl;
	cout << "--outputxml <filename>     (Save result as <filename.xml>)" << endl;
	cout << "--saveIterations           (Save results of all iterations)" << endl;
	cout << "--dtmc                     (DTMC file only, no config file (filename ends with .dtmc) )" << endl;
	cout << "--only_model_checking      (Do not search for a counterexample)" << endl;
	cout << "--no_model_checking        (Skip model checking (enforces concrete search))" << endl;
	//cout << "--xml                      (XML file should be loaded)" << endl;
	cout << "--abstract | --concrete    (Abstract or concrete search algorithm)" << endl;
	cout << "--global   | --local       (Global or local path search)" << endl;
	cout << "--subsystem  | --pathset     (Subsystem- or path-based counter example)" << endl;
	cout << "--iterationcount <count>   (Do <count> many iterations)" << endl;
	cout << "--stepsize <size>          (One iteration has <size> concretization steps)" << endl;
	cout << "--probBound <bound>        (Probability Bound for counterexample search)" << endl;
	cout << endl;
	cout << "Heuristics for the choice of abstract node to concretize:" << endl;
	cout << "--choose_by_probability    (By highest outgoing probability)" << endl;
	cout << "--choose_by_degree         (By lowest input/output degree)" << endl;
	cout << "--choose_by_membership     (By highest relative membership in subsystem)" << endl;
	cout << endl;
	cout << "Benchmarking:" << endl;
	cout << "--search-benchmark         (Try global, local, concrete, abstract)" << endl;
	cout << "--concretization-benchmark (Try concretizing 1, sqrt, all SCCs in each step)" << endl;
	//cout << "--selection-benchmark      (Consider top-most, down-most, all SCCs for concret.)" << endl;
	cout << "--heuristics-benchmark     (Try different concretization heuristics)" << endl;
	cout << "--complete-benchmark       (Combine all of the above benchmarks - slow!)" << endl;
	cout << "Default behavior: --abstract --global --closure" << endl;
	cout << endl;
	cout << "COMICS is distributed under the GPL conditions." << endl;
    cout << "The product comes with ABSOLUTELY NO WARRANTY." << endl;
    cout << "This is a free software, and you are welcome to redistribute it." << endl;
    cout << endl;
}

int main(int argc, char* argv[]) {
	if (argc == 1) {
		printHelp(argv[0]);
		return 0;
	}

	scc_cex::ConfigurationSingleton* conf = scc_cex::ConfigurationSingleton::getInstance();


	string firstElement = argv[1];

	if (firstElement == "--help") {
		printHelp(argv[0]);
		return 0;
	}

	string lastOption = argv[1];

	string fileName = "";
	string pb = "";

	string targetFile = DEFAULT_RESULT_FILE;
	unsigned int iterationcount = 1;
	unsigned int stepsize = UINT_MAX;

	conf->setIoflInputIsFile(true);
	unsigned heuristic_flags;

	for (int i = 1; i < argc; i++) {
		if (string(argv[i]) == "-o") {
			if (i + 1 < argc)
				targetFile = argv[i + 1];
			else
				targetFile = "NoneSet";
		} else if (string(argv[i]) == "--dtmc") {
			conf->setIoflDtmcOnly(true);
			conf->setIoflOnlyModelChecking(true);
		} else if (string(argv[i]) == "--only_model_checking") {
			conf->setIoflOnlyModelChecking(true);
		} else if (string(argv[i]) == "--no_model_checking") {
			conf->setIoflNoModelChecking(true);
		} else if (string(argv[i]) == "--xml") {
			conf->setIoflReadFromXml(true);
		} else if (string(argv[i]) == "--abstract") {
			//Default behavior
			conf->setIoflForceAbstract(true);
		} else if (string(argv[i]) == "--concrete") {
			conf->setIoflCexConcrete(true);
		} else if (string(argv[i]) == "--outputxml") {
			conf->setIoflOutputAsXml(true);
			if(fileName != "") {
				if (i + 1 < argc)
					targetFile = argv[i + 1];
				else
					targetFile = "NoneSet";
			} else {
				string ioInput = "";
				while(ioInput != "o" && ioInput != "i") {
					std::cout << "Is " << argv[i+1] << " your input or output file? Type 'o' for output and 'i' for input: ";
					getline(cin, ioInput);
				}
				if(ioInput == "o") {
					if (i + 1 < argc)
						targetFile = argv[i + 1];
				} else {
					fileName = argv[i+1];
				}
			}
		} else if (string(argv[i]) == "--outputdtmc") {
			conf->setIoflOutputAsDtmc(true);
			if(fileName != "") {
				if (i + 1 < argc)
					targetFile = argv[i + 1];
				else
					targetFile = "NoneSet";
			} else {
				string ioInput = "";
				while(ioInput != "o" && ioInput != "i") {
					std::cout << "Is " << argv[i+1] << " your input or output file? Type 'o' for output and 'i' for input: ";
					getline(cin, ioInput);
				}
				if(ioInput == "o") {
					if (i + 1 < argc)
						targetFile = argv[i + 1];
				} else {
					fileName = argv[i+1];
				}
			}
		} else if (string(argv[i]) == "--global") {
			//Default behavior
			conf->setIoflForceGlobal(true);
		} else if (string(argv[i]) == "--local") {
			conf->setIoflCexLocal(true);
		} else if (string(argv[i]) == "--subsystem") {
			//Default behavior
			conf->setIoflForceClosure(true);
		} else if(string(argv[i]) == "--probBound") {
			conf->setIoflForceProbBound(true);
			pb = argv[i+1];
		} else if (string(argv[i]) == "--saveIterations") {
			conf->setIoflIoflSaveIterations(true);
		} if ((string(argv[i]).substr(0,2) != "--") && (lastOption != "--outputdtmc") && (lastOption != "--outputxml") && (lastOption != "--probBound")) { //|| (string(argv[i]).substr(0,3) == "../") || (string(argv[i]).find_first_of("/") != string::npos)) {
			//primitive filedetector
			fileName = argv[i];
			string fileEnding = fileName.substr(fileName.find_last_of("."), fileName.size());
			if (fileName.find_last_of(".") == (fileName.find_last_of("/")-1))
				fileEnding = "";
			if(fileEnding != "") {
				if(fileEnding == ".dtmc") {
					conf->setIoflDtmcOnly(true);
				} else if (fileEnding == ".xml") {
					conf->setIoflReadFromXml(true);
				}
			}
			if (!scc_cex::doesFileExist(fileName)) {
				cout << "The given file (" << fileName << ") doesn't exist - terminating" << endl;
				return 0;
			}
		} else if (string(argv[i]) == "--pathset") {
			conf->setIoflBoundPathsum(true);
		} else if (string(argv[i]) == "--iterationcount") {
			if (i + 1 < argc) {
				stringstream ss;
				ss << argv[i + 1];
				try {
					ss >> iterationcount;
				} catch (...) {
					;
				}
			}
		} else if (string(argv[i]) == "--stepsize") {
			if (i + 1 < argc) {
				stringstream ss;
				ss << argv[i + 1];
				try {
					ss >> stepsize;
				} catch (...) {
					;
				}
			}
		} else if (string(argv[i]) == "--search-benchmark") {
			conf->setIoflBenchmarkSearch(true);
		} else if (string(argv[i]) == "--selection-benchmark") {
			conf->setIoflBenchmarkSelection(true);
		} else if (string(argv[i]) == "--heuristics-benchmark") {
			conf->setIoflBenchmarkHeuristic(true);
		} else if (string(argv[i]) == "--concretization-benchmark") {
			conf->setIoflBenchmarkConc(true);
		} else if (string(argv[i]) == "--complete-benchmark") {
			conf->setIoflBenchmarkSearch(true);
			conf->setIoflBenchmarkHeuristic(true);
			conf->setIoflBenchmarkConc(true);
		} else if (string(argv[i]) == "--bunny") {
			cout << "\\__/)" << endl;
			cout << "(O.o)" << endl;
			cout << "( }{ )" << endl;
			cout << "Das ist Hase. Kopiere Hase und hilf ihm, die Weltherrschaft zu uebernehmen." << endl;
		} else if (string(argv[i]) == "--help") {
			printHelp(argv[0]);
			return 0;
		}

		else if (string(argv[i]) == "--nils") {
			cout << "Nils Jansen is king.";
			return 0;
		} else if (string(argv[i]) == "--choose_by_probability") {
			cout << "choose probability" << endl;
			conf->setIoflChooseProbability(true);
		} else if (string(argv[i]) == "--choose_by_degree") {
			cout << "choose degree" << endl;
			conf->setIoflChooseDegree(true);
		} else if (string(argv[i]) == "--choose_by_membership") {
			cout << "choose membership" << endl;
			conf->setIoflChooseMembership(true);
		}
		lastOption = argv[i];
	}
	//Warn of and correct impossible configurations
	if ((conf->isIoflCexLocal()) && (conf->isIoflBoundPathsum())) {
		cout << "Can't use pathsum criterion for local search, because only path fragments are computed. Will use closures instead." << endl;
		conf->setIoflBoundPathsum(false);
	}
	if (((conf->isIoflChooseDegree()) && (conf->isIoflChooseProbability())) || ((conf->isIoflChooseMembership()) && (conf->isIoflChooseProbability()))) {
		cout << "Only one selection heuristic is possible, will choose 'select by probability'";
		conf->setIoflChooseDegree(false);
		conf->setIoflChooseMembership(false);
	}

	if (((conf->isIoflChooseDegree()) && (conf->isIoflChooseMembership()))) {
		cout << "Only one selection heuristic is possible, will choose 'select by membership'";
		conf->setIoflChooseDegree(false);
		conf->setIoflChooseProbability(false);
	}

	if (!scc_cex::doesFileExist(fileName)) {
		cout << "The given file (" << fileName << ") doesn't exist - terminating" << endl;
		return 0;
	}

	scc_cex::IOWrapper wrapper(fileName, 0, 0);

	if(targetFile != "") {
		wrapper.mResultFileName = targetFile;
	}


	if (stepsize < UINT_MAX && stepsize > 0)
		wrapper.setStepSize(stepsize);
	if (iterationcount > 1)
		wrapper.setIterationCount(iterationcount);
	if(pb != "") {
		DefaultInputParser<PTYPE> probParser;
		wrapper.setProbBound(probParser.parseProb(pb));
		wrapper.setGenerateCex(true);
	}
	wrapper.run();

	return 0;

}
