/**
 *   COMICS - Computing Minimal Counterexamples for Discrete-time Markov Chains
 *
 *   COMICS is a stand-alone tool which performs model checking and the generation
 *   of counterexamples for discrete-time Markov Chains (DTMCs). *
 *
 *   Copyright (C) <2012> <RWTH Aachen University>
 *   Authors: Nils Jansen, Erika Abraham, Jens Katelaan, Maik Scheffler, Matthias Volk, Andreas Vorpahl
 *
 *   This program is free software: you can redistribute it and/or modify
 *   it under the terms of the GNU General Public License as published by
 *   the Free Software Foundation, either version 3 of the License, or
 *   (at your option) any later version.
 *
 *   This program is distributed in the hope that it will be useful,
 *   but WITHOUT ANY WARRANTY; without even the implied warranty of
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *   GNU General Public License for more details.
 *
 *   You should have received a copy of the GNU General Public License
 *   along with this program.  If not, see <http://www.gnu.org/licenses/>.
 *
 *   Main Contact:
 *
 *   Nils Jansen
 *   Theory of Hybrid Systems
 *   RWTH Aachen
 *   52056 Aachen
 *   Germany
 *   nils.jansen@cs.rwth-aachen.de
 *
*/

/*
 * SCC_MC.cpp
 *
 *  Created on: 03.11.2010
 *      Author: jens
 */

#include <iostream>
#include <fstream>
#include <sstream>
#include <stdlib.h>
#include <queue>

#include "SCC_MC.h"
#include "../io/IOWrapper.h"
#include "../io/Configuration.h"

#ifdef USE_LOGLIB
#include <log4cplus/logger.h>
#endif

using namespace std;
using namespace log4cplus;

namespace scc_cex {

#ifdef DEBUGGING
DebuggingType debuggingMode = NONE;
#else
DebuggingType debuggingMode = NONE;
#endif

//vector<int> usedGraphIds;

bool doesFileExist(std::string filename) {
	FILE* fp = fopen( filename.c_str(), "rb" );
	if (fp != NULL)	{
		fclose(fp);
		return true;
	}
	else {
		return false;
	}
}

Logger mclogger = Logger::getInstance("SccMc");

SccMc::SccMc(bool isParametric) {
	//mGenerateCEx = generateCEx;

	mGlobalGraph = NULL;
	doPreprocessing = true;
	mAuxInputNode = NULL;
	counter = 0;

#ifdef DOUBLE
	if (isParametric) 	mctype = PARAM_DOUBLE;
	else 				mctype = NP_DOUBLE;
#else
	if (isParametric) 	mctype = PARAM_EXACT;
	else 				mctype = NP_EXACT;
#endif

	mclogger.setLogLevel(DEFAULT_LOG_LEVEL);
}

SccMc::~SccMc() {
	LOG4CPLUS_TRACE(mclogger, "Will now get rid of the model checker");
	if (!mGlobalGraph->limitedFree) delete mGlobalGraph;
	mAuxInputNode = NULL;
}

void SccMc::disablePreprocessing(bool doDisable) {
	doPreprocessing = !doDisable;
}

PTYPE SccMc::getTargetResult(Vertex<PTYPE>* target) const {
	unsigned int trgIndex = target->oindex;
	//No. of input nodes should be one, but we won't enforce it...
	if (mGlobalGraph->inputVertices.size() == 1) {
		LOG4CPLUS_TRACE(mclogger, "Graph has one input, as expected (" << (*(mGlobalGraph->inputVertices[0])) << ")");
	}
	else if (mGlobalGraph->inputVertices.size() > 1)
		LOG4CPLUS_TRACE(mclogger, "Graph has too many input vertices... (" << mGlobalGraph->inputVertices.size() << ")");
	else
		LOG4CPLUS_TRACE(mclogger, "Error: Graph doesn't have any input vertices...");

	for (unsigned int i = 0; i < mGlobalGraph->inputVertices.size(); i++) {
		Vertex<PTYPE>* vtx = mGlobalGraph->inputVertices[i];
		for (unsigned int j = 0; j < vtx->outEdges.size(); j++) {
			if (vtx->outEdges[j]->state == ABSTRACT && vtx->outEdges[j]->target->oindex == trgIndex)
				return vtx->outEdges[j]->prob;
		}
	}

	return 0;
}

PTYPE SccMc::getTargetResult(int trgIndex) const {
	for (vtx_it it = mGlobalGraph->outputVertices.begin(); it != mGlobalGraph->outputVertices.end(); it++) {
		if ((*it)->oindex == (unsigned int)trgIndex) return getTargetResult(*it);
	}
	return 0;
}

Graph<PTYPE>* SccMc::modelCheck_common() {
	Graph<PTYPE>& g = *mGlobalGraph;

	const clock_t t_start = clock();

	//recursively find SCCs
	//makes the SCCs available as subgraphs of g,
	//the SCCs in the SCCs as subgraphs of the subgraphs etc.
	assert(g.subgraphs.size()==0);
	assert(g.inputVertices.size()==1);
	strongComponents(&g);

	const clock_t t_scc = clock();

	//abstraction of SCCs
	LOG4CPLUS_TRACE(mclogger, "### starting abstraction... ###");

	for (unsigned int i=0; i<g.subgraphs.size(); i++) {
		abstraction(g.subgraphs[i]);
	}

	LOG4CPLUS_TRACE(mclogger, "### time for the final reduction... ###");
	reduceGraph(&g);

	const clock_t t_end = clock();

	LOG4CPLUS_DEBUG(mclogger, "SCC detection time: " << (float)((float)(t_scc-t_start)/(float)CLOCKS_PER_SEC));
	LOG4CPLUS_DEBUG(mclogger, "SCC elimination time: " << (float)((float)(t_end-t_scc)/(float)CLOCKS_PER_SEC));
	LOG4CPLUS_DEBUG(mclogger, "Total time: " << (float)((float)(t_end-t_start)/(float)CLOCKS_PER_SEC));

	LOG4CPLUS_DEBUG(mclogger,"MODEL CHECKING RESULT:\n" << g << "(" << counter << " SCCs with multiple input nodes handled)");

	if (mAuxInputNode!=NULL){
		if (mAuxInputNode->g !=NULL) {
			assert(mAuxInputNode->g->state==ABSTRACT);
			mAuxInputNode->g->state = CONCRETE;
		} else {
			LOG4CPLUS_WARN(mclogger, "aux. node not null (" << (*mAuxInputNode) << "), but no graph set");
		}
	}

	return &g;
}

Graph<PTYPE>* SccMc::modelCheck(Graph<PTYPE>* graph) {
	mGlobalGraph = graph;
	mGlobalGraph->limitedFree = true;

	return modelCheck_common();
}

Graph<PTYPE>* SccMc::modelCheck(vtx_vector & vertexList, int& initialNode, vector<bool>& targetNodes, map<int,AtPropType> &vtxToPropsMap) {
	/*if (mGlobalGraph!=NULL){
		delete mGlobalGraph;
		mGlobalGraph = NULL;
	}*/

	//initialize the graph if necessary
	//if (mGlobalGraph == NULL) {
	LOG4CPLUS_TRACE(mclogger, "initializing graph...");

	mGlobalGraph = new Graph<PTYPE>();
	Graph<PTYPE>& g = *mGlobalGraph;
	initGraph(vertexList, initialNode, targetNodes, g, vtxToPropsMap);

	LOG4CPLUS_TRACE(mclogger, "initialization finished");

	LOG4CPLUS_TRACE(mclogger, g);

	return modelCheck_common();
}


void SccMc::initGraph(vtx_vector & vertexList, unsigned int initialNode, vector<bool> & targetNodes, Graph<PTYPE> & g, map<int,AtPropType>& vtxToPropsMap) {
	assert(initialNode<vertexList.size());

	//doPreprocessing = false;

	LOG4CPLUS_TRACE(mclogger, vertexList.size() << " vertices given, initial node being " << initialNode);

	//remove all vertices from the graph g
	//that are not reachable from the initial vertex of g
	vector<Vertex<PTYPE>*> tmpList;
	tmpList.push_back(vertexList[initialNode]);
	vertexList[initialNode]->seen = true;
	for (unsigned int i = 0; i < tmpList.size(); i++) {
		for (unsigned int j = 0; j < tmpList[i]->outEdges.size(); j++) {
			Vertex<PTYPE>* v = tmpList[i]->outEdges[j]->target;
			if (!v->seen) {
				tmpList.push_back(v);
				v->seen = true;
			}
		}
	}
	int pos = 0;
	for (unsigned int i = 0; i < vertexList.size(); i++) {
		if (vertexList[i]!=NULL){
			if (!vertexList[i]->seen && i != initialNode && !targetNodes[i])
				delete vertexList[i];
			else {
				vertexList[i]->seen = false;
				vertexList[i]->g = &g;
				vertexList[pos++] = vertexList[i];
			}
		}
	}
	vertexList.resize(pos);


	vector<Vertex<PTYPE>*> toScale;

	//We iterate through all reachable vertices to determine their type and, if
	//feasible, remove vertices
	for (unsigned int i = 0; i < vertexList.size(); i++) {
		Vertex<PTYPE>* v = vertexList[i];

		int od = v->outEdges.size();
		int id = v->inEdges.size();

		LOG4CPLUS_TRACE(mclogger, (*v) << ": id = " << id << ", od = " << od << ": ");

		if (od > 0 || id > 0) {

			if (od == 0) {
				v->g = NULL;
				g.outputVertices.push_back(v);

				LOG4CPLUS_TRACE(mclogger, "found output vertex " << (*v));
			} else if (id == 0) {

				assert(v->oindex == initialNode);
				g.allVertices.push_back(v);
				g.inputVertices.push_back(v);
				v->isInputVertex = true;
				v->isConcretized = false;

				LOG4CPLUS_TRACE(mclogger, "found input vertex " << (*v));
			} else /* vtx has both incoming and outgoing edges */ {

				//eliminate vertices with a single input and
				//a single output edge
				if (od == 1 && id == 1 && doPreprocessing) {
					//       if (false) {

					LOG4CPLUS_DEBUG(mclogger, "eliminating vertex " << (*v));

					Vertex<PTYPE>* s = v->inEdges[0]->source;
					Vertex<PTYPE>* t = v->outEdges[0]->target;
					assert(s!=v);
					assert(v->oindex!=initialNode);
					assert(v->inEdges[0]->prob > 0);
					assert(v->outEdges[0]->prob > 0);
					if (s == t) {
						assert(toScale.size()==0);
						toScale.push_back(s);
						delete v;
						scaleProbabilities(toScale);
					} else {
						Edge<PTYPE>* e = NULL;
						for (unsigned int k=0; k<s->outEdges.size(); k++){
							if (s->outEdges[k]->target == t){
								e = s->outEdges[k];
								break;
							}
						}
						if (e==NULL) {
							new Edge<PTYPE>(s, t, v->inEdges[0]->prob * v->outEdges[0]->prob);
						} else {
							assert(e->prob > 0);
							e->prob += v->inEdges[0]->prob * v->outEdges[0]->prob;
						}
						delete v;
					}
				} else /* multiple incoming/outgoing or no preprocessing */ {

					g.allVertices.push_back(v);

					LOG4CPLUS_TRACE(mclogger, "keeping vertex " << (*v));
				}
			}
		}
		//v has neither incoming nor outgoing transitions and is no special node
		//--> unreachable node that we can safely remove
		//TODO: Labels will also play a role here
		else if (v->oindex != initialNode && !targetNodes[v->oindex] && doPreprocessing) {
			LOG4CPLUS_TRACE(mclogger, "removing vertex " << (*v));
			delete v;
		}
		//last case: no edges to/from v, but special vertex or no preprocessing --> must keep it
		else {
			LOG4CPLUS_TRACE(mclogger, "special vertex " << (*v));

			if (v->oindex == initialNode) {
				g.inputVertices.push_back(v);
				v->isInputVertex = true;
				v->isConcretized = false;
				g.allVertices.push_back(v);
			} else {
				g.outputVertices.push_back(v);
				v->g = NULL;
			}
		}
	}

	vertexList.clear();
}

void SccMc::computeWeight(Vertex<PTYPE> *in, Vertex<PTYPE> *out) const
{

	LOG4CPLUS_TRACE(mclogger, "comp. weight " << (*in) << " -> " << (*out) << ", trg having " << out->inEdges.size() << " in edges");

	out->weight = 0.0;
	for (unsigned int i=0; i<out->inEdges.size(); i++) {
		Vertex<PTYPE>* v = out->inEdges[i]->source;
		if (v->g->state == out->inEdges[i]->state &&
				(v->g->parent == NULL || v->g->parent->state == CONCRETE))
		{
			Graph<PTYPE>* g2 = v->g;
			while (g2!=NULL && g2!= in->g)
				g2 = g2->parent;
			if (g2==in->g){
				if (v->weight== -1.0)
					computeWeight(in,v);
				out->weight += v->weight * out->inEdges[i]->prob;
			}
		}
	}
	LOG4CPLUS_TRACE(mclogger, "result: " << (*out));
}

//TODO: Return to the algorithm that doesn't compute loops
void SccMc::abstraction(Graph<PTYPE>* g)
{
	//JK: Maybe a bit too much DEBUGGING here... Might remove later
	LOG4CPLUS_TRACE(mclogger, "--next level of abstraction--");

	for (unsigned int i=0; i<g->subgraphs.size(); i++)
		abstraction(g->subgraphs[i]);

	//Simple cases: Only one input and/or output vertex
	if (g->inputVertices.size()==1 || g->outputVertices.size()==1){
		//One call to reduce graph suffices; the abstract edges are added there
		reduceGraph(g);
	}
	//More interesting case: Multiple input and output nodes
	else {

		if (debuggingMode >= TRACING) {
			counter++;

			LOG4CPLUS_TRACE(mclogger, "abstract: reducing SCC " << g->graph_id << " with multiple input/multiple output");
			stringstream ss("input nodes are: ");

			for (unsigned int i=0; i<g->inputVertices.size(); i++) {
				ss << g->inputVertices[i]->oindex;
				ss << " ";
			}

			LOG4CPLUS_TRACE(mclogger, ss.str());
		}

		/* collect all vertices of the scc */
		vector<Vertex<PTYPE>*> vertices;

		//First output vertices
		for (unsigned int i=0; i<g->outputVertices.size(); i++){
			Vertex<PTYPE>* v = g->outputVertices[i];
			assert(i==vertices.size());
			v->dfs = i;
			vertices.push_back(v);
			v->seen = true;
		}
		//Then input vertices
		for (unsigned int i=0; i<g->inputVertices.size(); i++){
			Vertex<PTYPE>* v = g->inputVertices[i];
			v->seen = true;
			v->dfs = vertices.size();
			vertices.push_back(v);
		}

		const unsigned int numOV = g->outputVertices.size();
		//Needed for loop computations
		const unsigned int numIV = g->inputVertices.size();

		//Then intermediate vertices in the same SCC
		for (unsigned int i=numOV; i<vertices.size(); i++){
			Vertex<PTYPE>* source = vertices[i];
			for (unsigned int j=0; j<source->outEdges.size(); j++){
				Edge<PTYPE>* e = source->outEdges[j];
				if (e->state == source->g->state){
					Vertex<PTYPE>* target = e->target;
					if (!target->seen){
						target->seen = true;
						target->dfs = vertices.size();
						vertices.push_back(target);
					}
				}
			}
		}
		for (unsigned int i=0; i<vertices.size(); i++)
			vertices[i]->seen = false;

		const unsigned int n2 = vertices.size();
		const unsigned int n1 = n2 - numOV;
		//One row for each non-output node (i.e. all possible edge sources),
		//One column for each node (i.e. all possible target nodes of the scc's edges)
		PTYPE prob[n1][n2];

		/* Assemble initial matrix */
		for (unsigned int j1=0; j1<n1; j1++)
			for (unsigned int j2=0; j2<n2; j2++)
				prob[j1][j2] = 0.0;

		//Fill in the probabilities of all edges
		for (unsigned int j1=0; j1<n1; j1++){
			Vertex<PTYPE>* source = vertices[j1+numOV];
			assert(source->dfs == j1+numOV);
			for (unsigned int j2=0; j2<source->outEdges.size(); j2++){
				Edge<PTYPE>* e = source->outEdges[j2];
				if (e->state == source->g->state){
					prob[j1][e->target->dfs] = e->prob;
				}
			}
		}

		if (debuggingMode >= TRACING) {
			stringstream ss("initial matrix:\n");
			for (unsigned int j1=0; j1<n1; j1++) {
				ss << "p" << vertices[j1+numOV]->oindex << " = ";
				bool first = true;
				for (unsigned int j2=0; j2<n2; j2++) {
					if (prob[j1][j2]!=0.0) {
						if (first) first = false;
						else ss << " + ";
						ss << prob[j1][j2] << " * p" << vertices[j2]->oindex;
					}
				}
			}
			LOG4CPLUS_TRACE(mclogger, ss.str());

			ss.clear();

			for (unsigned int j1=0; j1<n1; j1++) {
				ss << "p" << vertices[j1+numOV]->oindex << " = ";
				for (unsigned int j2=0; j2<n2; j2++) {
					stringstream test;
					test << prob[j1][j2];
					string str = test.str();
					while (str.length() < 6) str = " "+str;
					ss << str << " ";
				}
				ss << "\n";
			}

			LOG4CPLUS_TRACE(mclogger, ss.str());
			LOG4CPLUS_TRACE(mclogger, "now computing final matrix...");
		}

		//We eliminate all intermediate nodes, i.e. nodes that are neither input nor output nodes
		//For each intermediate node...
		for (int currNode = n1-1; currNode >= numIV; currNode--) {
			const int currNodeCol = currNode + numOV;
			//...and for each node with an edge to that intermediate node,
			//   i.e. with a non-zero entry in column col...
			for (int srcNode = 0; srcNode < n1; srcNode++) {
				if (srcNode == currNode) continue; //Can't eliminate self-loops, of course!

				//...get the probability of taking a transition from node #srcNode to #currNode...
				PTYPE fac = prob[srcNode][currNodeCol];
				if (fac > 0.0) {
					//...and for each node newTarget ...
					//  (we only have to look at nodes left from currNode, because all nodes
					//   to the right of it have already been removed)
					PTYPE weight = 1-prob[currNode][currNodeCol];
					for (int newTarget = 0; newTarget < currNodeCol; newTarget++) {
						//...increase the probability of going from #srcNode to #newTarget
						//by the probability prob(srcNode, currNode) * prob (currNode, newTarget) / (1 - prob of staying in currNode)
						if (prob[currNode][newTarget] > 0.0)
							prob[srcNode][newTarget] += fac * prob[currNode][newTarget] / weight;
						prob[srcNode][currNodeCol] = 0;
					}
				}
			}

			//Once we have done this for each node, we won't look at it again
			//TODO: Setting row to all-zero should have no influence on the result. So maybe remove the following
			for (int col = 0; col < currNodeCol; col++)
				prob[currNode][col] = 0;
		}

		if (debuggingMode >= TRACING) {
			LOG4CPLUS_TRACE(mclogger, "matrix after elimination of all intermediate nodes");
			stringstream ss;
			for (unsigned int j1=0; j1<n1; j1++) {
				ss << "p" << vertices[j1+numOV]->oindex << " = ";
				for (unsigned int j2=0; j2<n2; j2++) {
					stringstream test;
					test << prob[j1][j2];
					string str = test.str();
					while (str.length() < 6) str = " "+str;
					ss << str << " ";
				}
				ss << "\n";
			}

			LOG4CPLUS_TRACE(mclogger, ss.str());
		}

		//Now only the input and output nodes are left in the matrix
		//Before we solve that subsystem, we compute for each input node the
		//loop probability, i.e. the prob of returning to that node
		//To do this, we have to solve a numIV*numIV system for each IV
		PTYPE ivMatrix[numIV][numIV];
		bool havePrinted = false;
		for (unsigned int currIV = 0; currIV < numIV; currIV++) {
			//Init matrix with the values from prob matrix
			for (unsigned int i = 0; i < numIV; i++) {
				for (unsigned int j = 0; j < numIV; j++) {
					ivMatrix[i][j] = prob[i][j + numOV];
				}
			}

			//If in tracing mode, print subsystem once
			if (debuggingMode >= TRACING && !havePrinted) {
				havePrinted = true;
				LOG4CPLUS_TRACE(mclogger, "initial matrix for the input vertex system:");
				stringstream ss;
				for (unsigned int i=0; i<numIV; i++) {
					ss << "p" << vertices[i+numOV]->oindex << " = ";
					for (unsigned int j=0; j<numIV; j++) {
						ss << ivMatrix[i][j] << " * p" << vertices[j+numOV]->oindex << " ";
					}
				}

				LOG4CPLUS_TRACE(mclogger, ss.str());
			}

			//Eliminate all nodes != iv using the same algorithm as above
			//For each node...
			for (int currNode = numIV-1; currNode >= 0; currNode--) {
				if (currNode == currIV) continue; //Skip the node for which we are solving the system

				//...and for each node with an edge to that node,
				//   i.e. with a non-zero entry in the corresponding column...
				for (unsigned int srcNode = 0; srcNode < numIV; srcNode++) {
					if (srcNode == currNode) continue; //Can't eliminate self-loops, of course!

					//...get the probability of taking a transition from node #srcNode to #currNode...
					PTYPE fac = ivMatrix[srcNode][currNode];
					if (fac > 0.0){
						//...and for each node newTarget ...
						//TODO: We only need to look at the nodes left of currNode and at currIV --> some optimization possible
						PTYPE weight = 1-ivMatrix[currNode][currNode];
						for (unsigned int newTarget = 0; newTarget < numIV; newTarget++){
							//...increase the probability of going from #srcNode to #newTarget
							//by the probability prob(srcNode, currNode) * prob (currNode, newTarget) / (1 - prob of staying in currNode)
							if (ivMatrix[currNode][newTarget] > 0.0)
								ivMatrix[srcNode][newTarget] += fac * ivMatrix[currNode][newTarget] / weight;
							ivMatrix[srcNode][currNode] = 0;
						}
					}
				}

				//Once we have done this for each node, we won't look at it again
				//TODO: Setting row to all-zero should have no influence on the result. So maybe remove the following
				for (int col = 0; col < currNode; col++)
					ivMatrix[currNode][col] = 0;
			}

			//Print final matrix for the current input node if in tracing mode
			if (debuggingMode >= TRACING) {
				LOG4CPLUS_TRACE(mclogger, "final matrix for input vertex " << vertices[currIV+numOV]->oindex << ":");
				stringstream ss;
				for (unsigned int i=0; i<numIV; i++) {
					ss << "p" << vertices[i+numOV]->oindex << " = ";
					for (unsigned int j=0; j<numIV; j++) {
						ss << ivMatrix[i][j] << " * p" << vertices[j+numOV]->oindex << " ";
					}
					ss << endl;
				}
				LOG4CPLUS_TRACE(mclogger, ss.str());
			}
		}

		//We have found all loop probs; now we can return to solving the original system
		//We have to remove the remaining intermediate edges
		//(which are only self-loops and input-input edges)

		//For each input node currNode...
		for (int currNode = numIV-1; currNode >= 0; currNode--){
			const int currNodeCol = currNode + numOV;
			//Eliminate self loop
			if (prob[currNode][currNodeCol] != 0.0) {
				PTYPE weight = 1-prob[currNode][currNodeCol];
				prob[currNode][currNodeCol] = 0.0;
				assert(weight>0.0 && weight<1.0);
				for (unsigned int targetNode=0; targetNode<numOV+numIV; targetNode++)
					if (prob[currNode][targetNode] > 0.0) prob[currNode][targetNode] /= weight;
			}

			//For each input node srcNode in the matrix...
			for (int srcNode = numIV-1; srcNode >= 0; srcNode--) {
				if (srcNode == currNode) continue; //There are no self-loops left

				//...get the probability of taking a transition back to currNode,
				PTYPE fac = prob[srcNode][currNodeCol];
				if (fac>0.0){
					//...and for each newTarget node...
					for (unsigned int newTarget=0; newTarget<numOV+numIV; newTarget++){
						//...if it is possible to go from non-output currNode to node newTarget,
						//the probability of going there from srcNode is
						//Prob(srcNode,currNode) * Prob(currNode,newTarget)
						//Therefore increase prob[srcNode][newTarget] by this product
						if (prob[currNode][newTarget] > 0.0) {
							prob[srcNode][newTarget] += fac * prob[currNode][newTarget];
						}
					}
					//Once we have done this for each output node, we don't need the transition any more
					prob[srcNode][currNodeCol] = 0.0;
				}
			}
		}

		//Computation complete --> print final matrix if in tracing mode
		if (debuggingMode >= TRACING) {
			LOG4CPLUS_TRACE(mclogger, "final matrix:");
			stringstream ss;
			for (unsigned int j1=0; j1<n1; j1++) {
				ss << "p" << vertices[j1+numOV]->oindex << " = ";
				bool first = true;
				for (unsigned int j2=0; j2<n2; j2++) {
					if (prob[j1][j2]!=0.0) {
						if (first) first = false;
						else ss << " + ";
						ss << prob[j1][j2] << " * p" << vertices[j2]->oindex;
					}
				}
				ss << endl;
			}
			LOG4CPLUS_TRACE(mclogger, ss.str());
		}

		if (debuggingMode >= TRACING) {
			for (unsigned int j1=0; j1<n1; j1++) {
				for (unsigned int j2=numOV; j2<n2; j2++)
					assert(prob[j1][j2]==0.0);
				PTYPE sum = 0.0;
				for (unsigned int j2=0; j2<numOV; j2++) {
					sum += prob[j1][j2];
				}

#ifdef USE_DOUBLE
				assert(sum>=0.99 && sum <= 1.01);
#else
				assert(sum == 1.0);
#endif
			}
		}

		for (unsigned int i=0; i<numOV; i++){
			for (unsigned int j=0; j<g->inputVertices.size(); j++){
				PTYPE p = prob[g->inputVertices[j]->dfs - numOV][i];
				if (p>0.0)
					new Edge<PTYPE>(g->inputVertices[j],g->outputVertices[i],p,ABSTRACT);
			}
		}

		g->state = ABSTRACT;
	}

	//JK: Maybe a bit too much DEBUGGING here... Might remove later
	LOG4CPLUS_TRACE(mclogger, "--leaving level of abstraction--");
}

void SccMc::scaleProbabilities(vector<Vertex<PTYPE>*> & selfLoopVertices)
{
	for (unsigned int i = 0; i < selfLoopVertices.size(); i++) {
		PTYPE sum = 0.0;
		for (unsigned int j = 0; j < selfLoopVertices[i]->outEdges.size(); j++) {
			assert(selfLoopVertices[i]->outEdges[j]->state == CONCRETE);
			sum += selfLoopVertices[i]->outEdges[j]->prob;
		}
		for (unsigned int j = 0; j < selfLoopVertices[i]->outEdges.size(); j++) {
			assert(sum>0.0);
			selfLoopVertices[i]->outEdges[j]->prob /= sum;
		}
	}
	selfLoopVertices.clear();
}

//Recursively find SCCs in Graph g, using Tarjan's algorithm
//
void SccMc::strongComponents(Graph<PTYPE>* g) const {
	assert(g->inputVertices.size()>0);

	//Initialize vertex state for Tarjan's algorithm
	for (unsigned int i=0; i<g->allVertices.size(); i++){
		Vertex<PTYPE>* v = g->allVertices[i];
		if (!v->isInputVertex){
			v->dfs = -1;
			v->lowlink = -1;
			v->seen = true;
		}
	}

	vector<Vertex<PTYPE>*> S; //Stack of visited nodes
	int max_dfs = 0; //To keep track of the sequence in which nodes are detected

	//Start Tarjan's algo for all undiscovered nodes
	for (unsigned int i = 0; i < g->inputVertices.size(); i++) {
		Vertex<PTYPE>* s = g->inputVertices[i];
		for (unsigned int j=0; j<s->outEdges.size(); j++){
			Vertex<PTYPE>* v = s->outEdges[j]->target;
			//If seen is false, v has already been discovered in a previous
			//call of tarjan(..)
			if (v->seen) {
				assert(!(v->isInputVertex));
				assert(S.size()==0);
				tarjan(v, g, S, max_dfs);
			}
		}
	}

	if (debuggingMode == COMPLETE) {
		for (unsigned int i=0; i<g->allVertices.size(); i++)
			assert(!g->allVertices[i]->seen);
	}

	//tarjan adds the SCCs as subgraphs to g, so that we can now recursively
	//look for SCCs in the SCCs of g
	for (unsigned int i=0; i<g->subgraphs.size(); i++)
		strongComponents(g->subgraphs[i]);
}

void SccMc::tarjan(Vertex<PTYPE> *v, Graph<PTYPE>* g, vtx_vector & S, int & max_dfs) const
{
	if (!(v->g == g)) {
		LOG4CPLUS_ERROR(mclogger, "Wrong graph pointer set for " << (*v));// << ", corresponding graphs have ids " << g->graph_id << " and" << v->g->graph_id);
	}

	assert(v->g==g);

	v->dfs = max_dfs;
	v->lowlink = max_dfs;
	max_dfs++;
	S.push_back(v);
	v->seen = false;

	//DFS + computing lowlink values
	for (unsigned int i = 0; i < v->outEdges.size(); i++) {
		Vertex<PTYPE>* w = v->outEdges[i]->target;

		if (w->g == g && !w->isInputVertex) {
			if (w->seen) {
				tarjan(w, g, S, max_dfs);
				int low1 = v->lowlink;
				int low2 = w->lowlink;
				v->lowlink = (low1 < low2 ? low1 : low2);
			} else if (w->dfs != -1) {
				int low1 = v->lowlink;
				int dfs2 = w->dfs;
				v->lowlink = (low1 < dfs2 ? low1 : dfs2);
			}
		}
	}

	int low = v->lowlink;
	int dfs = v->dfs;
	//Have we found the root of an SCC?
	if (low == dfs) {
		if (S[S.size()-1] == v){
			assert(v->g == g);
			assert(!v->isInputVertex);
			v->dfs = -1;
			S.pop_back();
		} else {
			//We've found a non-trivial SCC, so we add it as subgraph
			Graph<PTYPE>& g2 = *(g->addSubgraph());
			int idx = S.size()-1;
			while (true) {
				assert(idx>=0);
				Vertex<PTYPE>* w = S[idx--];
				w->g = &g2;
				if (v == w) break;
			}

			//Iterate through all vertices of the SCC again to...
			for (unsigned int i=idx+1; i<S.size(); i++){
				//(1) add them to the subgraphs vertex list
				g2.allVertices.push_back(S[i]);

				//(2) find output vertices (vertices reachable from g2 in one step)
				for (unsigned int j=0; j<S[i]->outEdges.size(); j++){
					Vertex<PTYPE>* target = S[i]->outEdges[j]->target;
					if (target->g != &g2){
						//Make sure we don't add an output vertex twice
						bool found = false;
						for (unsigned int k=0; k<g2.outputVertices.size(); k++){
							if (g2.outputVertices[k] == target){
								found = true;
								break;
							}
						}
						if (!found) {
							LOG4CPLUS_TRACE(mclogger, "Adding the target of " << (* S[i]->outEdges[j]) << " to targets");
							g2.outputVertices.push_back(target);
						}
					}
				}

				//(3) find input vertices
				for (unsigned int j=0; j<S[i]->inEdges.size(); j++){
					assert(!S[i]->isInputVertex);
					if (S[i]->inEdges[j]->source->g != &g2){
						g2.inputVertices.push_back(S[i]);
						S[i]->isInputVertex = true;
						S[i]->isConcretized = false;
						break;
					}
				}
			}

			S.resize(idx+1);
		}
	}
}

void SccMc::reduceGraph(Graph<PTYPE>* g) const
{
	assert(g->inputVertices.size()==1 || g->outputVertices.size()==1);

	if (g->outputVertices.size()==1){
		//Only one output vertex --> the probability to reach the output vertex
		//of the subgraph is 1.0 for all input vertices
		LOG4CPLUS_TRACE(mclogger, "reducegraph " << g->graph_id << ": single input/single output");

		for (unsigned int i=0; i<g->inputVertices.size(); i++){
			Edge<PTYPE>* e = new Edge<PTYPE>(g->inputVertices[i],g->outputVertices[0],1.0);
			e->state = ABSTRACT;
			LOG4CPLUS_TRACE(mclogger, "added abstract edge " << (*e));
		}
		g->state = ABSTRACT;

	} else {
		//Only one input vertex --> we need one abstract edge from the input to
		//each output vertex --> need to compute weights for the abstract edges

		LOG4CPLUS_TRACE(mclogger, "reducegraph " << g->graph_id << ": single input/multiple output");
		LOG4CPLUS_TRACE(mclogger, "computing weights:");
		LOG4CPLUS_TRACE(mclogger, "Graph:" << (*g));

		g->inputVertices[0]->weight = 1.0;
		//Compute weight for each output vertex
		PTYPE sum = 0.0;
		for (unsigned int j=0; j<g->outputVertices.size(); j++){
			LOG4CPLUS_TRACE(mclogger, "Outer loop for " << (*g->outputVertices[j]));
			computeWeight(g->inputVertices[0],g->outputVertices[j]);
			sum += g->outputVertices[j]->weight;
		}
		assert(sum>0.0);

		//Create the abstract edges
		//The actual probability has to be set to weight/sum to guarantee a total
		//probability of 1.0
		for (unsigned int j=0; j<g->outputVertices.size(); j++){
			//JK, Nov03/10: this assertion does not always hold!
			//assert(g.outputVertices[j]->weight > 0.0);
			Edge<PTYPE>* e = new Edge<PTYPE>(g->inputVertices[0],g->outputVertices[j],g->outputVertices[j]->weight/sum);
			e->state = ABSTRACT;
			LOG4CPLUS_TRACE(mclogger, "added abstract edge " << (*e));
		}
		g->state = ABSTRACT;

		//clean up weights
		for (unsigned int i=0; i<g->outputVertices.size(); i++)
			g->outputVertices[i]->weight = -1.0;

		for (unsigned int i=0; i<g->allVertices.size(); i++)
			g->allVertices[i]->weight = -1.0;
	}
}

void SccMc::setAuxInputNode(Vertex<PTYPE> *auxNode) {
	mAuxInputNode = auxNode;
}

}

/* -------------- functions for external callers --------------- */

int callModelCheckerWithFile(const string fileName) {
	//scc_cex::IOWrapper wrapper(fileName, IOFL_INPUT_IS_FILE | IOFL_DTMC_ONLY | IOFL_EXTERNAL_CALLER);
	scc_cex::IOWrapper wrapper(fileName, 8);
	wrapper.run();
	return 0;
}

int callModelCheckerWithString(const string dataString) {
	//scc_cex::IOWrapper wrapper(dataString, IOFL_INPUT_IS_DATA | IOFL_DTMC_ONLY | IOFL_EXTERNAL_CALLER);
	scc_cex::IOWrapper wrapper(dataString, 9);
	wrapper.run();
	return 0;
}
