/**
 *   COMICS - Computing Minimal Counterexamples for Discrete-time Markov Chains
 *
 *   COMICS is a stand-alone tool which performs model checking and the generation
 *   of counterexamples for discrete-time Markov Chains (DTMCs). *
 *
 *   Copyright (C) <2012> <RWTH Aachen University>
 *   Authors: Nils Jansen, Erika Abraham, Jens Katelaan, Maik Scheffler, Matthias Volk, Andreas Vorpahl
 *
 *   This program is free software: you can redistribute it and/or modify
 *   it under the terms of the GNU General Public License as published by
 *   the Free Software Foundation, either version 3 of the License, or
 *   (at your option) any later version.
 *
 *   This program is distributed in the hope that it will be useful,
 *   but WITHOUT ANY WARRANTY; without even the implied warranty of
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *   GNU General Public License for more details.
 *
 *   You should have received a copy of the GNU General Public License
 *   along with this program.  If not, see <http://www.gnu.org/licenses/>.
 *
 *   Main Contact:
 *
 *   Nils Jansen
 *   Theory of Hybrid Systems
 *   RWTH Aachen
 *   52056 Aachen
 *   Germany
 *   nils.jansen@cs.rwth-aachen.de
 *
*/

/*
 * SCC_MC.h
 *
 *  Created on: 03.11.2010
 *      Author: jens
 */

#ifndef SCC_MC_H_
#define SCC_MC_H_

#include <vector>
#include <map>
#include <string>
#include <time.h>

#include "../defines.h"
#include "../typedefs.h"
#include "../GraphTypes.h"

namespace scc_cex {

typedef enum MCType {
	NP_DOUBLE,		//Non-Parametric Markov chains, double precision
	NP_EXACT,		//Non-Parametric Markov chains, exact arithmetic
	PARAM_DOUBLE,	//Parametric Markov chains, double precision
	PARAM_EXACT		//Parametric Markov chains, exact arithmetic
} MCType;

class SccMc {
public:
	SccMc(bool isParametric);
	virtual ~SccMc();

	/* configuration functions */
    void setAuxInputNode(Vertex<PTYPE> *auxNode);
    void disablePreprocessing(bool doDisable);

    /* control functions */
	Graph<PTYPE>* modelCheck(vtx_vector & vertexList, int& initialNode, std::vector<bool>& targetNodes, std::map<int,AtPropType>& vtxToPropsMap);
	Graph<PTYPE>* modelCheck(Graph<PTYPE>* graph);
	PTYPE getTargetResult(Vertex<PTYPE>* target) const;
	PTYPE getTargetResult(int trgIndex) const;

	/* other functions */
	static void scaleProbabilities(std::vector<Vertex<PTYPE>*>& selfLoopVertices);
private:
    MCType mctype;

    Graph<PTYPE> *mGlobalGraph;
    Vertex<PTYPE> *mAuxInputNode;

    bool doPreprocessing;

    unsigned int counter;

    /* model checking functions */
    Graph<PTYPE>* modelCheck_common();
    void initGraph(vtx_vector & vertexList, unsigned int initialNode, std::vector<bool> & targetNodes, Graph<PTYPE> & g, std::map<int,AtPropType>& vtxToPropsMap);
    void abstraction(Graph<PTYPE>* g);
    void strongComponents(Graph<PTYPE>* g) const;
    void tarjan(Vertex<PTYPE> *v, Graph<PTYPE>* g, vtx_vector & S, int & max_dfs) const;
    void reduceGraph(Graph<PTYPE>* g) const;
    void computeWeight(Vertex<PTYPE> *in, Vertex<PTYPE> *out) const;
};

}

//For JNI calls
int callModelCheckerWithFile(const std::string fileName);
int callModelCheckerWithString(const std::string dataString);

#endif /* SCC_MC_H_ */
