/**
 *   COMICS - Computing Minimal Counterexamples for Discrete-time Markov Chains
 *
 *   COMICS is a stand-alone tool which performs model checking and the generation
 *   of counterexamples for discrete-time Markov Chains (DTMCs). *
 *
 *   Copyright (C) <2012> <RWTH Aachen University>
 *   Authors: Nils Jansen, Erika Abraham, Jens Katelaan, Maik Scheffler, Matthias Volk, Andreas Vorpahl
 *
 *   This program is free software: you can redistribute it and/or modify
 *   it under the terms of the GNU General Public License as published by
 *   the Free Software Foundation, either version 3 of the License, or
 *   (at your option) any later version.
 *
 *   This program is distributed in the hope that it will be useful,
 *   but WITHOUT ANY WARRANTY; without even the implied warranty of
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *   GNU General Public License for more details.
 *
 *   You should have received a copy of the GNU General Public License
 *   along with this program.  If not, see <http://www.gnu.org/licenses/>.
 *
 *   Main Contact:
 *
 *   Nils Jansen
 *   Theory of Hybrid Systems
 *   RWTH Aachen
 *   52056 Aachen
 *   Germany
 *   nils.jansen@cs.rwth-aachen.de
 *
 */

package comics;

import java.awt.Desktop;
import java.io.File;
import java.util.ArrayList;
import javax.swing.JOptionPane;

import comics.data.Config;
import comics.data.CounterExampleResult;
import comics.data.CounterExampleSearchParameters;
import comics.data.ModelCheckResult;
import comics.graph.data.Labels;
import comics.graph.data.MarkovChain;
import comics.graph.data.Node;
import comics.gui.GUI;
import comics.gui.ParsingProgress;
import comics.io.ExtensionType;
import comics.io.FileHandler;
import comics.io.input.ConfReader;
import comics.io.input.InputParser;
import comics.io.input.LabReader;
import comics.io.input.PrismReader;
import comics.io.input.DtmcReader;
import comics.io.input.TraReader;
import comics.io.input.XmlParser;
import comics.io.output.DtmcWriter;
import comics.io.output.LabWriter;
import comics.io.output.OutputFormatter;
import comics.io.output.TraWriter;
import comics.io.output.XmlWriter;
import comics.utilities.Output;
import comics.utilities.Pair;

/**
 * @author mazz
 * 
 */
public class SCC_MC {

	private MarkovChain markovChain;
	private GUI gui;
	private FileHandler fileHandler;
	private boolean graphLocked = false;
	private ModelCheckResult modelCheckResult;

	public CounterExampleSearchParameters counterExampleSearchParameters;
	public CounterExampleResult counterExampleResult;

	// Singleton pattern
	private static SCC_MC instance = null;

	/**
	 * Private constructor, because this is a singleton
	 */
	private SCC_MC() {
		gui = new GUI(this);
		fileHandler = new FileHandler();
		unlockGraph();
		resetMarkovChain();
	}

	public static SCC_MC getInstance() {
		if (instance == null) {
			instance = new SCC_MC();
		}
		return instance;
	}

	public GUI getGui() {
		return gui;
	}

	public ModelCheckResult getModelCheckResult() {
		return modelCheckResult;
	}

	/**
	 * Set model check result and enable counter example search if possible
	 * 
	 * @param result
	 *            model check result
	 */
	public void setModelCheckResult(ModelCheckResult result) {
		modelCheckResult = result;

		if (modelCheckResult != null) {
			if (modelCheckResult.getProbability() > 0) {
				// Enable counter example search
				gui.getToolbar().getToolbarCounterExample().enableCounterExampleSearch();
			}
		}
	}

	public CounterExampleSearchParameters getCounterExampleSearchParameters() {
		return counterExampleSearchParameters;
	}

	/**
	 * Unlock graph and reset model checking and cex search
	 * 
	 */
	public void unlockGraph() {
		graphLocked = false;
		counterExampleResult = null;
		modelCheckResult = null;
		counterExampleSearchParameters = null;
		gui.getToolbar().getToolbarCounterExample().reset();
		gui.getToolbar().getToolbarEdit().makeEnabled(true);
		gui.wrapper.lockGraph(false);
	}

	/**
	 * Lock graph while doing modelchecking and cex search
	 */
	public void lockGraph() {
		graphLocked = true;
		gui.getToolbar().getToolbarEdit().makeEnabled(false);
		gui.wrapper.lockGraph(true);
	}

	/**
	 * Getter
	 * 
	 * @return true, if graph is locked
	 */
	public boolean isGraphLocked() {
		return graphLocked;
	}

	/**
	 * Parse file and show progress while doing so
	 * 
	 * @param parser
	 *            Parser for specific file
	 */
	public void executeParser(InputParser parser) {
		ParsingProgress progress = new ParsingProgress(GUI.getFrame(), parser);
		long start = System.currentTimeMillis();
		boolean couldParse = progress.run();
		long end = System.currentTimeMillis();
		Output.print("Parsing took " + (end - start) + "ms");
		if (couldParse) {
			if (parser instanceof ConfReader) {
				progress.destroy();
				return;
			}
			if (parser.getFile() != null) {
				Config.filename = parser.getFile().getName();
			}
			gui.wrapper.lockGraph(false);
			resetMarkovChain();
			markovChain = parser.getParseResult();

			// Only show graph, if stage count is not too high to avoid
			// performance drops
			boolean showGraph = true;
			int numberEdges;
			if (markovChain.isReduced()) {
				numberEdges = markovChain.getNumberOfReducedEdges();
			} else {
				numberEdges = markovChain.getNumberOfEdges();
			}
			if (numberEdges > Config.edgeCountVisible) {
				// Ask, if graph should be displayed
				String question = "The graph has " + numberEdges + " edges." + Output.getLineBreak()
						+ "This might cause problems while displaying." + Output.getLineBreak()
						+ "Would you like to display the graph anyhow?";
				showGraph = GUI.showYesNoQuestion("Show graph?", question);
			}
			if (showGraph) {
				// Show graph
				gui.wrapper.setGraphEnabled(true);
				gui.wrapper.setMarkovChain(markovChain);
			} else {
				// Don't display graph
				gui.wrapper.setGraphEnabled(false);
				gui.getOutputPanel().loadMarkovChain();
			}
			gui.wrapper.lockGraph(isGraphLocked());
			progress.destroy();
		} else {
			progress.destroy();
			GUI.showMessage("File couldn't be loaded", "Error while loading", JOptionPane.ERROR_MESSAGE);
		}
	}

	/**
	 * Calls FileOpen-Dialog, reads .tra, .xml, .dtmc and .conf files, deletes
	 * current Markov Chain and draws new MC from file.
	 */
	public void openMarkovChainFile() {
		unlockGraph();
		File file;
		ExtensionType[] extensions = new ExtensionType[] { ExtensionType.TRA, ExtensionType.XML, ExtensionType.DTMC,
				ExtensionType.CONF };
		file = fileHandler.getOpenFile(extensions);

		if (file == null)
			return;
		handleInputFile(file);
	}

	/**
	 * Handle file depending on file extension
	 * 
	 * @param file
	 *            File to handle
	 */
	public void handleInputFile(File file) {
		switch (ExtensionType.getFileExtension(file)) {
		case TRA:
			try {
				InputParser reader = new TraReader(file);
				executeParser(reader);

				gui.setLoadLabItemEnabled(true);
				boolean yesOption = GUI.showYesNoQuestion("Load labels?", "Do you want to load a file containing labels?");
				if (yesOption) {
					readLab();
				}
			} catch (Exception e1) {
				e1.printStackTrace();
			}
			break;
		case DTMC:
			try {
				InputParser reader = new DtmcReader(file);
				executeParser(reader);
				gui.setLoadLabItemEnabled(true);
			} catch (Exception e1) {
				e1.printStackTrace();
			}
			break;
		case CONF:
			try {
				counterExampleSearchParameters = new CounterExampleSearchParameters();
				InputParser reader = new ConfReader(file, counterExampleSearchParameters);
				executeParser(reader);
				if (Config.filename.isEmpty()) {
					Output.print("DTMC File: not specified");
				}
				File dtmc_file = new File(Config.path, Config.filename);
				handleInputFile(dtmc_file);
				break;
			} catch (Exception e1) {
				e1.printStackTrace();
			}
			break;
		case XML:
			handleXML(file);
			gui.setLoadLabItemEnabled(true);
			break;
		// case PM:
		// handlePrism(file);
		// break;

		case LAB:
			handleLab(file);
		}

	}

	/**
	 * Handle xml file and open markov chain
	 * 
	 * @param file
	 */
	public void handleXML(File file) {
		if (file == null) {
			return;
		}
		handleXMLString(FileHandler.fileContentToString(file));
	}

	/**
	 * Handles xml string and loads markov chain
	 * 
	 * @param xmlDoc
	 */
	public void handleXMLString(String xmlDoc) {
		InputParser p = new XmlParser(xmlDoc);
		executeParser(p);
		FileHandler.StringToFileContent(markovChain.toString(), new File("parse_result.txt"));
	}

	/**
	 * Show open file dialog for prism files
	 */
	// public void openPrism() {
	// unlockGraph();
	// File file = fileHandler.getOpenFile(ExtensionType.PM);
	// handlePrism(file);
	// }

	/**
	 * Handle prism file and open markov chain
	 * 
	 * @param file
	 *            File
	 */
	public void handlePrism(File file) {
		if (file == null)
			return;
		InputParser reader = new PrismReader(file);
		executeParser(reader);
	}

	/**
	 * Show open file dialog for lab files
	 * 
	 */
	public void readLab() {
		File file = fileHandler.getOpenFile(ExtensionType.LAB);
		handleLab(file);
	}

	/**
	 * Handle lab file and import labels
	 * 
	 * @param file
	 */
	public void handleLab(File file) {
		if (file == null) {
			return;
		}

		LabReader labelReader = new LabReader(file);
		labelReader.parse();

		// Set labels
		for (String s : labelReader.declarationList) {
			markovChain.addLabel(s);
		}
		for (Pair<Integer, String> p : labelReader.nodeList) {
			Node n = markovChain.getNodeByIntName(p.getLeft());
			markovChain.addLabelNode(n, p.getRight());
		}
		chooseLabelTarget();

		boolean yesOption = GUI.showYesNoQuestion("Set initial label?", "Do you want to set an initial label?");
		if (yesOption) {
			chooseLabelInitial();
		}
	}

	/**
	 * Choose label for targets
	 */
	public void chooseLabelTarget() {
		// Ask for target labels
		Labels result = GUI.showLabelList(markovChain.getLabels(), "Choose one or more labels for target states");
		for (String label : result) {
			markovChain.setTargetLabel(label);
		}
		gui.getOutputPanel().clear();
		gui.getOutputPanel().loadMarkovChain();
		gui.wrapper.updateDisplayedGraph();
	}

	public void chooseLabelInitial() {
		Object[] declarations = markovChain.getLabels().toArray();
		String label = (String) JOptionPane.showInputDialog(GUI.getFrame(), "Choose one label for the intial state",
				"Input label", JOptionPane.PLAIN_MESSAGE, null, declarations, null);

		ArrayList<Node> nodeList = markovChain.getNodesForLabel(label);

		Object o = JOptionPane.showInputDialog(GUI.getFrame(), "Select unique input node", "Select unique input node",
				JOptionPane.PLAIN_MESSAGE, null, nodeList.toArray(), "");
		Node initialNode = (Node) o;
		chooseInitialNode(initialNode);
	}

	public void resetMarkovChain() {
		markovChain = new MarkovChain();
		gui.wrapper.setMarkovChain(markovChain);
		gui.setLoadLabItemEnabled(false);
		gui.resetHide();
		gui.getOutputPanel().clear();
	}

	// *******************************************************//
	// tool functions //
	// *******************************************************//

	public void export(ExtensionType extension) {
		if (markovChain == null) {
			GUI.showWarningMessage("There is no Markov Chain!");
			return;
		}
		File file = fileHandler.getSaveFile(extension, null);
		if (file == null) {
			return;
		}
		OutputFormatter formatter = null;
		switch (extension) {
		case DTMC:
			if (markovChain.getInitialNode() == null) {
				GUI.showWarningMessage("An initial node needs to be defined!");
			} else if (markovChain.getTargetNodes().isEmpty()) {
				GUI.showWarningMessage("At least one target node needs to be defined!");
			} else {
				formatter = new DtmcWriter(markovChain, file);
			}
			break;
		case TRA:
			formatter = new TraWriter(markovChain, file);
			break;
		case LAB:
			formatter = new LabWriter(markovChain, file);
			break;
		case XML:
			double probability = 0;
			if (modelCheckResult != null) {
				probability = modelCheckResult.getProbability();
			}
			formatter = new XmlWriter(markovChain, file, true, probability);
			break;
		}
		if (formatter != null && formatter.write()) {
			GUI.showMessage("File successfully written", "Save successfull", JOptionPane.INFORMATION_MESSAGE);
		} else {
			GUI.showErrorMessage("Couldn't write file");
		}
	}

	public void chooseInitialNode(Node node) {
		if (node == null)
			return;
		markovChain.setInitialNode(node);
		gui.getOutputPanel().getTabNodes().changeNode(node);
		gui.wrapper.updateDisplayedGraph();
	}

	public void toggleTargetState(Node n) {
		if (n == null)
			return;
		markovChain.setTargetNode(n, !n.isTarget());
		if (n.isTarget()) {
			gui.getOutputPanel().getTabTargetNodes().removeNode(n);
		} else {
			gui.getOutputPanel().getTabTargetNodes().addNode(n);
		}
		gui.getOutputPanel().getTabNodes().changeNode(n);
		gui.wrapper.updateDisplayedGraph();
	}

	public boolean isMarkovChainConsistent() {
		for (Node node : this.markovChain.getNodes()) {
			if (!node.isConsistent()) {
				Output.print("Error node:" + node.getNumber());
				return false;
			}
		}
		return true;
	}

	public MarkovChain getMarkovChain() {
		return markovChain;
	}

	public int getMaxNode() {
		return markovChain.getNumberOfNodes();
	}

	public void searchNode(int node, boolean center) {
		Node n = markovChain.getNodeByIntName(node);
		gui.wrapper.setSelectedNode(n, true);
		if (center) {
			gui.wrapper.centerNode(n);
		}
	}

	/**
	 * Open link to website
	 */
	public void openLink() {
		try {
			String url = Config.url;
			java.awt.Desktop.getDesktop().browse(java.net.URI.create(url));
		} catch (java.io.IOException e) {
			Output.print(e.getMessage());
		}
	}

	/**
	 * Open manual
	 */
	public void openManual() {
		try {
			File pdfFile = new File(Config.pathToManualPDF);
			if (pdfFile.exists()) {
				if (Desktop.isDesktopSupported()) {
					Desktop.getDesktop().open(pdfFile);
				} else {
					Output.print("Awt Desktop is not supported!");
				}
			} else {
				Output.print("File does not exist!");
			}
			Output.print("Done");
		} catch (Exception ex) {
			ex.printStackTrace();
		}
	}

	/**
	 * Do things before terminating
	 */
	public void terminate() {
		unlockGraph();
		GUI.getFrame().dispose();
		Output.print("... and they lived happily ever after.");
	}
}
