/**
 *   COMICS - Computing Minimal Counterexamples for Discrete-time Markov Chains
 *
 *   COMICS is a stand-alone tool which performs model checking and the generation
 *   of counterexamples for discrete-time Markov Chains (DTMCs). *
 *
 *   Copyright (C) <2012> <RWTH Aachen University>
 *   Authors: Nils Jansen, Erika Abraham, Jens Katelaan, Maik Scheffler, Matthias Volk, Andreas Vorpahl
 *
 *   This program is free software: you can redistribute it and/or modify
 *   it under the terms of the GNU General Public License as published by
 *   the Free Software Foundation, either version 3 of the License, or
 *   (at your option) any later version.
 *
 *   This program is distributed in the hope that it will be useful,
 *   but WITHOUT ANY WARRANTY; without even the implied warranty of
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *   GNU General Public License for more details.
 *
 *   You should have received a copy of the GNU General Public License
 *   along with this program.  If not, see <http://www.gnu.org/licenses/>.
 *
 *   Main Contact:
 *
 *   Nils Jansen
 *   Theory of Hybrid Systems
 *   RWTH Aachen
 *   52056 Aachen
 *   Germany
 *   nils.jansen@cs.rwth-aachen.de
 *
 */

package comics.data;

import comics.utilities.Output;

/**
 * Holds parameters for counter example search
 * 
 */
public class CounterExampleSearchParameters {
	private boolean abstractSearch = true;
	private boolean globalSearch = true;
	private boolean closure = true;

	private double probabilityBound = 0;

	private boolean noAutoConcretize = false;
	private boolean stepsTillEnd = true;
	private int stepSize = 1;
	private Heuristic heuristic = Heuristic.PROBABILITY;
	private ConcretizationMode concretizationMode = ConcretizationMode.SQRT;

	private int[] concretizedNodes = new int[0];

	public boolean isAbstractSearch() {
		return abstractSearch;
	}

	public void setAbstractSearch(boolean abstractSearch) {
		this.abstractSearch = abstractSearch;
	}

	public boolean isGlobalSearch() {
		return globalSearch;
	}

	public void setGlobalSearch(boolean globalSearch) {
		this.globalSearch = globalSearch;
		if (!globalSearch) {
			// Local search must use closures
			setClosure(true);
		}
	}

	public boolean isClosure() {
		return closure;
	}

	public void setClosure(boolean closure) {
		if (!isGlobalSearch() && !closure) {
			Output.print("Can not use pathsum with local search. I will ignore this input.");
		} else {
			this.closure = closure;
		}
	}

	public double getProbabilityBound() {
		return probabilityBound;
	}

	public void setProbabilityBound(double probabilityBound) {
		this.probabilityBound = probabilityBound;
	}

	public boolean isNoAutoConcretize() {
		return noAutoConcretize;
	}

	public void setNoAutoConcretize(boolean noAutoConcretize) {
		this.noAutoConcretize = noAutoConcretize;
	}

	public boolean isStepsTillEnd() {
		return stepsTillEnd;
	}

	public void setStepsTillEnd(boolean stepsTillEnd) {
		this.stepsTillEnd = stepsTillEnd;
	}

	public int getStepSize() {
		return stepSize;
	}

	public void setStepSize(int stepSize) {
		this.stepSize = stepSize;
	}

	public Heuristic getHeuristic() {
		return heuristic;
	}

	public void setHeuristic(Heuristic heuristic) {
		this.heuristic = heuristic;
	}

	public ConcretizationMode getConcretizationMode() {
		return concretizationMode;
	}

	public void setConcretizationMode(ConcretizationMode concretizationMode) {
		this.concretizationMode = concretizationMode;
	}

	public int[] getConcretizedNodes() {
		return concretizedNodes;
	}

	public void setConcretizedNodes(int[] concretizedNodes) {
		this.concretizedNodes = concretizedNodes;
	}

	/**
	 * Heuristics for sorting SCCs to concretize. Numbers correlate with C++, so
	 * don't modify.
	 */
	public enum Heuristic {
		NONE("None: do not sort", 0), PROBABILITY("Probability: by highest outgoing probability", 2), DEGREE(
				"Degree: by lowest input/output degree", 3), MEMBERSHIP(
				"Membership: by highest relative membership in subsystem", 4);

		private final String name;
		private final int no;

		private Heuristic(String name, int no) {
			this.name = name;
			this.no = no;
		}

		public String toString() {
			return name;
		}

		public int getNo() {
			return no;
		}
	}

	/**
	 * Count of SCCs to concretize in each step. Numbers correlate with C++, so
	 * don't modify.
	 */
	public enum ConcretizationMode {
		SQRT("Square root: concretize only square root of all possible", 0), ONE("One: concretize only one", 1), ALL(
				"All: concretize all", 2);

		private final String name;
		private final int no;

		private ConcretizationMode(String name, int no) {
			this.name = name;
			this.no = no;
		}

		public String toString() {
			return name;
		}

		public int getNo() {
			return no;
		}
	}

}
