/**
 *   COMICS - Computing Minimal Counterexamples for Discrete-time Markov Chains
 *
 *   COMICS is a stand-alone tool which performs model checking and the generation
 *   of counterexamples for discrete-time Markov Chains (DTMCs). *
 *
 *   Copyright (C) <2012> <RWTH Aachen University>
 *   Authors: Nils Jansen, Erika Abraham, Jens Katelaan, Maik Scheffler, Matthias Volk, Andreas Vorpahl
 *
 *   This program is free software: you can redistribute it and/or modify
 *   it under the terms of the GNU General Public License as published by
 *   the Free Software Foundation, either version 3 of the License, or
 *   (at your option) any later version.
 *
 *   This program is distributed in the hope that it will be useful,
 *   but WITHOUT ANY WARRANTY; without even the implied warranty of
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *   GNU General Public License for more details.
 *
 *   You should have received a copy of the GNU General Public License
 *   along with this program.  If not, see <http://www.gnu.org/licenses/>.
 *
 *   Main Contact:
 *
 *   Nils Jansen
 *   Theory of Hybrid Systems
 *   RWTH Aachen
 *   52056 Aachen
 *   Germany
 *   nils.jansen@cs.rwth-aachen.de
 *
 */

package comics.graph.data;

import java.io.Serializable;
import java.util.ArrayList;
import java.util.TreeSet;

import comics.graph.gui.GraphModificationListener;
import comics.graph.gui.GraphWrapper;
import comics.gui.GUI;
import comics.utilities.Output;
import com.mxgraph.model.mxCell;

public class MarkovChain implements Comparable<MarkovChain>, Serializable, GraphModificationListener {

	private static final long serialVersionUID = 1L;

	private int maxNodeNo;
	private int id;
	private int recursionDepth;
	// if isReduced-flag is false, object is a standard Markov chain without
	// subgraphs. if isReduced, MarkovChain has only InputNodes and OutputNodes.
	private boolean isReduced;
	private boolean expandedInGraph;

	private MarkovChain parent;

	// JK: Only read lists via getters if you want to obtain sorted lists!
	private TreeSet<Node> nodes;
	private TreeSet<Edge> edges;
	private TreeSet<Node> targetNodes;
	private Node initialNode = null; // Only set if Markov Chain is not reduced
	private TreeSet<Node> inputNodes; // Only set if MC is reduced
	private TreeSet<MarkovChain> subgraphs; // underlying SCCs
	private TreeSet<Edge> reducedEdges = null; // (Abstract) Edges of reduced
												// graph

	private Labels labels;

	public MarkovChain() {
		nodes = new TreeSet<Node>();
		edges = new TreeSet<Edge>();
		inputNodes = new TreeSet<Node>();
		targetNodes = new TreeSet<Node>();
		subgraphs = new TreeSet<MarkovChain>();
		labels = new Labels();
		maxNodeNo = 0;
		recursionDepth = 0;
	}

	public boolean isEmpty() {
		return nodes.isEmpty();
	}

	public int getRecursionDepth() {
		return recursionDepth;
	}

	public void setId(int id) {
		this.id = id;
	}

	public int getId() {
		return id;
	}

	public void setParent(MarkovChain parent) {
		this.parent = parent;
		recursionDepth = parent.getRecursionDepth() + 1;
	}

	public MarkovChain getParent() {
		return parent;
	}

	public boolean isReduced() {
		return isReduced;
	}

	public void setReduced(boolean isReduced) {
		this.isReduced = isReduced;
		if (isReduced && reducedEdges == null) {
			reducedEdges = new TreeSet<Edge>();
		}
	}

	/**
	 * Mark as expanded or reduced in Graph at the moment
	 * 
	 * @param expanded
	 *            if true, SCC is expanded
	 */
	public void setExpandedInGraph(boolean expanded) {
		this.expandedInGraph = expanded;
	}

	/**
	 * Getter
	 * 
	 * @return if true, SCC is expanded at the moment
	 */
	public boolean isExpandedInGraph() {
		return expandedInGraph;
	}

	/**
	 * Add new label to markov chain
	 * 
	 * @param label
	 *            new label
	 */
	public void addLabel(String label) {
		labels.add(label);
	}

	/**
	 * Getter
	 * 
	 * @return labels
	 */
	public Labels getLabels() {
		return labels;
	}

	/**
	 * Getter
	 * 
	 * @param node
	 *            Node
	 * @return labels of this node
	 */
	public Labels getLabelsNode(Node node) {
		return node.getLabels();
	}

	public void createAllNodesTil(int n) {
		while (maxNodeNo < n) {
			addNode();
		}
	}

	/**
	 * Add node to markov chain
	 * 
	 * @return new created node
	 */
	public Node addNode() {
		return addNode(maxNodeNo + 1);
	}

	/**
	 * Add node to markov chain with given id
	 * 
	 * @param id
	 *            Id of new Node
	 * @return new created node
	 */
	public Node addNode(int id) {
		Node node = new Node(id);
		nodes.add(node);
		if (getMaxNodeNo() < id) {
			maxNodeNo = id;
		}
		node.setCorrespondingMarkovChain(this);
		return node;
	}

	/**
	 * Remove node
	 * 
	 * @param node
	 *            node to remove
	 */
	public void removeNode(Node node) {
		if (node == null) {
			return;
		}
		deleteConnections(node);
		nodes.remove(node);
		if (node.isInitial()) {
			setInitialNode(null);
		}
		if (node.isInputOfScc()) {
			inputNodes.remove(node);
		}
		if (node.isTarget()) {
			targetNodes.remove(node);
		}
	}

	/**
	 * Deletes for node all connecting edges. TODO more efficient!
	 * 
	 * @param node
	 *            node which edges should be removed
	 */
	private void deleteConnections(Node node) {
		// delete outgoing edges
		for (Edge edge : getConnections(node)) {
			removeEdge(edge);
		}
	}

	/**
	 * Add a concrete edge to the MC
	 * 
	 * @param sourceNode
	 *            source node
	 * @param targetNode
	 *            target node
	 * @param probability
	 *            probability
	 * @return new created edge
	 * @throws ProbabilityExceededException
	 *             if probability is not valid
	 */
	public Edge addEdge(Node sourceNode, Node targetNode, double probability) throws ProbabilityExceededException {
		return addEdge(sourceNode, targetNode, probability, false);
	}

	/**
	 * Add a edge to the MC
	 * 
	 * @param sourceNode
	 *            source node
	 * @param targetNode
	 *            target node
	 * @param probability
	 *            probability
	 * @param reduced
	 *            concrete or reduced
	 * @return new created edge
	 * @throws ProbabilityExceededException
	 *             if probability is not valid
	 */
	public Edge addEdge(Node sourceNode, Node targetNode, double probability, boolean reduced)
			throws ProbabilityExceededException {
		Edge edge = new Edge(sourceNode, targetNode, probability, reduced);
		addEdge(edge);
		return edge;
	}

	/**
	 * Adds an edge either as concrete or as reduced as, depending on the
	 * corresponding edge property
	 * 
	 * @param edge
	 *            The edge to be added
	 * @return true, if edge exists already
	 * @throws ProbabilityExceededException
	 */
	private boolean addEdge(Edge edge) throws ProbabilityExceededException {
		boolean existing = false;
		if (!edge.isReduced()) {
			if (edges.contains(edge)) {
				// Edge exists already
				// TODO check if correct
				Edge existingEdge = edges.floor(edge);
				// Check for consistent probability or throw exception
				double probability = existingEdge.getProbability() + edge.getProbability();
				if (isConsistent(probability)) {
					existingEdge.setProbability(probability);
					existing = true;
				} else {
					throw new ProbabilityExceededException("" + probability);
				}
			} else {
				edges.add(edge);
				edge.getSourceNode().getAdjacentNodes().add(edge.getTargetNode());
				edge.getTargetNode().getAdjacentToNodes().add(edge.getSourceNode());
			}
			edge.getSourceNode().setOutgoingProbability(edge.getProbability() + edge.getSourceNode().getOutgoingProbability());
		} else {
			reducedEdges.add(edge);
			edge.getSourceNode().getAdjacentNodes().add(edge.getTargetNode());
			edge.getTargetNode().getAdjacentToNodes().add(edge.getSourceNode());
		}
		return existing;
	}

	/**
	 * Remove edge
	 * 
	 * @param edge
	 *            Edge to remove
	 */
	public void removeEdge(Edge edge) {
		if (edge == null) {
			return;
		}
		edge.getSourceNode().getAdjacentNodes().remove(edge.getTargetNode());
		double outgoingProbability = edge.getSourceNode().getOutgoingProbability();
		edge.getSourceNode().setOutgoingProbability(outgoingProbability - edge.getProbability());
		edge.getTargetNode().getAdjacentToNodes().remove(edge.getSourceNode());
		if (edge.isReduced()) {
			reducedEdges.remove(edge);
		} else {
			edges.remove(edge);
		}
	}

	public int getNumberOfNodes() {
		return nodes.size();
	}

	public int getNumberOfEdges() {
		return edges.size();
	}

	public int getMaxNodeNo() {
		return maxNodeNo;
	}

	public int getNumberOfReducedEdges() {
		return reducedEdges.size();
	}

	// TODO encapsulating?
	public TreeSet<Node> getNodes() {
		return nodes;
	}

	// TODO encapsulating?
	public TreeSet<Edge> getEdges() {
		return edges;
	}

	/**
	 * Only to be called on reduced markov chains!
	 * 
	 * @return The set of reduced edges
	 */
	// TODO encapsulating?
	public TreeSet<Edge> getReducedEdges() {
		if (isReduced) {
			return reducedEdges;
		} else
			return null;
	}

	// TODO encapsulating?
	public TreeSet<Node> getTargetNodes() {
		return targetNodes;
	}

	// TODO encapsulating?
	public TreeSet<Node> getInputNodes() {
		return inputNodes;
	}

	// TODO encapsulating?
	public TreeSet<Edge> getConnections(Node node) {
		TreeSet<Edge> edges = new TreeSet<Edge>();
		for (Edge e : edges) {
			if (e.getSourceNode().equals(node) || e.getTargetNode().equals(node)) {
				edges.add(e);
			}
		}
		return edges;
	}

	// TODO encapsulating?
	public TreeSet<Edge> getOutgoingEdges(Node node) {
		TreeSet<Edge> edges = new TreeSet<Edge>();
		for (Edge e : edges) {
			if (e.getSourceNode().equals(node)) {
				edges.add(e);
			}
		}
		return edges;
	}

	// TODO encapsulating?
	public TreeSet<Edge> getIngoingEdges(Node node) {
		TreeSet<Edge> edges = new TreeSet<Edge>();
		for (Edge e : edges) {
			if (e.getTargetNode().equals(node)) {
				edges.add(e);
			}
		}
		return edges;
	}

	public Node getInitialNode() {
		return initialNode;
	}

	public void setInitialNode(Node node) {
		if (initialNode != null) {
			initialNode.setInitial(false);
		}
		initialNode = node;

		if (initialNode != null) {
			initialNode.setInitial(true);
		}
	}

	public void setInputOfSccNode(Node node) {
		setInputOfSccNode(node, true);
	}

	public void setInputOfSccNode(Node node, boolean input) {
		if (input) {
			node.setInputOfScc(true);
			inputNodes.add(node);
		} else {
			node.setInputOfScc(false);
			inputNodes.remove(node);
		}
	}

	public void setOutputOfSccNode(Node node) {
		setOutputOfSccNode(node, true);
	}

	public void setOutputOfSccNode(Node node, boolean output) {
		if (output) {
			node.setOutputOfScc(true);
			targetNodes.add(node);
		} else {
			node.setOutputOfScc(false);
			targetNodes.remove(node);
		}
	}

	/**
	 * Set node as reduced
	 * 
	 * @param node
	 *            node to set
	 * @param reduced
	 *            if true, node is reduced
	 */
	public void setReducedNode(Node node, boolean reduced) {
		if (reduced) {
			if (!isReduced) {
				Output.print("Markov Chain is not reduced!");
				return;
			}
		}
		node.setReduced(reduced);
	}

	/**
	 * Add node to target node list and mark as target node as well.
	 * 
	 * @param node
	 *            node to add
	 */
	public void setTargetNode(Node node) {
		targetNodes.add(node);
		node.setTarget(true);
	}

	/**
	 * Set/unset node as target node
	 * 
	 * @param node
	 *            node to remove
	 * @param target
	 *            if true, node is set as target, otherwise target is unset
	 */
	public void setTargetNode(Node node, boolean target) {
		if (target) {
			setTargetNode(node);
		} else {
			targetNodes.remove(node);
			node.setTarget(false);
		}
	}

	/**
	 * Mark node as normal node which is no input/output/target node
	 * 
	 * @param node
	 *            node to set
	 */
	public void setNormalNode(Node node) {
		setOutputOfSccNode(node, false);
		setInputOfSccNode(node, false);
		setTargetNode(node, false);
		nodes.add(node);
	}

	/**
	 * Set this markov chain as reference for node
	 * 
	 * @param node
	 *            Node to set mc
	 */
	public void setCorrespondingMarkovChainNode(Node node) {
		node.setCorrespondingMarkovChain(this);
	}

	/**
	 * Add label to node
	 * 
	 * @param node
	 *            Node with label
	 * @param label
	 *            new label
	 */
	public void addLabelNode(Node node, String label) {
		if (labels.contains(label)) {
			node.addLabel(label);
		}
	}

	/**
	 * Change probability for Edge
	 * 
	 * @param edge
	 *            Edge to change probability
	 * @param probability
	 *            new probability to set
	 * @throws ProbabilityExceededException
	 *             if probability is not valid
	 */
	public void setProbabilityEdge(Edge edge, double probability) throws ProbabilityExceededException {
		if (!isConsistent(probability)) {
			throw new ProbabilityExceededException(edge.toTraString());
		} else {
			double outgoingWithoutEdge = edge.getSourceNode().getOutgoingProbability() - edge.getProbability();
			edge.getSourceNode().setOutgoingProbability(outgoingWithoutEdge + probability);
			edge.setProbability(probability);
		}
	}

	/**
	 * Checks if node with given id exists already
	 * 
	 * @param id
	 *            id of node to check
	 * @return true, if node already exists
	 */
	public boolean containsNode(int id) {
		Node tmp = new Node(id);
		return nodes.contains(tmp);
	}

	/**
	 * The node of given id, or null if none is found (which should not happen!)
	 * 
	 * @param id
	 * @return The corresponding node object, or null
	 */
	public Node getNodeByIntName(int id) {
		Node tmp = new Node(id);
		Node bestFit = nodes.floor(tmp);
		if (tmp.equals(bestFit)) {
			return bestFit;
		} else {
			assert (false);
			return null;
		}
	}

	public Edge getEdgeByNodes(Node source, Node target) {
		for (Edge edge : edges) {
			if (edge.getSourceNode().equals(source) && edge.getTargetNode().equals(target)) {
				return edge;
			}
		}
		assert (false);
		return null;
	}

	public boolean containsNode(Node node) {
		return nodes.contains(node);
	}

	// TODO encapsulating?
	public TreeSet<MarkovChain> getSubgraphs() {
		return subgraphs;
	}

	public void addSubgraph(MarkovChain subgraph) {
		subgraphs.add(subgraph);
		subgraph.setParent(this);
	}

	public String to_scc_mc_string() throws InconsistentMarkovChainException {
		String lb = Output.getLineBreak();
		String traString = "";
		try {
			traString = traString + ("STATES " + getMaxNodeNo() + lb);
			traString = traString + ("TRANSITIONS " + getNumberOfEdges() + lb);
			traString = traString + ("INITIAL " + getInitialNode().getNumber() + lb);
			for (Node node : getTargetNodes()) {
				traString = traString + ("TARGET " + node.getNumber() + lb);
			}
			for (Edge edge : getEdges()) {
				traString = traString
						+ (edge.getSourceNode().getNumber() + " " + edge.getTargetNode().getNumber() + " "
								+ edge.getProbability() + lb);
			}
		} catch (Exception e) {
			throw new InconsistentMarkovChainException("Markov chain in incomplete state.");
		}
		return traString;
	}

	public String to_mrmc_string() {
		String lb = Output.getLineBreak();
		String traString = "";
		traString = traString + ("STATES " + getMaxNodeNo() + lb);
		traString = traString + ("TRANSITIONS " + getNumberOfEdges() + lb);
		for (Edge edge : getEdges()) {
			traString = traString
					+ (edge.getSourceNode().getNumber() + " " + edge.getTargetNode().getNumber() + " " + edge.getProbability() + lb);
		}
		return traString;
	}

	@Override
	public String toString() {
		String lb = Output.getLineBreak();
		StringBuffer output = new StringBuffer("------");
		output.append(lb + "DTMC" + lb);
		if (this.isReduced)
			output.append("Reduced; Recursion depth: " + recursionDepth + lb + lb);

		output.append("Nodes: ");
		for (Node node : getNodes()) {
			output.append(node.getNumber() + ", ");
		}
		output.append(lb);

		output.append("Edges: ");
		for (Edge edge : getEdges()) {
			output.append(lb + edge.toString());
		}

		output.append(lb + "Initial Node/Input Node(s): ");
		if (this.inputNodes.size() > 0) {
			for (Node node : getInputNodes()) {
				output.append(node.getNumber() + ", ");
			}
		}
		// Note: Use the accessor so that the node is validated first
		else if (getInitialNode() != null)
			output.append(getInitialNode().getNumber() + ", ");

		output.append(lb + "Target Nodes: ");
		if (getTargetNodes().size() > 0) {
			for (Node node : getTargetNodes()) {
				output.append(node.getNumber() + ", ");
			}
		}

		if (this.subgraphs.size() > 0) {
			output.append(lb + "Subgraphs:" + lb);
			int i = 1;
			for (MarkovChain mc : getSubgraphs()) {
				output.append("Subgraph #" + i + lb);
				output.append(mc.toString());
			}
		}

		output.append(lb + "(end DTMC)" + lb);
		output.append(lb + "------" + lb);
		return output.toString();
	}

	public String testOutput(String identation) {
		String lb = Output.getLineBreak();
		StringBuffer output = new StringBuffer(identation + "------");
		output.append(lb + identation + "Markov Chain" + lb);
		output.append(identation + "Id: " + id + lb);
		if (this.isReduced)
			output.append(identation + "Reduced; Recursion depth: " + recursionDepth + lb + lb);

		output.append(identation + "Nodes: ");
		for (Node node : getNodes()) {
			output.append(node.getNumber() + ", ");
		}
		output.append(lb);

		output.append(identation + "Edges: ");
		for (Edge edge : getEdges()) {
			output.append(lb + identation + edge.toPropertyString());
		}
		output.append(lb + identation + "Reduced Edges: ");
		if (getReducedEdges() != null) {
			for (Edge edge : getReducedEdges()) {
				output.append(lb + identation + edge.toPropertyString());
			}
		}

		output.append(lb + identation + "Initial Node(s): ");
		if (this.inputNodes.size() > 0) {
			for (Node node : getInputNodes()) {
				output.append(node.getNumber() + ", ");
			}
		}
		// Note: Use the accessor so that the node is validated first
		else if (getInitialNode() != null)
			output.append(getInitialNode().getNumber() + ", ");

		output.append(lb + identation + "Target Nodes: ");
		if (getTargetNodes().size() > 0) {
			for (Node node : getTargetNodes()) {
				output.append(node.getNumber() + ", ");
			}
		}

		if (this.subgraphs.size() > 0) {
			output.append(lb + identation + "Subgraphs:" + getSubgraphs().size());
			int i = 1;
			for (MarkovChain mc : getSubgraphs()) {
				output.append(lb + identation + "Subgraph #" + i + lb);
				i++;
				output.append(mc.testOutput(identation + "  "));
				output.append(lb + identation + "(end Subgraph)");
			}
		}

		output.append(lb + identation + "(end DTMC)" + lb);
		output.append(lb + identation + "------" + lb);
		return output.toString();
	}

	/**
	 * Get second edge in opposite/same direction
	 * 
	 * @param edge
	 *            Edge
	 * @param sameDirection
	 *            if true, edge in same direction is searched
	 * @return second edge, null if not exists
	 */
	public Edge getSecondEdge(Edge edge, boolean sameDirection) {
		for (Edge edge2 : edges) {
			if (sameDirection & edge2.equals(edge))
				return edge2;
			else if ((edge2.getSourceNode().equals(edge.getTargetNode()) && edge2.getTargetNode().equals(edge.getSourceNode())))
				return edge2;
		}
		return null;
	}

	/**
	 * Self loops are inserted to ensure outgoing probability of 1.
	 * 
	 * @param gui
	 *            GUI to display changes on GUI
	 */
	public void insertSelfLoops(GUI gui) {
		for (Node n : getNodes()) {
			if (!n.isConsistent()) {
				double prob = 1 - n.getOutgoingProbability();
				Edge selfLoop = getEdgeByNodes(n, n);
				if (selfLoop != null) {
					// Only change probability
					selfLoop.setProbability(prob + selfLoop.getProbability());
					n.setOutgoingProbability(1);
					gui.wrapper.updateDisplayedGraph();
					gui.getOutputPanel().getTabEdges().changeEdge(selfLoop);
				} else {
					try {
						selfLoop = addEdge(n, n, prob);
					} catch (ProbabilityExceededException e) {
					}
					gui.wrapper.addEdgeToGraph(selfLoop);
				}
			}
		}
	}

	/**
	 * Transform to markov chain with only one remaining target node. Therefore
	 * insert new target node, make former targets non-targets and add edges
	 * from former targets to new target with probability 1.
	 * 
	 * @param wrapper
	 *            Graphwrapper to display changes on GUI
	 */
	public void transformToOneTarget(GraphWrapper wrapper) {
		if (getTargetNodes().size() > 1) {
			// Add new target
			Node target = addNode();
			wrapper.addNodeToGraph(target);

			// Add edges and make non-target
			for (Node n : getTargetNodes()) {
				n.setTarget(false);
				Edge e = null;
				try {
					e = addEdge(n, target, 1);
				} catch (ProbabilityExceededException e1) {
				}
				wrapper.addEdgeToGraph(e);
			}
			getTargetNodes().clear();

			// Make new node target
			setTargetNode(target);

			wrapper.updateDisplayedGraph();
		}
	}

	/**
	 * Set all nodes with given label as target nodes. Old targets are reset.
	 * 
	 * @param label
	 *            labels for target states
	 */
	public void setTargetLabel(String label) {
		// Reset old targets
		while (!getTargetNodes().isEmpty()) {
			setTargetNode(getTargetNodes().first(), false);
		}

		// Set new targets
		for (Node node : getNodes()) {
			if (node.getLabels().contains(label)) {
				setTargetNode(node);
			}
		}
	}

	/**
	 * Return all nodes with given label
	 * 
	 * @param label
	 *            label
	 */
	public ArrayList<Node> getNodesForLabel(String label) {
		ArrayList<Node> nodeList = new ArrayList<Node>();
		for (Node node : getNodes()) {
			if (node.getLabels().contains(label)) {
				nodeList.add(node);
			}
		}
		return nodeList;
	}

	public boolean isMarkovChainConsistent() {
		for (Node node : getNodes()) {
			if (!node.isConsistent()) {
				return false;
			}
		}
		return true;
	}

	/**
	 * Checks Node for consistency
	 * 
	 * @param node
	 *            Node to check
	 * @return true, if consistent
	 * @throws ProbabilityExceededException
	 * @throws InconsistentMarkovChainException
	 */
	public boolean checkConsistency(Node node) throws ProbabilityExceededException, InconsistentMarkovChainException {
		// TODO make efficient
		for (Edge e : edges) {
			if (!isConsistent(e.getProbability())) {
				throw new ProbabilityExceededException(e.toTraString());
			}
		}
		if (!node.isConsistent()) {
			throw new InconsistentMarkovChainException("Node " + node.toString() + ": " + node.getOutgoingProbability());
		}
		return true;
	}

	/**
	 * Checks given probability for consistency
	 * 
	 * @param probability
	 *            probability to check
	 * @return true, if consistent
	 */
	public static boolean isConsistent(double probability) {
		if (probability < 0 || probability > 1)
			return false;
		else
			return true;
	}

	/**
	 * Check for initial and target states
	 * 
	 * @return true, if inital and target states exist
	 */
	public boolean hasInitialAndTarget() {
		// Check for initial node
		if (isReduced) {
			if (getInputNodes().isEmpty()) {
				return false;
			}
		} else if (getInitialNode() == null) {
			return false;
		}
		// Check for target states
		if (getTargetNodes().isEmpty()) {
			return false;
		}
		return true;
	}

	@Override
	public void nodeAdded(mxCell cell) {
		Node node = addNode();
		// Link cell
		cell.setValue(node);
		node.setGraphCell(cell);
	}

	@Override
	public void nodeRemoved(mxCell n) {
		removeNode((Node) n.getValue());
	}

	@Override
	public boolean edgeAdded(mxCell cell) {
		Node src = (Node) cell.getSource().getValue();
		Node trg = (Node) cell.getTarget().getValue();
		if (src == null || trg == null) {
			Output.print("Error: One of the edge terminals is null");
		}

		Edge e = new Edge(src, trg, 0);
		boolean existed = false, successfull = false;
		while (!successfull) {
			try {
				e.setProbability(GUI.getProbability("Enter probability for edge", 1));
				existed = addEdge(e);
				successfull = true;
			} catch (ProbabilityExceededException exception) {
				GUI.showErrorMessage("Probability can't be greater than 1");
			}
		}
		// Link cell
		if (!existed) {
			e.setGraphCell(cell);
		}
		cell.setValue(e);
		return existed;
	}

	@Override
	public void edgeRemoved(mxCell e) {
		removeEdge((Edge) e.getValue());
	}

	@Override
	public int compareTo(MarkovChain o) {
		return Math.abs(this.getId()) - Math.abs(o.getId());
	}
}
