/**
 *   COMICS - Computing Minimal Counterexamples for Discrete-time Markov Chains
 *
 *   COMICS is a stand-alone tool which performs model checking and the generation
 *   of counterexamples for discrete-time Markov Chains (DTMCs). *
 *
 *   Copyright (C) <2012> <RWTH Aachen University>
 *   Authors: Nils Jansen, Erika Abraham, Jens Katelaan, Maik Scheffler, Matthias Volk, Andreas Vorpahl
 *
 *   This program is free software: you can redistribute it and/or modify
 *   it under the terms of the GNU General Public License as published by
 *   the Free Software Foundation, either version 3 of the License, or
 *   (at your option) any later version.
 *
 *   This program is distributed in the hope that it will be useful,
 *   but WITHOUT ANY WARRANTY; without even the implied warranty of
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *   GNU General Public License for more details.
 *
 *   You should have received a copy of the GNU General Public License
 *   along with this program.  If not, see <http://www.gnu.org/licenses/>.
 *
 *   Main Contact:
 *
 *   Nils Jansen
 *   Theory of Hybrid Systems
 *   RWTH Aachen
 *   52056 Aachen
 *   Germany
 *   nils.jansen@cs.rwth-aachen.de
 *
 */

package comics.graph.data;

import java.io.Serializable;
import java.util.TreeSet;

import comics.graph.gui.GraphWrapper;
import comics.utilities.Output;

import com.mxgraph.model.mxCell;

/**
 * Class for node
 */
public class Node implements Comparable<Node>, Serializable {

	private static final long serialVersionUID = 1L;
	private int no;
	private double outgoingProbability;
	private boolean initial;
	private boolean target;
	private boolean reduced;
	private boolean hidden;
	private boolean inputOfScc;
	private boolean outputOfScc;

	private boolean positionSet = false;
	private double xpos = -1;
	private double ypos = -1;

	private TreeSet<Node> adjacentNodes = new TreeSet<Node>();
	private TreeSet<Node> adjacentToNodes = new TreeSet<Node>();

	private Labels labels = new Labels();

	private mxCell graphCell = null;

	// note, that the following are only instantiated, if Node is input or
	// output node of scc, respectively
	private MarkovChain correspondingMarkovChain = null;

	protected Node(int no) {
		this.no = no;
		this.setInitial(false);
		this.setTarget(false);
		this.setReduced(false);
		this.setHidden(false);
		this.setInputOfScc(false);
		this.setOutputOfScc(false);
		this.setOutgoingProbability(0);
	}

	protected Node(String name) throws NumberFormatException {
		this(Integer.parseInt(name));
	}

	protected Node(int name, double xpos, double ypos) {
		this(name);
		this.setPosition(xpos, ypos);
	}

	public int getNumber() {
		return no;
	}

	public mxCell getGraphCell() {
		return graphCell;
	}

	public void setGraphCell(mxCell graphCell) {
		this.graphCell = graphCell;
	}

	public boolean isPositionSet() {
		return positionSet;
	}

	public double getXpos() {
		if (positionSet) {
			return xpos;
		} else {
			return -1;
		}
	}

	public double getYpos() {
		if (positionSet) {
			return ypos;
		} else {
			return -1;
		}
	}

	public void setPosition(double xpos, double ypos) {
		this.xpos = xpos;
		this.ypos = ypos;
		this.positionSet = true;
	}

	public double getOutgoingProbability() {
		return outgoingProbability;
	}

	protected void setOutgoingProbability(double outgoingProbability) {
		if (outgoingProbability > 1 || outgoingProbability < 0) {
			Output.print("Outgoing probability " + outgoingProbability + " for Node " + getNumber() + " is not valid.");
		} else {
			this.outgoingProbability = outgoingProbability;
		}
	}

	public boolean isInitial() {
		return initial;
	}

	protected void setInitial(boolean isInitial) {
		this.initial = isInitial;
		adjustStyle();
	}

	public boolean isTarget() {
		return target;
	}

	protected void setTarget(boolean isTarget) {
		this.target = isTarget;
		adjustStyle();
	}

	public boolean isReduced() {
		return reduced;
	}

	protected void setReduced(boolean isReduced) {
		this.reduced = isReduced;
		adjustStyle();
	}

	public boolean isHidden() {
		return hidden;
	}

	public void setHidden(boolean isHidden) {
		this.hidden = isHidden;
	}

	protected Labels getLabels() {
		return labels;
	}

	protected void addLabel(String label) {
		labels.add(label);
	}

	public boolean isInputOfScc() {
		return inputOfScc;
	}

	protected void setInputOfScc(boolean inputOfScc) {
		this.inputOfScc = inputOfScc;
		adjustStyle();
	}

	public boolean isOutputOfScc() {
		return this.outputOfScc;
	}

	protected void setOutputOfScc(boolean outputOfScc) {
		this.outputOfScc = outputOfScc;
		adjustStyle();
	}

	public MarkovChain getCorrespondingMarkovChain() {
		return correspondingMarkovChain;
	}

	protected void setCorrespondingMarkovChain(MarkovChain correspondingMarkovChain) {
		this.correspondingMarkovChain = correspondingMarkovChain;
	}

	protected TreeSet<Node> getAdjacentNodes() {
		return adjacentNodes;
	}

	protected TreeSet<Node> getAdjacentToNodes() {
		return adjacentToNodes;
	}

	/**
	 * Node is absorbing if it only has a self loop with probability 1 or no
	 * outgoing transition
	 * 
	 * @return true, if absorbing, else otherwise
	 */
	public boolean isAbsorbing() {
		if (getAdjacentNodes().size() == 0) {
			return true;
		} else if (getAdjacentNodes().size() == 1 && this.equals(getAdjacentNodes().first())) {
			return true;
		} else {
			return false;
		}
	}

	/**
	 * Checks Node for consistency without Exception
	 * 
	 * @return true, if consistent
	 * 
	 */
	public boolean isConsistent() {
		// TODO make efficient
		if (getOutgoingProbability() < 1) {
			Output.print("WARNING: Outgoing probability of Node " + getNumber() + " is too low (by "
					+ (1 - getOutgoingProbability()) + ")");
			return false;
		}
		if (getOutgoingProbability() > 1) {
			Output.print("WARNING: Outgoing probability of Node " + getNumber() + " is too high (by "
					+ (getOutgoingProbability() - 1) + ")");
			return false;
		}
		return true;
	}

	@Override
	public int compareTo(Node o) {
		return Math.abs(this.getNumber()) - Math.abs(o.getNumber());
	}

	@Override
	public boolean equals(Object o) {
		if (!(o instanceof Node)) {
			return false;
		} else {
			return ((Node) o).getNumber() == getNumber();
		}
	}

	/**
	 * Checks, if node has adjacent nodes
	 * 
	 * @return true, if has adjacent nodes
	 */
	public boolean hasAdjacentNodes() {
		if (!getAdjacentNodes().isEmpty()) {
			return true;
		} else if (!getAdjacentToNodes().isEmpty()) {
			return true;
		} else {
			return false;
		}
	}

	/**
	 * Set specific style for node depending on several properties (initial,
	 * target, reduced)
	 * 
	 */
	public void adjustStyle() {
		if (graphCell == null)
			return;

		String style = GraphWrapper.STYLE_DEFAULT_NODE;
		if (isReduced()) {
			style = GraphWrapper.STYLE_REDUCED_NODE;
		} else if (isInputOfScc()) {
			style = GraphWrapper.STYLE_SUBGRAPH_NODE;
		} else {
			if (isTarget()) {
				if (isInitial()) {
					style = GraphWrapper.STYLE_INITIAL_TARGET_NODE;
				} else {
					style = GraphWrapper.STYLE_TARGET_NODE;
				}
			} else if (isInitial()) {
				style = GraphWrapper.STYLE_INITIAL_NODE;
			}
		}

		graphCell.setStyle(style);
	}

	/**
	 * Return string with properties of node
	 * 
	 * @return string
	 */
	public String propertyString() {
		String s = "";
		if (isInitial()) {
			s += "Initial";
		}
		if (isTarget()) {
			if (!s.isEmpty()) {
				s += ", ";
			}
			s += "Target";
		}
		if (isInputOfScc()) {
			if (!s.isEmpty()) {
				s += ", ";
			}
			s += "InputOfSCC";
		}
		if (reduced) {
			if (!s.isEmpty()) {
				s += ", ";
			}
			s += "Reduced";
		}

		if (!labels.isEmpty()) {
			if (!s.isEmpty()) {
				s += ", ";
			}
			s += "[";
			s += labels.toString();
			s += "]";
		}
		return s;
	}

	@Override
	public String toString() {
		return "" + getNumber();
	}

}
