/**
 *   COMICS - Computing Minimal Counterexamples for Discrete-time Markov Chains
 *
 *   COMICS is a stand-alone tool which performs model checking and the generation
 *   of counterexamples for discrete-time Markov Chains (DTMCs). *
 *
 *   Copyright (C) <2012> <RWTH Aachen University>
 *   Authors: Nils Jansen, Erika Abraham, Jens Katelaan, Maik Scheffler, Matthias Volk, Andreas Vorpahl
 *
 *   This program is free software: you can redistribute it and/or modify
 *   it under the terms of the GNU General Public License as published by
 *   the Free Software Foundation, either version 3 of the License, or
 *   (at your option) any later version.
 *
 *   This program is distributed in the hope that it will be useful,
 *   but WITHOUT ANY WARRANTY; without even the implied warranty of
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *   GNU General Public License for more details.
 *
 *   You should have received a copy of the GNU General Public License
 *   along with this program.  If not, see <http://www.gnu.org/licenses/>.
 *
 *   Main Contact:
 *
 *   Nils Jansen
 *   Theory of Hybrid Systems
 *   RWTH Aachen
 *   52056 Aachen
 *   Germany
 *   nils.jansen@cs.rwth-aachen.de
 *
 */

package comics.graph.gui;

import java.awt.Color;
import java.awt.Cursor;
import java.awt.event.MouseEvent;
import java.awt.event.MouseWheelListener;
import java.util.ArrayList;
import java.util.Hashtable;
import java.util.LinkedList;
import java.util.List;

import comics.SCC_MC;
import comics.data.Config;
import comics.graph.data.Edge;
import comics.graph.data.MarkovChain;
import comics.graph.data.Node;
import comics.graph.data.ProbabilityExceededException;
import comics.gui.EditMode;
import comics.gui.GUI;
import comics.gui.actions.GraphMouseListener;
import comics.gui.output_panel.OutputPanel;
import comics.utilities.Output;

import com.mxgraph.layout.mxCircleLayout;
import com.mxgraph.layout.mxCompactTreeLayout;
import com.mxgraph.layout.mxFastOrganicLayout;
import com.mxgraph.layout.mxGraphLayout;
import com.mxgraph.layout.mxIGraphLayout;
import com.mxgraph.layout.mxParallelEdgeLayout;
import com.mxgraph.layout.mxPartitionLayout;
import com.mxgraph.layout.mxStackLayout;
import com.mxgraph.layout.hierarchical.mxHierarchicalLayout;
import com.mxgraph.model.mxCell;
import com.mxgraph.model.mxGraphModel.mxChildChange;
import com.mxgraph.swing.mxGraphComponent;
import com.mxgraph.swing.handler.mxGraphHandler;
import com.mxgraph.swing.handler.mxRubberband;
import com.mxgraph.swing.mxGraphComponent.mxGraphControl;
import com.mxgraph.util.mxConstants;
import com.mxgraph.util.mxEvent;
import com.mxgraph.util.mxEventObject;
import com.mxgraph.util.mxPoint;
import com.mxgraph.util.mxUndoManager;
import com.mxgraph.util.mxUndoableEdit;
import com.mxgraph.util.mxEventSource.mxIEventListener;
import com.mxgraph.util.mxUndoableEdit.mxUndoableChange;
import com.mxgraph.view.mxGraphView;
import com.mxgraph.view.mxLayoutManager;
import com.mxgraph.view.mxStylesheet;

public class GraphWrapper {

	// static GUI variables
	public static enum LayoutType {
		FASTORGANIC, GRID, TREE, HIERARCHICAL, CIRCLE, PARALLEL_EDGE, PARTITION, STACK, NONE
	};

	public static final int GM_VERTEX_ADDED = 0;
	public static final int GM_VERTEX_REMOVED = 1;
	public static final int GM_EDGE_ADDED = 2;
	public static final int GM_EDGE_REMOVED = 3;

	public static final String STYLE_DEFAULT_NODE = "DEFAULT_NODE";
	public static final String STYLE_TARGET_NODE = "TARGET_NODE";
	public static final String STYLE_INITIAL_NODE = "INITIAL_NODE";
	public static final String STYLE_INITIAL_TARGET_NODE = "INITIAL_TARGET_NODE";
	public static final String STYLE_REDUCED_NODE = "REDUCED_NODE";
	public static final String STYLE_SUBGRAPH_NODE = "SUBGRAPH_NODE";
	public static final String STYLE_SUBGRAPH = "SUBGRAPH";
	public static final String STYLE_DEFAULT_EDGE = "DEFAULT_EDGE";
	public static final String STYLE_REDUCED_EDGE = "REDUCED_EDGE";

	public Color defaultNodeColor = Color.BLUE;
	public Color initialNodeColor = Color.GREEN;
	public Color targetNodeColor = Color.RED;
	public Color reducedNodeColor = Color.MAGENTA;
	public Color subgraphNodeColor = Color.YELLOW;
	private Color rubberbandColor = new Color(0, 0, 1.0f, 0.2f);

	private boolean cellsMovable;
	private boolean otherLayoutBefore;

	private CustomGraph graph;
	private OutputPanel outputPanel;
	protected mxUndoManager undoManager = new mxUndoManager();
	private mxGraphComponent graphComponent = null;
	private GraphMouseListener listener;
	private MarkovChain markovChain;
	private ArrayList<GraphModificationListener> modificationListeners = new ArrayList<GraphModificationListener>();
	private EditMode editMode;
	private mxLayoutManager layoutManager = new mxLayoutManager(graph);

	public GraphWrapper(OutputPanel outputPanel) {
		this.outputPanel = outputPanel;
		this.graph = new CustomGraph();

		editMode = EditMode.NONE;

		graphComponent = new mxGraphComponent(graph);
		graphComponent.setToolTips(true);
		graphComponent.getViewport().setBackground(Color.WHITE);

		initGraph();

		// Make sure that whenever an edge cell is created,
		// an edge object is associated with it
		mxIEventListener connectHandler = new mxIEventListener() {

			public void invoke(Object source, mxEventObject evt) {
				Output.print("woohoo, an edge, we have!");

				mxCell cell = (mxCell) evt.getProperty("cell");
				if (cell.isEdge()) {
					// Set style
					cell.setStyle(STYLE_DEFAULT_EDGE);
					// Notify listeners that an edge was added
					fireGraphModification(GM_EDGE_ADDED, cell);
				}
			}
		};
		graphComponent.getConnectionHandler().addListener(mxEvent.CONNECT, connectHandler);

		listener = new GraphMouseListener();
		mxGraphControl control = graphComponent.getGraphControl();
		control.addMouseListener(listener);
		control.addMouseWheelListener((MouseWheelListener) listener);
	}

	/**
	 * Initialize graph (styles, listener)
	 * 
	 */
	private void initGraph() {
		registerStyles();

		// Notify graph modification listeners when cells were removed
		mxIEventListener removeHandler = new mxIEventListener() {
			public void invoke(Object source, mxEventObject evt) {
				Object[] cells = (Object[]) evt.getProperty("cells");
				for (Object o : cells) {
					mxCell cell = (mxCell) o;
					if (cell.isEdge())
						fireGraphModification(GM_EDGE_REMOVED, cell);
					else
						fireGraphModification(GM_VERTEX_REMOVED, cell);
				}
			}
		};
		graph.addListener(mxEvent.CELLS_REMOVED, removeHandler);

		// Multiple selection
		mxRubberband rubberband = new mxRubberband(graphComponent);
		rubberband.setFillColor(rubberbandColor);
		mxIEventListener selectionHandler = new mxIEventListener() {
			public void invoke(Object sender, mxEventObject evt) {
				updateDisplayedGraph();
			}
		};
		graph.getSelectionModel().addListener(mxEvent.CHANGE, selectionHandler);

		// Moving cells
		mxIEventListener moveHandler = new mxIEventListener() {
			public void invoke(Object sender, mxEventObject evt) {
				updateDisplayedGraph();
			}
		};
		graph.addListener(mxEvent.MOVE_CELLS, moveHandler);

		// NJ: UndoHandling deactivated for the moment
		// initializeUndoHandling();

		// Register the layout
		layoutManager.destroy();
		layoutManager = new mxLayoutManager(graph) {
			mxGraphLayout layout = new mxParallelEdgeLayout(graph);

			public mxIGraphLayout getLayout(Object parent) {
				if (graph.getModel().getChildCount(parent) > 0) {
					// Save actual property
					cellsMovable = graph.isCellsMovable();
					// Make cells movable for layout
					graph.setCellsMovable(true);

					// Remove all EdgePoints
					if (otherLayoutBefore) {
						for (Edge e : markovChain.getEdges()) {
							if (e.getGraphCell() != null) {
								e.getGraphCell().getGeometry().setPoints(null);
							}
						}
					}
					otherLayoutBefore = false;
					return layout;
				}
				return null;
			}
		};

		mxIEventListener layoutHandler = new mxIEventListener() {
			// Restore old property
			public void invoke(Object sender, mxEventObject evt) {
				graph.setCellsMovable(cellsMovable);
			}
		};
		layoutManager.addListener(mxEvent.LAYOUT_CELLS, layoutHandler);

		otherLayoutBefore = true;
	}

	private void fireGraphModification(int modificationType, mxCell cell) {
		for (GraphModificationListener l : modificationListeners) {
			switch (modificationType) {
			case GM_VERTEX_ADDED:
				Output.print("GRAPH MODIFICATION: STATE ADDED");
				l.nodeAdded(cell);
				Node node = (Node) (cell.getValue());
				if (markovChain.getInitialNode() == null) {
					markovChain.setInitialNode(node);
				}
				outputPanel.getTabNodes().addNode(node);
				break;
			case GM_VERTEX_REMOVED:
				Output.print("GRAPH MODIFICATION: STATE REMOVED");
				Node n = (Node) (cell.getValue());
				l.nodeRemoved(cell);
				outputPanel.getTabNodes().removeNode(n);
				break;
			case GM_EDGE_ADDED:
				Output.print("GRAPH MODIFICATION: EDGE ADDED");
				// Remove edge again
				boolean existed = l.edgeAdded(cell);
				Edge e = (Edge) cell.getValue();
				e.setGraphCell(cell);
				// Remove completely, if canceled or existed
				if (e.getProbability() == 0 || existed) {
					if (!existed) {
						// only one edge remaining
						markovChain.edgeRemoved(cell);
					}
					alignSecondEdge(e);
					cell.removeFromParent();
					setLayout(LayoutType.PARALLEL_EDGE);
				}

				// Gui output
				if (e.getProbability() != 0) {
					if (!existed) {
						outputPanel.getTabEdges().addEdge(e);
					} else {
						outputPanel.getTabEdges().changeEdge(markovChain.getSecondEdge(e, true));
					}
				}
				break;
			case GM_EDGE_REMOVED:
				Output.print("GRAPH MODIFICATION: EDGE REMOVED");
				l.edgeRemoved(cell);
				Edge edge = (Edge) cell.getValue();
				alignSecondEdge(edge);
				outputPanel.getTabEdges().removeEdge(edge);
			}
		}
		updateDisplayedGraph();
	}

	/**
	 * Remove edgePoints for second edge to get straight edge again.
	 * 
	 * @param original
	 *            removed edge
	 */
	private void alignSecondEdge(Edge original) {
		otherLayoutBefore = true;
		Edge edgeSameDirection = markovChain.getSecondEdge(original, true);
		Edge edgeOppositeDirection = markovChain.getSecondEdge(original, false);
		if (edgeSameDirection != null) {
			if (edgeOppositeDirection != null) {
				// Both edges exists
				edgeSameDirection.getGraphCell().getGeometry().setPoints(null);
				edgeOppositeDirection.getGraphCell().getGeometry().setPoints(null);
			} else {
				// Only same direction exists
				edgeSameDirection.getGraphCell().getGeometry().setPoints(null);
			}
		}
	}

	public void setEditMode(EditMode editMode) {
		this.editMode = editMode;
		switch (editMode) {
		case NONE:
			setCursor(Cursor.getPredefinedCursor(Cursor.MOVE_CURSOR));
			break;
		case ADD_EDGE:
			setCursor(Cursor.getPredefinedCursor(Cursor.CROSSHAIR_CURSOR));
			break;
		case ADD_NODE:
			setCursor(Cursor.getPredefinedCursor(Cursor.DEFAULT_CURSOR));
		}
	}

	public EditMode getEditMode() {
		return editMode;
	}

	@SuppressWarnings("unused")
	private void initializeUndoHandling() {
		mxIEventListener undoHandler = new mxIEventListener() {
			public void invoke(Object source, mxEventObject evt) {
				undoManager.undoableEditHappened((mxUndoableEdit) evt.getProperty("edit"));
				Output.print("register undo event");
			}
		};
		graph.getModel().addListener(mxEvent.UNDO, undoHandler);
		graph.getView().addListener(mxEvent.UNDO, undoHandler);

		// Keeps the selection in sync with the command history
		undoHandler = new mxIEventListener() {
			public void invoke(Object source, mxEventObject evt) {
				Output.print("execute undo event");

				List<mxUndoableChange> changes = ((mxUndoableEdit) evt.getProperty("edit")).getChanges();
				graph.setSelectionCells(graph.getSelectionCellsForChanges(changes));

				// TODO Tmp (Testing)
				mxUndoableEdit edit = (mxUndoableEdit) evt.getProperty("edit");
				Output.print("Undoable change:");
				for (mxUndoableChange change : edit.getChanges()) {
					Output.print("* " + change.toString());

					if (change instanceof mxChildChange) {
						mxChildChange childChng = (mxChildChange) change;
						mxCell cell = (mxCell) childChng.getChild();
						if (cell.isVertex())
							Output.print("Node change: " + ((Node) cell.getValue()).getNumber());
						else
							Output.print("Edge change: " + ((Edge) cell.getValue()).toString());
					}

				}
			}
		};
		undoManager.addListener(mxEvent.UNDO, undoHandler);
		undoManager.addListener(mxEvent.REDO, undoHandler);

		mxIEventListener addCellHandler = new mxIEventListener() {
			public void invoke(Object source, mxEventObject evt) {
				// When cells are added, the event object is created as follows:
				// fireEvent(new mxEventObject(mxEvent.ADD_CELLS, "cells",
				// cells,
				// "parent", parent, "index", index, "source", source,
				// "target", target)); (s. mxGraph.java)
				/*
				 * Output.print("The following cells were added: "); for (Object
				 * o : (Object[])evt.getProperty("cells")) { mxCell cell =
				 * (mxCell) o; Output.print("* " + cellToString(cell)); }
				 */
			}
		};
		graph.addListener(mxEvent.ADD_CELLS, addCellHandler);
		graph.getModel().addListener(mxEvent.ADD_CELLS, addCellHandler);

		mxIEventListener removeCellHandler = new mxIEventListener() {
			public void invoke(Object source, mxEventObject evt) {
				// When cells are added, the event object is created as follows:
				// fireEvent(new mxEventObject(mxEvent.ADD_CELLS, "cells",
				// cells,
				// "parent", parent, "index", index, "source", source,
				// "target", target)); (s. mxGraph.java)
				Output.print("The following cells were removed: ");
				for (Object o : (Object[]) evt.getProperty("cells")) {
					mxCell cell = (mxCell) o;
					Output.print("* " + cellToString(cell));
				}
			}
		};
		graph.addListener(mxEvent.REMOVE_CELLS, removeCellHandler);
		graph.getModel().addListener(mxEvent.REMOVE_CELLS, removeCellHandler);
	}

	private void registerStyles() {
		mxStylesheet stylesheet = graph.getStylesheet();

		Hashtable<String, Object> style = getBasicNodeStyle(mxConstants.SHAPE_ELLIPSE, defaultNodeColor);
		if (Config.EASTEREGG) {
			// Overwrite default nodes with egg image
			graphComponent.setTripleBuffered(true);
			style = getBasicNodeStyle(mxConstants.SHAPE_IMAGE, defaultNodeColor);
			style.put(mxConstants.STYLE_IMAGE, "/resources/egg.png");
		}
		stylesheet.putCellStyle(STYLE_DEFAULT_NODE, style);

		style = getBasicNodeStyle(mxConstants.SHAPE_RHOMBUS, initialNodeColor);
		stylesheet.putCellStyle(STYLE_INITIAL_NODE, style);

		style = getBasicNodeStyle(mxConstants.SHAPE_DOUBLE_ELLIPSE, targetNodeColor);
		style.put(mxConstants.STYLE_STROKECOLOR, "black");
		style.put(mxConstants.STYLE_STROKEWIDTH, 2);
		stylesheet.putCellStyle(STYLE_TARGET_NODE, style);

		Color initialTargetNodeColor = mixColor(initialNodeColor, targetNodeColor);
		style = getBasicNodeStyle(mxConstants.SHAPE_HEXAGON, initialTargetNodeColor);
		stylesheet.putCellStyle(STYLE_INITIAL_TARGET_NODE, style);

		style = getBasicNodeStyle(mxConstants.SHAPE_RECTANGLE, reducedNodeColor);
		stylesheet.putCellStyle(STYLE_REDUCED_NODE, style);

		style = getBasicNodeStyle(mxConstants.SHAPE_RECTANGLE, subgraphNodeColor);
		stylesheet.putCellStyle(STYLE_SUBGRAPH_NODE, style);

		style = getBasicNodeStyle(mxConstants.SHAPE_RECTANGLE, Color.LIGHT_GRAY);
		style.put(mxConstants.STYLE_OPACITY, 95);
		stylesheet.putCellStyle(STYLE_SUBGRAPH, style);

		style = new Hashtable<String, Object>();
		style.put(mxConstants.STYLE_ENDARROW, mxConstants.ARROW_CLASSIC);
		style.put(mxConstants.STYLE_BENDABLE, "1");
		style.put(mxConstants.STYLE_ROUNDED, "1");
		style.put(mxConstants.STYLE_STROKEWIDTH, "2");
		// TODO: (JG6) Find the attributes for this in the new api
		// GraphConstants.setRouting(edge.getAttributes(),
		// JGraphpadParallelSplineRouter.sharedInstance);
		// GraphConstants.setLineStyle(edge.getAttributes(),
		// GraphConstants.STYLE_SPLINE);
		// GraphConstants.setSelectable(edge.getAttributes(), true);
		stylesheet.putCellStyle(STYLE_DEFAULT_EDGE, style);

		style = new Hashtable<String, Object>();
		style.put(mxConstants.STYLE_ENDARROW, mxConstants.ARROW_CLASSIC);
		style.put(mxConstants.STYLE_BENDABLE, "1");
		style.put(mxConstants.STYLE_GRADIENTCOLOR, colorToHexRepresentation(Color.PINK));
		// TODO: (JG6) Find the attributes for this in the new api
		// GraphConstants.setRouting(edge.getAttributes(),
		// GraphConstants.ROUTING_DEFAULT);
		// GraphConstants.setEndFill(edge.getAttributes(), true);
		stylesheet.putCellStyle(STYLE_REDUCED_EDGE, style);
	}

	private Hashtable<String, Object> getBasicNodeStyle(String shape, Color color) {
		Hashtable<String, Object> style = new Hashtable<String, Object>();
		style.put(mxConstants.STYLE_SHAPE, shape);
		style.put(mxConstants.STYLE_OPACITY, 50);
		style.put(mxConstants.STYLE_GRADIENTCOLOR, colorToHexRepresentation(color));
		style.put(mxConstants.STYLE_RESIZABLE, 0);
		style.put(mxConstants.STYLE_CLONEABLE, 0);
		return style;
	}

	public mxGraphComponent getGraphComponent() {
		return graphComponent;
	}

	public mxUndoManager getUndoManager() {
		return undoManager;
	}

	public GraphMouseListener getGraphMouseListener() {
		return listener;
	}

	public void setMarkovChain(MarkovChain mc) {
		// There might already be a graph displayed ->
		// clear that first
		clearGraph();

		this.markovChain = mc;

		// There is up to one listener from the previously set MC
		// This listener has to be removed, of course!
		for (int i = 0; i < modificationListeners.size(); i++) {
			if (modificationListeners.get(i) instanceof MarkovChain) {
				modificationListeners.remove(i);
				break;
			}
		}
		// Add the modification listener of the new MC
		modificationListeners.add(mc);

		graph.getModel().beginUpdate();
		if (mc.isReduced()) {
			drawReducedMarkovChain();
		} else {
			drawMarkovChain();
		}
		graph.getModel().endUpdate();
		setLayout(LayoutType.PARALLEL_EDGE);
		centerInitial();
		updateDisplayedGraph();
	}

	public Node getSelectedNode() {
		if (isSelectionEmpty())
			return null;

		mxCell selection = (mxCell) graph.getSelectionCell();
		if (selection.isVertex())
			return (Node) selection.getValue();
		else
			return null;
	}

	public Edge getSelectedEdge() {
		if (isSelectionEmpty())
			return null;

		mxCell selection = (mxCell) graph.getSelectionCell();
		if (selection.isEdge())
			return (Edge) selection.getValue();
		else
			return null;
	}

	public void clearSelection() {
		graph.clearSelection();
	}

	/**
	 * Select one node
	 * 
	 * @param node
	 *            Node
	 */
	public void setSelectedNode(Node node) {
		setSelectedNode(node, false);
	}

	/**
	 * Select node(s)
	 * 
	 * @param node
	 *            Node
	 * @param multipleSelect
	 *            only one node is selected if false
	 */
	public void setSelectedNode(Node node, boolean multipleSelect) {
		if (node.getGraphCell() == null)
			return;
		if (multipleSelect)
			graph.addSelectionCell(node.getGraphCell());
		else
			graph.setSelectionCell(node.getGraphCell());
	}

	public void setSelectedEdge(Edge edge) {
		if (edge.getGraphCell() == null)
			return;
		graph.setSelectionCell(edge.getGraphCell());
	}

	public boolean isSelectionEmpty() {
		return graph.isSelectionEmpty();
	}

	public boolean isGraphEmpty() {
		return graph.getModel().getChildCount(graph.getDefaultParent()) == 0;
	}

	private void drawMarkovChain() {
		int[] grid = { 0, 0 };
		double[] nodePositions;

		graph.getModel().beginUpdate();
		for (Node s : markovChain.getNodes()) {
			if (s.isPositionSet()) {
				nodePositions = new double[] { s.getXpos() - 40, s.getYpos() - 40 };
			} else {
				nodePositions = getGridPos(grid);
			}
			addNodeToGraph(s, nodePositions);
		}

		// Note: We always layout the edges now, as there seem to be no other
		// function calls anyway
		for (Edge edge : markovChain.getEdges()) {
			addEdgeToGraph(edge);
		}

		graph.getModel().endUpdate();
	}

	/**
	 * Draw first level of a reduced Markov Chain
	 */
	private void drawReducedMarkovChain() {
		int[] grid = { 0, 0 };
		double[] nodePos;

		// Add input nodes
		for (Node s : markovChain.getInputNodes()) {
			markovChain.setReducedNode(s, true);
			nodePos = getGridPos(grid);
			addNodeToGraph(s, nodePos);
		}

		// Add target nodes
		for (Node s : markovChain.getTargetNodes()) {
			nodePos = getGridPos(grid);
			addNodeToGraph(s, nodePos);
		}

		// Add concrete edges, if there are no reduced edges
		if (markovChain.getReducedEdges().size() == 0) {
			Node inputNode = markovChain.getInputNodes().first();
			expandSCC(inputNode);
			markovChain.setInputOfSccNode(inputNode, false);
			// MarkovChain is not expanded, because it is root chain
			markovChain.setExpandedInGraph(false);
		} else {
			// Add reduced edges
			for (Edge edge : markovChain.getReducedEdges()) {
				addEdgeToGraph(edge);
			}
		}
	}

	/**
	 * Returns grid values for given node position
	 * 
	 * @param currentPos
	 *            node position as reference
	 * @return grid as [i, j]
	 */
	private int[] getGrid(double[] currentPos) {
		if (currentPos.length < 2)
			return null;

		int i = (int) (currentPos[0] / 100);
		int j = (int) (currentPos[1] / 100);
		int[] grid = { i, j };
		return grid;
	}

	/**
	 * Returns next position for node on grid layout
	 * 
	 * @param grid
	 *            current positions on grid layout as [i, j]. These positions
	 *            are updated in the function
	 * @return node position as [xpos, ypos]
	 */
	private double[] getGridPos(int[] grid) {
		if (grid.length < 2)
			return null;

		int i = grid[0];
		int j = grid[1];
		int gridWidth = (int) Math.sqrt(markovChain.getNumberOfNodes());

		double[] pos = { i * 100, j * 100 };

		// Note: We always layout the nodes now, as there seem to be no
		// other function calls anyway
		i++;
		if (i % gridWidth == 0) {
			j++;
			i = 0;
		}
		grid[0] = i;
		grid[1] = j;
		return pos;
	}

	// TODO unused
	public void redrawMarkovChain() {
		clearGraph();
		graph.getModel().beginUpdate();

		if (markovChain.isReduced()) {
			drawReducedMarkovChain();
		} else {
			drawMarkovChain();
		}
		graph.getModel().endUpdate();
	}

	private void setCursor(Cursor cursor) {
		mxGraphHandler.DEFAULT_CURSOR = cursor;
	}

	/**
	 * To be called with new (Markov chain) nodes so that they appear in the
	 * visual graph as well
	 * 
	 * @param node
	 *            Node to be displayed
	 */
	public void addNodeToGraph(Node node) {
		if (node.isPositionSet())
			addNodeToGraph(node, node.getXpos(), node.getYpos());
		else {
			addNodeToGraph(node, 100, 100);
		}
	}

	/**
	 * Add node to graph
	 * 
	 * @param node
	 *            Node
	 * @param pos
	 *            node position as [xpos, ypos]
	 */
	public void addNodeToGraph(Node node, double[] pos) {
		if (pos.length < 2)
			return;
		addNodeToGraph(node, pos[0], pos[1]);
	}

	public void addNodeToGraph(Node node, double xpos, double ypos) {
		addNodeToGraph(node, graph.getDefaultParent(), xpos, ypos);
	}

	public void addNodeToGraph(Node node, Object parent, double xpos, double ypos) {
		if (node.isPositionSet()) {
			xpos = node.getXpos();
			ypos = node.getYpos();
		}
		if (graph.isEnabled()) {
			// Only display if enabled
			node.setGraphCell((mxCell) graph.insertVertex(parent, null, node, xpos, ypos, 40, 40));
			node.adjustStyle();
		}
		outputPanel.getTabNodes().addNode(node);
		outputPanel.getTabTargetNodes().addNode(node);
	}

	public mxCell addNodeCell(double x, double y) {
		mxCell result = (mxCell) graph.insertVertex(graph.getDefaultParent(), null, null, x, y, 40, 40, STYLE_DEFAULT_NODE);
		fireGraphModification(GM_VERTEX_ADDED, result);
		return result;
	}

	/**
	 * Add new edge to graph
	 * 
	 * @param edge
	 *            edge to be displayed
	 */
	public void addEdgeToGraph(Edge edge) {
		addEdgeToGraph(edge, graph.getDefaultParent());
	}

	/**
	 * To be called with new (Markov chain) edges so that they appear in the
	 * visual graph as well
	 * 
	 * @param edge
	 *            Edge to be displayed
	 * @param parent
	 *            parent cell object
	 */
	public void addEdgeToGraph(Edge edge, Object parent) {
		if (graph.isEnabled()) {
			// Only display if enabled
			edge.setGraphCell((mxCell) graph.insertEdge(parent, null, edge, edge.getSourceNode().getGraphCell(), edge
					.getTargetNode().getGraphCell(), edge.isReduced() ? STYLE_REDUCED_EDGE : STYLE_DEFAULT_EDGE));
		}
		outputPanel.getTabEdges().addEdge(edge);
	}

	public mxCell addEdgeCell(mxCell src, mxCell trg) {
		mxCell result = (mxCell) graph.insertEdge(graph.getDefaultParent(), null, null, src, trg, STYLE_DEFAULT_EDGE);
		result.setStyle(STYLE_DEFAULT_EDGE);
		fireGraphModification(GM_EDGE_ADDED, result);
		return result;
	}

	/**
	 * Hide node by making invisible
	 * 
	 * @param node
	 *            node to hide
	 */
	private void hideNode(Node node) {
		if (node == null || node.isHidden() || node.getGraphCell() == null)
			return;

		for (Edge e : markovChain.getConnections(node)) {
			hideEdge(e);
		}

		if (graph.isEnabled()) {
			// Only display if enabled
			node.getGraphCell().setVisible(false);
		}
		node.setHidden(true);
		outputPanel.getTabNodes().removeNode(node);
	}

	/**
	 * Unhide node, which was visible eventually before
	 * 
	 * @param node
	 *            node to unhide
	 */
	private void unHideNode(Node node) {
		if (node == null || !node.isHidden() || node.getGraphCell() == null)
			return;
		if (graph.isEnabled()) {
			// Only display if enabled
			node.getGraphCell().setVisible(true);
		}
		node.setHidden(false);
		outputPanel.getTabNodes().addNode(node);
	}

	/**
	 * Set nodes (in)visible to hide them
	 * 
	 * @param nodes
	 *            array of nodes
	 * @param visible
	 *            if true, nodes are made invisible
	 */
	public void setNodesVisible(Node[] nodes, boolean visible) {
		for (Node n : nodes) {
			if (n.getGraphCell() != null) {
				if (graph.isEnabled()) {
					// Only display if enabled
					n.getGraphCell().setVisible(visible);
				}
				n.setHidden(!visible);
			}
		}
		updateDisplayedGraph();
	}

	/**
	 * Hide edge by making invisible
	 * 
	 * @param edge
	 *            edge to hide
	 */
	private void hideEdge(Edge edge) {
		if (edge == null || edge.isHidden() || edge.getGraphCell() == null)
			return;

		removeCell(edge.getGraphCell(), false);
		edge.setHidden(true);
		outputPanel.getTabEdges().removeEdge(edge);
	}

	/**
	 * Unhide edge, which was visible eventually before
	 * 
	 * @param edge
	 *            edge to unhide
	 */
	private void unHideEdge(Edge edge) {
		if (edge == null || !edge.isHidden() || edge.getGraphCell() == null)
			return;

		addEdgeToGraph(edge);
		edge.setHidden(false);
		outputPanel.getTabEdges().addEdge(edge);
	}

	public void removeEdge(Edge edge) {
		removeCell(edge.getGraphCell());
	}

	public void removeNode(Node node) {
		removeCell(node.getGraphCell());
	}

	/**
	 * Remove Cell. fireGraphModification is called implicitly.
	 * 
	 * @param cell
	 */
	public void removeCell(mxCell cell) {
		removeCell(cell, true);
	}

	/**
	 * Remove cell and event could be fired
	 * 
	 * @param cell
	 *            cell to remove
	 * @param fireEvent
	 *            if true, event is fired
	 */
	public void removeCell(mxCell cell, boolean fireEvent) {
		if (graph.isEnabled()) {
			// Only display if enabled
			removeCells(new mxCell[] { cell }, fireEvent);
		}
	}

	/**
	 * Remove cells and event could be fired
	 * 
	 * @param cells
	 *            array of cells to remove
	 * @param fireEvent
	 *            if true, event is fired
	 */
	public void removeCells(mxCell[] cells, boolean fireEvent) {
		graph.setSelectionCells(cells);

		// Delete could be disabled because of model checking, but nodes should
		// only be hidden
		boolean deletable = graph.isCellsDeletable();
		if (!deletable)
			graph.setCellsDeletable(true);

		if (!fireEvent)
			// Disable event
			graph.setEventsEnabled(false);

		// Remove from graph
		if (graph.isEnabled()) {
			// Only display if enabled
			graph.removeCells();
		}
		// Restore default
		if (!fireEvent)
			graph.setEventsEnabled(true);
		if (!deletable)
			graph.setCellsDeletable(false);
	}

	public void sourroundNodesWithRectangle(Node[] nodes) {

		ArrayList<mxCell> cellList = new ArrayList<mxCell>();
		for (Node n : nodes) {
			if (n.getGraphCell() != null)
				cellList.add(n.getGraphCell());
		}
		mxCell[] cells = cellList.toArray(new mxCell[0]);

		graph.groupCells(null, 2, cells);

		// TODO: Complete?!
		// TODO: Anonymous group or explicit parent?

		// TODO: Do we need to do this explicitly?
		updateDisplayedGraph();
	}

	public void sourroundNodeWithRectangle(Node node) {
		sourroundNodesWithRectangle(new Node[] { node });
	}

	/**
	 * Clean graph display (no impact on MC!)
	 */
	public void clearGraph() {
		// Removing all cells has bad performance, so a new graph is created
		// instead
		graph = new CustomGraph();
		graphComponent.setGraph(graph);
		initGraph();
		graphComponent.refresh();
	}

	/**
	 * Deletes all currently selected nodes both from graph and Markov Chain
	 */
	public void deleteSelectedNodes() {
		if (!graph.isSelectionEmpty()) {
			for (Object obj : graph.getSelectionCells()) {
				mxCell cell = (mxCell) obj;

				if (cell.isVertex()) {
					for (Edge e : markovChain.getConnections((Node) cell.getValue())) {
						removeEdge(e);
					}
					removeCell(cell);
				} else if (cell.isEdge()) {
					removeCell(cell);
				}
			}

			// Remove selection including edges
			graph.removeCells(graph.getSelectionCells(), true);
		}
	}

	/**
	 * Reduce scc to only show only one abstract node for complete underlying
	 * scc
	 * 
	 * @param inputNode
	 *            input node for scc
	 * 
	 */
	public void reduceSCC(Node inputNode) {
		if (inputNode == null || inputNode.isReduced()) {
			return;
		}
		reduceSCC(inputNode.getCorrespondingMarkovChain());

		updateDisplayedGraph();

		// Hide absorbing nodes
		if (SCC_MC.getInstance().getGui().isHideAbsorbingNodes()) {
			SCC_MC.getInstance().getGui().hideAbsorbingNodes(true);
		}
	}

	/**
	 * Reduce scc to only show only one abstract node for complete underlying
	 * SCCs
	 * 
	 * @param mc
	 *            MarkovChain to reduce
	 */
	private void reduceSCC(MarkovChain mc) {
		if (mc.getInitialNode() != null) {
			mc.setReducedNode(mc.getInitialNode(), true);
		}
		for (Node n : mc.getInputNodes()) {
			mc.setReducedNode(n, true);
		}

		mc.setExpandedInGraph(false);

		// remove concrete edges
		for (Edge e : mc.getEdges()) {
			hideEdge(e);
		}

		// reduce subgraphs, remove reduced edges in subgraphs and input nodes
		// of these subgraphs
		for (MarkovChain subgraph : mc.getSubgraphs()) {
			reduceSCC(subgraph);
			for (Edge edge : subgraph.getReducedEdges()) {
				hideEdge(edge);
			}
			for (Node subgraphInput : subgraph.getInputNodes()) {
				hideNode(subgraphInput);
			}
		}

		// add reduced edges again
		for (Edge e : mc.getReducedEdges()) {
			unHideEdge(e);
		}

		// all nodes with no edges can be hidden as well
		for (Node n : mc.getNodes()) {
			if (n.getGraphCell() != null && n.getGraphCell().getEdgeCount() == 0) {
				hideNode(n);
			}
		}
	}

	/**
	 * Expand SCC to show concrete nodes and edges beneath
	 * 
	 * @param inputNode
	 *            input node for scc
	 */
	public void expandSCC(Node inputNode) {
		if (inputNode == null || !inputNode.isReduced()) {
			return;
		}

		// Check for too many edges in expanded graph
		boolean showGraph = true;
		int numberEdges = inputNode.getCorrespondingMarkovChain().getNumberOfEdges()
				- inputNode.getCorrespondingMarkovChain().getNumberOfReducedEdges();
		for (MarkovChain subgraph : inputNode.getCorrespondingMarkovChain().getSubgraphs()) {
			numberEdges += subgraph.getNumberOfReducedEdges();
		}

		if (numberEdges > Config.edgeCountVisible) {
			// Ask, if graph should be displayed
			String question = "The expanded graph would have " + numberEdges + " additional edges." + Output.getLineBreak()
					+ "This might cause problems while displaying." + Output.getLineBreak()
					+ "Would you like to display the graph anyhow?";
			showGraph = GUI.showYesNoQuestion("Show graph?", question);
		}
		if (!showGraph) {
			// Don't display graph
			clearGraph();
			setGraphEnabled(false);
		}

		inputNode.getCorrespondingMarkovChain().setReducedNode(inputNode, false);
		inputNode.getCorrespondingMarkovChain().setExpandedInGraph(true);

		// remove reduced edges
		for (Edge e : inputNode.getCorrespondingMarkovChain().getReducedEdges()) {
			hideEdge(e);
		}

		double[] pos = { 0, 0 };
		if (showGraph) {
			pos[0] = inputNode.getGraphCell().getGeometry().getX();
			pos[1] = inputNode.getGraphCell().getGeometry().getY();
		}
		int[] grid = getGrid(pos);
		// Expansion starts in row below
		grid[1]++;
		// draw input nodes of subgraphs and edges from these input nodes to
		// target states
		for (MarkovChain subgraph : inputNode.getCorrespondingMarkovChain().getSubgraphs()) {
			for (Node subgraphInput : subgraph.getInputNodes()) {
				subgraph.setReducedNode(subgraphInput, true);
				if (subgraphInput.isHidden()) {
					unHideNode(subgraphInput);
				} else {
					double[] nodePos = getGridPos(grid);
					addNodeToGraph(subgraphInput, nodePos);
				}
			}
			for (Edge e : subgraph.getReducedEdges()) {
				drawEdge(e, grid);
			}
		}

		// draw concrete edges
		for (Edge e : inputNode.getCorrespondingMarkovChain().getEdges()) {
			drawEdge(e, grid);
		}

		updateDisplayedGraph();

		// Hide absorbing nodes
		if (SCC_MC.getInstance().getGui().isHideAbsorbingNodes()) {
			SCC_MC.getInstance().getGui().hideAbsorbingNodes(true);
		}
	}

	public int[] getConcretizedGraphs() {
		LinkedList<MarkovChain> mcs = new LinkedList<MarkovChain>();
		ArrayList<Integer> result = new ArrayList<Integer>();
		mcs.add(markovChain);
		while (!mcs.isEmpty()) {
			MarkovChain mc = mcs.pollFirst();
			mcs.addAll(mc.getSubgraphs());
			if (mc.isExpandedInGraph()) {
				result.add(mc.getId());
			}
		}
		int[] resultArray = new int[result.size()];
		for (int i = 0; i < result.size(); i++) {
			resultArray[i] = result.get(i);
		}
		return resultArray;

	}

	/**
	 * Draw edge on graph and draw source and target node as well, if they are
	 * not visible
	 * 
	 * @param e
	 *            Edge to draw
	 * @param grid
	 *            Current grid for node positioning
	 */
	private void drawEdge(Edge e, int[] grid) {
		double[] nodePos;
		// draw source node if not visible
		Node n = e.getSourceNode();
		if (n.isHidden()) {
			unHideNode(n);
		} else if (n.getGraphCell() == null) {
			nodePos = getGridPos(grid);
			addNodeToGraph(n, nodePos);
		}

		// draw target node if not visible
		n = e.getTargetNode();
		if (n.isHidden()) {
			unHideNode(n);
		} else if (n.getGraphCell() == null) {
			nodePos = getGridPos(grid);
			addNodeToGraph(n, nodePos);
		}
		if (e.isHidden()) {
			unHideEdge(e);
		} else {
			addEdgeToGraph(e);
		}
	}

	public void changeProbability(Edge edge) {
		double probability = 0;
		probability = GUI.getProbability("Please enter the probability", edge.getProbability());
		if (probability != 0) {
			try {
				markovChain.setProbabilityEdge(edge, probability);
			} catch (ProbabilityExceededException e) {
				Output.print(e.toString());
			}
			graphComponent.refresh();
			outputPanel.getTabEdges().changeEdge(edge);
		}
	}

	public void setInitialNodeColor(Color c) {
		initialNodeColor = c;
		registerStyles();
		updateDisplayedGraph();
	}

	public void setTargetNodeColor(Color c) {
		targetNodeColor = c;
		registerStyles();
		updateDisplayedGraph();
	}

	public void setDefaultNodeColor(Color c) {
		defaultNodeColor = c;
		registerStyles();
		updateDisplayedGraph();
	}

	public void setReducedNodeColor(Color c) {
		reducedNodeColor = c;
		registerStyles();
		updateDisplayedGraph();
	}

	public Color getInitialNodeColor() {
		return initialNodeColor;
	}

	public Color getTargetNodeColor() {
		return targetNodeColor;
	}

	public Color getDefaultNodeColor() {
		return defaultNodeColor;
	}

	public Color getReducedNodeColor() {
		return reducedNodeColor;
	}

	/**
	 * Returns means of two ints
	 * 
	 * @param n1
	 *            number 1
	 * @param n2
	 *            number 2
	 * @return mean of n1 and n2
	 */
	private int getMean(int n1, int n2) {
		return (n1 + n2) / 2;
	}

	/**
	 * Mix two colors to get a composition of these colors
	 * 
	 * @param color1
	 *            color 1
	 * @param color2
	 *            color 2
	 * @return Mixed color
	 */
	private Color mixColor(Color color1, Color color2) {
		int red = getMean(color1.getRed(), color2.getRed());
		int green = getMean(color1.getGreen(), color2.getGreen());
		int blue = getMean(color1.getBlue(), color2.getBlue());
		return new Color(red, green, blue);
	}

	/**
	 * Rescale view by factor
	 * 
	 * @param factor
	 *            rescale factor
	 */
	public void rescaleByFactor(double factor) {
		mxGraphView view = graph.getView();
		view.setScale(view.getScale() * factor);
	}

	public void rescaleToAbsoluteValue(double scale) {
		mxGraphView view = graph.getView();
		view.setScale((double) scale / 100d);
	}

	/**
	 * Set layout for graph
	 * 
	 * @param layoutType
	 *            see top of class for layoutTypes
	 */
	public void setLayout(LayoutType layoutType) {
		// Instantiate given layout type
		// Note: Must be final, must be if (instead of switch)
		// Otherwise the anonymous instantiation below fails
		final mxGraphLayout newLayout;

		switch (layoutType) {
		case GRID:
			redrawMarkovChain();
			return;

		case FASTORGANIC:
			mxFastOrganicLayout layout = new mxFastOrganicLayout(graph);

			// The force constant by which the attractive forces are divided and
			// the replusive forces are multiple by the square of. The value
			// equates to the average radius there is of free space around each
			// node. Default is 50.
			layout.setForceConstant(200);
			// Start value of temperature. Default is 200.
			layout.setInitialTemp(200);
			// Minimal distance limit. Default is 2. Prevents of dividing by
			// zero.
			layout.setMinDistanceLimit(10);
			// The maximum distance between vertices, beyond which their
			// repulsion no longer has an effect
			layout.setMaxDistanceLimit(500);

			newLayout = layout;
			break;
		case TREE:
			newLayout = new mxCompactTreeLayout(graph);
			break;
		case CIRCLE:
			newLayout = new mxCircleLayout(graph);
			break;
		case PARALLEL_EDGE:
			newLayout = new mxParallelEdgeLayout(graph);
			break;
		case PARTITION:
			newLayout = new mxPartitionLayout(graph);
			break;
		case STACK:
			newLayout = new mxStackLayout(graph);
			break;
		case HIERARCHICAL:
			newLayout = new mxHierarchicalLayout(graph);
			break;
		case NONE: // fall through!
		default:
			newLayout = null;
		}

		// Save actual property
		boolean movableCells = graph.isCellsMovable();
		// Make cells movable for layout
		graph.setCellsMovable(true);

		// Remove all EdgePoints
		for (Edge e : markovChain.getEdges()) {
			if (e.getGraphCell() != null) {
				e.getGraphCell().getGeometry().setPoints(null);
			}
		}

		otherLayoutBefore = true;

		// Excute new layout
		newLayout.execute(graph.getDefaultParent());

		graph.setCellsMovable(movableCells);

		// Without redrawin, nothin to be seen
		// redrawMarkovChain();
		updateDisplayedGraph();
	}

	/**
	 * Lock graph, so that nothing can be changed or edited. Only moving of
	 * cells is allowed
	 * 
	 * @param locked
	 *            if true, graph is locked
	 */
	public void lockGraph(boolean locked) {
		graph.setCellsDeletable(!locked);
	}

	/**
	 * Enable or disable graph, so that nothing is displayed
	 * 
	 * @param enabled
	 *            if false, graph is not displayed
	 */
	public void setGraphEnabled(boolean enabled) {
		if (!enabled) {
			clearGraph();
		}
		lockGraph(!enabled);
		graph.setEnabled(enabled);
		if (!SCC_MC.getInstance().isGraphLocked()) {
			SCC_MC.getInstance().getGui().getToolbar().getToolbarEdit().makeEnabled(enabled);
		}
		if (enabled) {
			graphComponent.getViewport().setBackground(Color.WHITE);
		} else {
			graphComponent.getViewport().setBackground(Color.GRAY);
		}
	}

	public void beginUpdate() {
		graph.getModel().beginUpdate();
	}

	public void endUpdate() {
		graph.getModel().endUpdate();
	}

	public void updateDisplayedGraph() {
		graph.refresh();
	}

	/**
	 * (Un)hide isolated nodes with no edge
	 * 
	 * @param hide
	 *            if true, nodes should be hidden
	 */
	public void hideIsolatedNodes(boolean hide) {
		ArrayList<Node> isolatedNodes = new ArrayList<Node>();
		for (Node n : markovChain.getNodes()) {
			if (!n.hasAdjacentNodes()) {
				isolatedNodes.add(n);
			}
		}
		setNodesVisible(isolatedNodes.toArray(new Node[0]), !hide);
	}

	/**
	 * (Un)hide absorbing nodes which are not targets
	 * 
	 * @param hide
	 *            if true, nodes should be hidden
	 */
	public void hideAbsorbingNodes(boolean hide) {
		ArrayList<Node> absorbingNodes = new ArrayList<Node>();
		for (Node n : markovChain.getNodes()) {
			if (n.isAbsorbing() && !n.isTarget()) {
				absorbingNodes.add(n);
			}
		}
		setNodesVisible(absorbingNodes.toArray(new Node[0]), !hide);
	}

	public void requestFocusForGraph() {
		graphComponent.requestFocus();
	}

	/**
	 * Center view to initial node
	 * 
	 */
	public void centerInitial() {
		centerNode(markovChain.getInitialNode());
	}

	/**
	 * Center view to given node and select it
	 * 
	 * @param n
	 *            node
	 */
	public void centerNode(Node n) {
		if (n != null) {
			centerCell(n.getGraphCell());
			setSelectedNode(n);
		}
	}

	/**
	 * Center view to given edge and select it
	 * 
	 * @param e
	 *            edge
	 */
	public void centerEdge(Edge e) {
		if (e != null) {
			centerCell(e.getGraphCell());
			setSelectedEdge(e);
		}
	}

	/**
	 * Center view to given cell
	 * 
	 * @param cell
	 *            cell
	 */
	public void centerCell(mxCell cell) {
		graphComponent.scrollCellToVisible(cell, true);
	}

	/**
	 * Center view to given point
	 * 
	 * @param x
	 *            x-value
	 * @param y
	 *            y-value
	 */
	public mxCell getCellUnderMouse(MouseEvent event) {
		mxPoint p = new mxPoint(event.getX(), event.getY());
		mxCell cell = (mxCell) graphComponent.getCellAt((int) p.getX(), (int) p.getY());
		return cell;
	}

	/*
	 * Helper methods that don't necessarily belong here...
	 */
	private static String colorToHexRepresentation(Color c) {
		String color = Integer.toHexString(c.getRGB());
		if (color.length() > 6)
			color = color.substring(color.length() - 6);
		while (color.length() < 6)
			color = '0' + color;
		return '#' + color;
	}

	/**
	 * Just a convenience method for debug output
	 * 
	 * @param cell
	 *            Cell to be converted to meaningful string
	 * @return Hopefully meaningful string
	 */
	private static String cellToString(mxCell cell) {
		if (cell != null) {
			if (cell.isVertex()) {
				Node n = (Node) cell.getValue();
				return "Vtx: " + n.getNumber();
			} else {
				return "Edg: " + cell.getValue().toString();
			}
		}

		return "null";
	}
}
