/**
 *   COMICS - Computing Minimal Counterexamples for Discrete-time Markov Chains
 *
 *   COMICS is a stand-alone tool which performs model checking and the generation
 *   of counterexamples for discrete-time Markov Chains (DTMCs). *
 *
 *   Copyright (C) <2012> <RWTH Aachen University>
 *   Authors: Nils Jansen, Erika Abraham, Jens Katelaan, Maik Scheffler, Matthias Volk, Andreas Vorpahl
 *
 *   This program is free software: you can redistribute it and/or modify
 *   it under the terms of the GNU General Public License as published by
 *   the Free Software Foundation, either version 3 of the License, or
 *   (at your option) any later version.
 *
 *   This program is distributed in the hope that it will be useful,
 *   but WITHOUT ANY WARRANTY; without even the implied warranty of
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *   GNU General Public License for more details.
 *
 *   You should have received a copy of the GNU General Public License
 *   along with this program.  If not, see <http://www.gnu.org/licenses/>.
 *
 *   Main Contact:
 *
 *   Nils Jansen
 *   Theory of Hybrid Systems
 *   RWTH Aachen
 *   52056 Aachen
 *   Germany
 *   nils.jansen@cs.rwth-aachen.de
 *
 */

package comics.gui;

import java.awt.BorderLayout;
import java.awt.Color;
import java.awt.Container;
import java.awt.event.ActionListener;
import java.awt.event.WindowAdapter;
import java.awt.event.WindowEvent;
import java.util.Scanner;

import javax.swing.ImageIcon;
import javax.swing.JFrame;
import javax.swing.JLabel;
import javax.swing.JList;
import javax.swing.JMenu;
import javax.swing.JMenuBar;
import javax.swing.JMenuItem;
import javax.swing.JOptionPane;
import javax.swing.JPanel;
import javax.swing.JSplitPane;

import comics.SCC_MC;
import comics.data.Config;
import comics.graph.data.Labels;
import comics.graph.data.MarkovChain;
import comics.graph.gui.GraphWrapper;
import comics.gui.actions.FileMenuActionListener;
import comics.gui.actions.GraphLayoutAction;
import comics.gui.actions.NodeColorAction;
import comics.gui.actions.ToolMenuActionListener;
import comics.gui.actions.ZoomAbsoluteAction;
import comics.gui.actions.ZoomRelativeAction;
import comics.gui.output_panel.OutputPanel;

import comics.utilities.Output;

public class GUI extends JPanel {
	private static final long serialVersionUID = 1L;

	private boolean hideIsolatedNodes;
	private boolean hideAbsorbingNodes;

	// swing variables
	private static JFrame frame;
	private JMenu graphMenu;
	private JMenu colorMenu;
	private JMenu layoutMenu;
	private JMenu zoomMenu;
	private JMenu fileMenu;
	private JMenu toolMenu;
	private JMenu infoMenu;
	private JMenuItem loadLabItem;
	private JMenuItem hideIsolatedNodesItem;
	private JMenuItem hideAbsorbingNodesItem;
	private JMenuBar menuBar;
	private Statusbar statusBar;
	private Toolbar toolbar;
	private OutputPanel outputPanel;
	private ContextMenu contextMenu;
	public GraphWrapper wrapper;

	private SCC_MC scc_mc;

	public GUI(SCC_MC scc_mc) {
		this.scc_mc = scc_mc;

		init();
	}

	public static JFrame getFrame() {
		return frame;
	}

	private void init() {
		System.setProperty("sun.java2d.d3d", "false");

		frame = new JFrame("COMICS 1.0");
		statusBar = new Statusbar(1);
		outputPanel = new OutputPanel();
		contextMenu = new ContextMenu();

		initGraphWrapper();
		initMenus();

		toolbar = new Toolbar(this);

		// Actually fill & display the frame
		frame.setDefaultCloseOperation(JFrame.EXIT_ON_CLOSE);
		frame.setSize(800, 600);
		frame.setBackground(Color.white);
		frame.addWindowListener(new WindowAdapter() {
			public void windowClosing(WindowEvent e) {
				scc_mc.terminate();
			}
		});
		frame.setLocationRelativeTo(null);

		Container contentPane = frame.getContentPane();

		contentPane.setLayout(new BorderLayout());
		contentPane.add(statusBar, BorderLayout.SOUTH);
		contentPane.add(toolbar, BorderLayout.PAGE_START);
		contentPane.add(wrapper.getGraphComponent(), BorderLayout.CENTER);
		contentPane.add(outputPanel, BorderLayout.EAST);

		JSplitPane splitPane = new JSplitPane(JSplitPane.HORIZONTAL_SPLIT, wrapper.getGraphComponent(), outputPanel);
		splitPane.setOneTouchExpandable(true);
		splitPane.setDividerLocation(500);

		contentPane.add(splitPane, BorderLayout.CENTER);
		frame.setVisible(true);
	}

	private void initGraphWrapper() {
		wrapper = new GraphWrapper(outputPanel);
	}

	private void initMenus() {
		menuBar = new JMenuBar();
		frame.setJMenuBar(menuBar);

		addFileMenu();
		addToolMenu();
		addGraphMenu();
		addZoomMenu();
		addInfoMenu();
	}

	private void addInfoMenu() {
		infoMenu = new JMenu("Help");

		ActionListener listener = new FileMenuActionListener(scc_mc);

		JMenuItem item = new JMenuItem("About");
		item.setActionCommand("about");
		item.addActionListener(listener);
		infoMenu.add(item);

		item = new JMenuItem("Go to webpage");
		item.setActionCommand("link");
		item.addActionListener(listener);
		infoMenu.add(item);

		item = new JMenuItem("Open manual");
		item.setActionCommand("manual");
		item.addActionListener(listener);
		infoMenu.add(item);

		menuBar.add(infoMenu);
	}

	private void addFileMenu() {
		fileMenu = new JMenu("File");

		// -----------------
		// File menu actions
		// -----------------
		ActionListener listener = new FileMenuActionListener(scc_mc);

		// Reset
		JMenuItem item = new JMenuItem("Reset");
		item.setActionCommand("reset");
		item.addActionListener(listener);
		fileMenu.add(item);

		// Open file
		item = new JMenuItem("Open file");
		item.setActionCommand("open_file");
		item.addActionListener(listener);
		fileMenu.add(item);

		// Open Prism
		// item = new JMenuItem("Open Prism");
		// item.setActionCommand("open_prism");
		// item.addActionListener(listener);
		// fileMenu.add(item);

		// Open lab file
		loadLabItem = new JMenuItem("Open labels file");
		loadLabItem.setActionCommand("open_labfile");
		loadLabItem.addActionListener(listener);
		loadLabItem.setEnabled(false);
		fileMenu.add(loadLabItem);

		// Save file
		item = new JMenuItem("Save");
		item.setActionCommand("save");
		item.addActionListener(listener);
		fileMenu.add(item);

		// Save as
		item = new JMenuItem("Save as XML");
		item.setActionCommand("save_as_xml");
		item.addActionListener(listener);
		fileMenu.add(item);

		// Export to MRMC
		item = new JMenuItem("Export to MRMC");
		item.setActionCommand("export_mrmc");
		item.addActionListener(listener);
		fileMenu.add(item);

		// Export labels
		item = new JMenuItem("Export labels");
		item.setActionCommand("export_labels");
		item.addActionListener(listener);
		fileMenu.add(item);

		// Exit
		item = new JMenuItem("Exit");
		item.setActionCommand("exit");
		item.addActionListener(listener);
		fileMenu.add(item);

		menuBar.add(fileMenu);
	}

	private void addToolMenu() {
		toolMenu = new JMenu("Tools");
		// hideMenu = new JMenu("(Un)hide props");
		// toolMenu.add(hideMenu);

		// -----------------
		// Tool menu actions
		// -----------------
		ActionListener listener = new ToolMenuActionListener(scc_mc, this);

		// Search
		JMenuItem item = new JMenuItem("Search states");
		item.setActionCommand("search");
		item.addActionListener(listener);
		toolMenu.add(item);

		// Center initial node
		item = new JMenuItem("Center initial state");
		item.setActionCommand("centerInitial");
		item.addActionListener(listener);
		toolMenu.add(item);

		// Transform to 1 target
		item = new JMenuItem("Transform to 1 target");
		item.setActionCommand("oneTarget");
		item.addActionListener(listener);
		toolMenu.add(item);

		// (Un)hide isolated nodes
		hideIsolatedNodesItem = new JMenuItem("Hide isolated states");
		hideIsolatedNodesItem.setActionCommand("hideIsolatedNodes");
		hideIsolatedNodesItem.addActionListener(listener);
		hideIsolatedNodes = false;
		toolMenu.add(hideIsolatedNodesItem);

		// (Un)hide absorbing nodes
		hideAbsorbingNodesItem = new JMenuItem("Hide absorbing states");
		hideAbsorbingNodesItem.setActionCommand("hideAbsorbingNodes");
		hideAbsorbingNodesItem.addActionListener(listener);
		hideAbsorbingNodes = false;
		toolMenu.add(hideAbsorbingNodesItem);

		// Check Markov Chain for consistency
		item = new JMenuItem("Check consistency");
		item.setActionCommand("checkConsistency");
		item.addActionListener(listener);
		toolMenu.add(item);

		// Choose label for initial state
		item = new JMenuItem("Choose label for initial state");
		item.setActionCommand("chooseLabelInitial");
		item.addActionListener(listener);
		toolMenu.add(item);

		// Choose label for target states
		item = new JMenuItem("Choose label for target states");
		item.setActionCommand("chooseLabelTarget");
		item.addActionListener(listener);
		toolMenu.add(item);

		menuBar.add(toolMenu);
	}

	private void addGraphMenu() {
		graphMenu = new JMenu("Graph");
		colorMenu = new JMenu("Choose colour");
		graphMenu.add(colorMenu);
		layoutMenu = new JMenu("Layout");
		graphMenu.add(layoutMenu);

		// ------------
		// Node Colors
		// ------------
		colorMenu.add(new NodeColorAction(this, NodeColorAction.INITIAL_NODE));
		colorMenu.add(new NodeColorAction(this, NodeColorAction.TARGET_NODE));
		colorMenu.add(new NodeColorAction(this, NodeColorAction.DEFAULT_NODE));
		colorMenu.add(new NodeColorAction(this, NodeColorAction.REDUCED_NODE));

		// --------------------------------------
		// The various graph layout algorithms
		// --------------------------------------
		layoutMenu.add(new GraphLayoutAction(this, wrapper, GraphWrapper.LayoutType.FASTORGANIC));
		layoutMenu.add(new GraphLayoutAction(this, wrapper, GraphWrapper.LayoutType.HIERARCHICAL));
		layoutMenu.add(new GraphLayoutAction(this, wrapper, GraphWrapper.LayoutType.TREE));
		layoutMenu.add(new GraphLayoutAction(this, wrapper, GraphWrapper.LayoutType.PARTITION));
		layoutMenu.add(new GraphLayoutAction(this, wrapper, GraphWrapper.LayoutType.STACK));
		layoutMenu.add(new GraphLayoutAction(this, wrapper, GraphWrapper.LayoutType.CIRCLE));
		layoutMenu.add(new GraphLayoutAction(this, wrapper, GraphWrapper.LayoutType.PARALLEL_EDGE));
		layoutMenu.add(new GraphLayoutAction(this, wrapper, GraphWrapper.LayoutType.GRID));

		menuBar.add(graphMenu);
	}

	private void addZoomMenu() {
		zoomMenu = new JMenu("Zoom");

		// -------------------
		// Zoom In / Zoom Out
		// -------------------
		zoomMenu.add(new ZoomRelativeAction(this, true));
		zoomMenu.add(new ZoomRelativeAction(this, false));

		// -------------------
		// Absolute Zoom
		// -------------------
		zoomMenu.add(new ZoomAbsoluteAction(this, 100));
		zoomMenu.add(new ZoomAbsoluteAction(this, 75));
		zoomMenu.add(new ZoomAbsoluteAction(this, 50));
		zoomMenu.add(new ZoomAbsoluteAction(this, 25));

		menuBar.add(zoomMenu);
	}

	public Toolbar getToolbar() {
		return toolbar;
	}

	public OutputPanel getOutputPanel() {
		return outputPanel;
	}

	public ContextMenu getContextMenu() {
		return contextMenu;
	}

	public void setLoadLabItemEnabled(boolean enabled) {
		loadLabItem.setEnabled(enabled);
	}

	/**
	 * Reset labels for hide/unhide buttons to initial value "hide"
	 * 
	 */
	public void resetHide() {
		if (hideIsolatedNodes) {
			toggleHideIsolatedNodes();
		}
		if (hideAbsorbingNodes) {
			toggleHideAbsorbingNodes();
		}
	}

	/**
	 * Hide/Unhide isolated nodes
	 * 
	 * @param hide
	 *            if true, nodes are hidden
	 */
	public void hideIsolatedNodes(boolean hide) {
		hideIsolatedNodes = hide;
		if (hideIsolatedNodes) {
			hideIsolatedNodesItem.setText("Unhide isolated states");
		} else {
			hideIsolatedNodesItem.setText("Hide isolated states");
		}
		wrapper.hideIsolatedNodes(hideIsolatedNodes);
	}

	/**
	 * Toggle hide isolated nodes
	 * 
	 */
	public void toggleHideIsolatedNodes() {
		hideIsolatedNodes(!hideIsolatedNodes);
	}

	/**
	 * Hide/Unhide absorbing nodes
	 * 
	 * @param hide
	 *            if true, nodes are hidden
	 */
	public void hideAbsorbingNodes(boolean hide) {
		hideAbsorbingNodes = hide;
		if (hideAbsorbingNodes) {
			hideAbsorbingNodesItem.setText("Unhide absorbing states");
		} else {
			hideAbsorbingNodesItem.setText("Hide absorbing states");
		}
		wrapper.hideAbsorbingNodes(hideAbsorbingNodes);
	}

	/**
	 * Toggle hide absorbing nodes
	 * 
	 */
	public void toggleHideAbsorbingNodes() {
		hideAbsorbingNodes(!hideAbsorbingNodes);
	}

	public boolean isHideAbsorbingNodes() {
		return hideAbsorbingNodes;
	}

	public void popSearchMenu(boolean connectedNodesAvailable) {
		boolean center = true;
		if (connectedNodesAvailable) {
			String s = javax.swing.JOptionPane.showInputDialog(getFrame(),
					"Which nodes should be highlighted?" + Output.getLineBreak() + "Seperate Nodes by blanks");
			if (s != null && !s.isEmpty() && scc_mc.getMarkovChain() != null) {
				if (s.matches(".*answer.*") && 42 <= scc_mc.getMaxNode()) {
					scc_mc.searchNode(42, true);
				} else {
					Scanner sc = new Scanner(s).useDelimiter("\\s* ");
					wrapper.clearSelection();
					while (sc.hasNextInt()) {
						int i = sc.nextInt();
						if (i < scc_mc.getMaxNode() + 1 && i > 0) {
							scc_mc.searchNode(i, center);
							center = false;
						} else
							popSearchMenu2();
					}
				}
			}
		} else
			showErrorMessage("There are no connected nodes");
	}

	public void popSearchMenu2() {
		boolean center = true;
		String s = javax.swing.JOptionPane.showInputDialog(getFrame(), "State not found...try again");
		if (s != null && !s.isEmpty() && scc_mc.getMarkovChain() != null) {
			Scanner sc = new Scanner(s).useDelimiter("\\s*");
			wrapper.clearSelection();
			while (sc.hasNextInt()) {
				int i = sc.nextInt();
				if (i < scc_mc.getMaxNode() + 1 && i > 0) {
					scc_mc.searchNode(i, center);
					center = false;
				} else
					popSearchMenu2();
			}
		}
	}

	public void setStatusBarMessage(String s) {
		statusBar.setText(0, s);
	}

	public void showInfo() {
		java.net.URL imgURL = getClass().getResource(Config.pathToLogo);
		ImageIcon icon;
		if (imgURL != null) {
			icon = new ImageIcon(imgURL, "COMICS Logo");
		} else {
			icon = null;
			Output.print("Couldn't find file");
		}
		String s = "<html><h1>Comics 1.0</h1>";
		s += "<h2>Computing Minimal Subsystems</h2><br><ul>";
		s += "<u>Authors:</u><br><ul>";
		s += "<li>Nils Jansen</li>";
		s += "<li>Erika Abraham</li>";
		s += "<li>Jens Katelaan</li>";
		s += "<li>Maik Scheffler</li>";
		s += "<li>Matthias Volk</li>";
		s += "<li>Andreas Vorpahl</li></ul>";
		s += "<br><a href=\"" + Config.url + "\">" + Config.url + "</a><br><br>";
		s += "COMICS is distributed under the GPL conditions.<br>";
		s += "The product comes with ABSOLUTELY NO WARRANTY.<br>";
		s += "This is a free software, and you are welcome to redistribute it.<br>";
		s += "<html>";
		JOptionPane.showMessageDialog(getFrame(), s, "Info", JOptionPane.PLAIN_MESSAGE, icon);
	}

	/**
	 * Get probability (default Value is 1)
	 * 
	 * @param message
	 *            Message to display
	 * @returnProbality, 0 if cancelled
	 */
	public static double getProbability(String message) {
		return getProbability(message, 1);
	}

	/**
	 * Get probability
	 * 
	 * @param message
	 *            Message to display
	 * @param defaultValue
	 *            default value to be selected
	 * @returnProbality, 0 if cancelled
	 */
	public static double getProbability(String message, double defaultValue) {
		double probability = 0;
		while (true) {
			probability = 0;
			String s = javax.swing.JOptionPane.showInputDialog(getFrame(), message, defaultValue);
			try {
				if (s != null) {
					s = s.replace(",", ".");
					probability = Double.valueOf(s).doubleValue();
				} else {
					// Cancelled
					return 0;
				}
			} catch (NumberFormatException e1) {
				showWarningMessage("Please enter a double!");
			}
			if (!MarkovChain.isConsistent(probability)) {
				showWarningMessage("Please enter a number between 0 and 1!");
			} else {
				break;
			}
		}
		return probability;
	}

	public static Labels showLabelList(Labels labels, String message) {
		JLabel text = new JLabel(message);
		JList list = new JList(labels.toArray());
		Object[] content = new Object[] { text, list };
		JOptionPane.showMessageDialog(GUI.getFrame(), content, message, JOptionPane.PLAIN_MESSAGE);
		Labels result = new Labels();
		for (Object o : list.getSelectedValues()) {
			result.add((String) o);
		}
		return result;
	}

	/**
	 * Show message
	 * 
	 * @param message
	 *            Message text
	 * @param title
	 *            Window title
	 * @param messageType
	 *            Type from JOptionPane
	 */
	public static void showMessage(String message, String title, int messageType) {
		Output.print(title + ": " + message);
		javax.swing.JOptionPane.showMessageDialog(getFrame(), message, title, messageType);
	}

	/**
	 * Show error message
	 * 
	 * @param message
	 *            Message text
	 */
	public static void showErrorMessage(String message) {
		showMessage(message, "Error (DON'T PANIC!)", JOptionPane.ERROR_MESSAGE);
	}

	/**
	 * Show warning message
	 * 
	 * @param message
	 *            Message text
	 */
	public static void showWarningMessage(String message) {
		showMessage(message, "Warning", JOptionPane.WARNING_MESSAGE);
	}

	/**
	 * Show information message
	 * 
	 * @param message
	 *            Message text
	 */
	public static void showInfoMessage(String message) {
		showMessage(message, "Information", JOptionPane.INFORMATION_MESSAGE);
	}

	/**
	 * Show question dialog with yes/no as options
	 * 
	 * @param title
	 *            dialog title
	 * @param message
	 *            question text
	 * @return true, if Yes was chosen, false otherwise
	 */
	public static boolean showYesNoQuestion(String title, String message) {
		int n = JOptionPane.showConfirmDialog(getFrame(), message, title, JOptionPane.YES_NO_OPTION);
		if (n == JOptionPane.YES_OPTION) {
			return true;
		} else {
			return false;
		}
	}

}