/**
 *   COMICS - Computing Minimal Counterexamples for Discrete-time Markov Chains
 *
 *   COMICS is a stand-alone tool which performs model checking and the generation
 *   of counterexamples for discrete-time Markov Chains (DTMCs). *
 *
 *   Copyright (C) <2012> <RWTH Aachen University>
 *   Authors: Nils Jansen, Erika Abraham, Jens Katelaan, Maik Scheffler, Matthias Volk, Andreas Vorpahl
 *
 *   This program is free software: you can redistribute it and/or modify
 *   it under the terms of the GNU General Public License as published by
 *   the Free Software Foundation, either version 3 of the License, or
 *   (at your option) any later version.
 *
 *   This program is distributed in the hope that it will be useful,
 *   but WITHOUT ANY WARRANTY; without even the implied warranty of
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *   GNU General Public License for more details.
 *
 *   You should have received a copy of the GNU General Public License
 *   along with this program.  If not, see <http://www.gnu.org/licenses/>.
 *
 *   Main Contact:
 *
 *   Nils Jansen
 *   Theory of Hybrid Systems
 *   RWTH Aachen
 *   52056 Aachen
 *   Germany
 *   nils.jansen@cs.rwth-aachen.de
 *
*/

package comics.gui;

import java.awt.event.ActionEvent;
import java.awt.event.ActionListener;
import java.awt.event.KeyEvent;
import java.util.Enumeration;

import javax.swing.AbstractButton;
import javax.swing.ButtonModel;
import javax.swing.JToggleButton;
import javax.swing.JToolBar;

import comics.graph.gui.GraphWrapper;
import comics.gui.actions.GraphKeyHandle;
import comics.gui.actions.ToolbarKeyListener;

public class ToolbarEdit extends JToolBar implements ActionListener {

	private static final long serialVersionUID = 1L;
	private GUI gui;
	private NoneSelectedButtonGroup group = new NoneSelectedButtonGroup();

	public ToolbarEdit(GUI gui) {
		this.gui = gui;
		setFloatable(false);
		setRollover(true);

		initButtons();
		// Listener for Escape Button
		for (Enumeration<AbstractButton> e = group.getElements(); e.hasMoreElements();) {
			e.nextElement().addKeyListener(new ToolbarKeyListener());
		}
		new GraphKeyHandle(gui);
	}

	private void initButtons() {
		JToggleButton tButton = new JToggleButton("Add State");
		tButton.setVerticalTextPosition(AbstractButton.CENTER);
		tButton.setHorizontalTextPosition(AbstractButton.LEADING);
		tButton.setMnemonic(KeyEvent.VK_N);
		tButton.setActionCommand("addNode");
		tButton.addActionListener(this);
		add(tButton);
		group.add(tButton);

		tButton = new JToggleButton("Add Edge");
		tButton.setVerticalTextPosition(AbstractButton.CENTER);
		tButton.setHorizontalTextPosition(AbstractButton.LEADING);
		tButton.setMnemonic(KeyEvent.VK_E);
		tButton.setActionCommand("addEdge");
		tButton.addActionListener(this);
		add(tButton);
		group.add(tButton);

		tButton = new JToggleButton("Delete Selected");
		tButton.setVerticalTextPosition(AbstractButton.CENTER);
		tButton.setHorizontalTextPosition(AbstractButton.LEADING);
		tButton.setMnemonic(KeyEvent.VK_D);
		tButton.setActionCommand("deleteSelected");
		tButton.addActionListener(this);
		add(tButton);
		group.add(tButton);
	}

	@Override
	public void actionPerformed(ActionEvent e) {
		ButtonModel src = group.getSelection();

		GraphWrapper wrapper = gui.wrapper;

		clear();

		// Reset status bar & edit mode
		if (src == null) {
			group.clearSelection();
		}

		// Add node
		else if (src.getActionCommand().equals("addNode")) {
			gui.setStatusBarMessage("Click to add states...");
			wrapper.setEditMode(EditMode.ADD_NODE);
			wrapper.getGraphComponent().getGraph().setCellsMovable(false);
		}

		// Add edge
		else if (src.getActionCommand().equals("addEdge")) {
			gui.setStatusBarMessage("Choose source state...");
			wrapper.setEditMode(EditMode.ADD_EDGE);
			wrapper.getGraphComponent().setConnectable(true);
			wrapper.getGraphComponent().getGraph().setCellsMovable(false);
		}

		// Delete selected
		else if (src.getActionCommand().equals("deleteSelected")) {
			if (wrapper.isSelectionEmpty()) {
				gui.setStatusBarMessage("Selection is empty!");
			} else {
				wrapper.deleteSelectedNodes();
			}
			group.clearSelection();
		}
	}

	/**
	 * Reset toolbar to default (none selected)
	 */
	public void reset() {
		group.clearSelection();
		clear();
	}

	/**
	 * Clear toolbar (button can be selected)
	 */
	private void clear() {
		gui.setStatusBarMessage("");
		gui.wrapper.setEditMode(EditMode.NONE);
		gui.wrapper.getGraphMouseListener().resetState();
		gui.wrapper.getGraphComponent().setConnectable(false);
		gui.wrapper.getGraphComponent().getGraph().setCellsMovable(true);
	}

	/**
	 * Set enabled and reset
	 * 
	 * @param enabled
	 *            if true: enabled, otherwise disabled
	 */
	public void makeEnabled(boolean enabled) {
		reset();
		this.setEnabled(enabled);
		for (Enumeration<AbstractButton> e = group.getElements(); e.hasMoreElements();) {
			e.nextElement().setEnabled(enabled);
		}
	}
}
