/**
 *   COMICS - Computing Minimal Counterexamples for Discrete-time Markov Chains
 *
 *   COMICS is a stand-alone tool which performs model checking and the generation
 *   of counterexamples for discrete-time Markov Chains (DTMCs). *
 *
 *   Copyright (C) <2012> <RWTH Aachen University>
 *   Authors: Nils Jansen, Erika Abraham, Jens Katelaan, Maik Scheffler, Matthias Volk, Andreas Vorpahl
 *
 *   This program is free software: you can redistribute it and/or modify
 *   it under the terms of the GNU General Public License as published by
 *   the Free Software Foundation, either version 3 of the License, or
 *   (at your option) any later version.
 *
 *   This program is distributed in the hope that it will be useful,
 *   but WITHOUT ANY WARRANTY; without even the implied warranty of
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *   GNU General Public License for more details.
 *
 *   You should have received a copy of the GNU General Public License
 *   along with this program.  If not, see <http://www.gnu.org/licenses/>.
 *
 *   Main Contact:
 *
 *   Nils Jansen
 *   Theory of Hybrid Systems
 *   RWTH Aachen
 *   52056 Aachen
 *   Germany
 *   nils.jansen@cs.rwth-aachen.de
 *
 */

package comics.gui.counterexample;

import java.awt.GridLayout;
import java.awt.event.ActionEvent;
import java.awt.event.ActionListener;
import javax.swing.ButtonGroup;
import javax.swing.JComboBox;
import javax.swing.JDialog;
import javax.swing.JLabel;
import javax.swing.JOptionPane;
import javax.swing.JPanel;
import javax.swing.JRadioButton;
import javax.swing.JSpinner;
import javax.swing.SpinnerNumberModel;

import comics.data.CounterExampleSearchParameters;
import comics.data.CounterExampleSearchParameters.ConcretizationMode;
import comics.data.CounterExampleSearchParameters.Heuristic;
import comics.gui.GUI;

public class CounterExampleStepDialog implements ActionListener {
	private ButtonGroup stepCountButtonGroup = new ButtonGroup();
	private JRadioButton noAutoConcretizeButton;
	private JRadioButton stepTillEndButton;
	private JRadioButton stepSizeButton;
	private JSpinner stepSpinner;
	private JComboBox heuristicComboBox;
	private JComboBox concretizationComboBox;
	private JPanel stepPanel = new JPanel();
	private JPanel stepSizePanel = new JPanel();
	private JPanel heuristicPanel = new JPanel();
	private JPanel concretizationPanel = new JPanel();

	/**
	 * Ask for counterexample Parameters
	 * 
	 * @param parameters
	 *            default parameters for this dialog from last step, afterwards
	 *            this holds the new parameters
	 * @return false, if cancelled, true otherwise
	 */
	public boolean show(CounterExampleSearchParameters parameters) {
		// No automatic concretize, steps till end or user defined step count
		stepPanel.setLayout(new GridLayout(4, 1));
		JLabel stepLabel = new JLabel("Concretization Mode");
		stepPanel.add(stepLabel);

		stepTillEndButton = new JRadioButton("Search/concretize until concrete system results");
		stepTillEndButton.setActionCommand("stepsTillEnd");
		stepTillEndButton.addActionListener(this);
		stepCountButtonGroup.add(stepTillEndButton);
		stepPanel.add(stepTillEndButton);

		stepSizeButton = new JRadioButton("Choose number of search/concretization iterations");
		stepSizeButton.setActionCommand("userStepSize");
		stepSizeButton.addActionListener(this);
		stepCountButtonGroup.add(stepSizeButton);
		stepPanel.add(stepSizeButton);

		noAutoConcretizeButton = new JRadioButton("No automatic concretization");
		noAutoConcretizeButton.setActionCommand("noConcretize");
		noAutoConcretizeButton.addActionListener(this);
		stepCountButtonGroup.add(noAutoConcretizeButton);
		stepPanel.add(noAutoConcretizeButton);

		// Count of automatic steps
		SpinnerNumberModel model = new SpinnerNumberModel(parameters.getStepSize(), 1, Integer.MAX_VALUE, 1);
		stepSpinner = new JSpinner(model);
		stepSizePanel.setLayout(new GridLayout(2, 1));
		JLabel stepSizeLabel = new JLabel("Number of automatic steps");
		stepSizePanel.add(stepSizeLabel);
		stepSizePanel.add(stepSpinner);

		// Heuristic for choosing next SCC to concretize
		heuristicComboBox = new JComboBox(Heuristic.values());
		heuristicComboBox.setSelectedItem(parameters.getHeuristic());
		heuristicPanel.setLayout(new GridLayout(2, 1));
		JLabel heuristicLabel = new JLabel("Heuristics for the choice of abstract node to concretize:");
		heuristicPanel.add(heuristicLabel);
		heuristicPanel.add(heuristicComboBox);

		// Concretization mode
		concretizationComboBox = new JComboBox(ConcretizationMode.values());
		concretizationComboBox.setSelectedItem(parameters.getConcretizationMode());
		concretizationPanel.setLayout(new GridLayout(2, 1));
		JLabel concretizationLabel = new JLabel("Count of SCCs to concretize in each step");
		concretizationPanel.add(concretizationLabel);
		concretizationPanel.add(concretizationComboBox);

		initialize(parameters);
		Object[] content = { stepPanel, stepSizePanel, heuristicPanel, concretizationPanel };
		JOptionPane contentPane = new JOptionPane(content, JOptionPane.QUESTION_MESSAGE, JOptionPane.OK_CANCEL_OPTION);
		JDialog dialog = contentPane.createDialog(GUI.getFrame(), "Choose Parameters for Counterexample");

		dialog.setVisible(true);

		// Dialog is closed, return value
		if (contentPane.getValue() == null || !contentPane.getValue().equals(JOptionPane.YES_OPTION)) {
			return false;
		} else {
			parameters.setNoAutoConcretize(noAutoConcretizeButton.isSelected());
			if (!parameters.isNoAutoConcretize()) {
				parameters.setStepsTillEnd(stepTillEndButton.isSelected());
				if (!parameters.isStepsTillEnd()) {
					parameters.setStepSize(model.getNumber().intValue());
				}
				parameters.setHeuristic((Heuristic) heuristicComboBox.getSelectedItem());
				parameters.setConcretizationMode((ConcretizationMode) concretizationComboBox.getSelectedItem());
			}
			return true;
		}
	}

	/**
	 * Event for selection radio box
	 * 
	 * @param e
	 *            Event
	 */
	public void actionPerformed(ActionEvent e) {
		if (e.getActionCommand().equals("noConcretize")) {
			setEnabledParameter(false);
			stepSpinner.setEnabled(false);
		} else if (e.getActionCommand().equals("stepsTillEnd")) {
			setEnabledParameter(true);
			stepSpinner.setEnabled(false);
		} else {
			setEnabledParameter(true);
			stepSpinner.setEnabled(true);
		}
	}

	/**
	 * Initialize gui correctly depending on former parameters
	 * 
	 * @param parameters
	 *            parameters from last step
	 */
	private void initialize(CounterExampleSearchParameters parameters) {
		if (parameters.isNoAutoConcretize()) {
			noAutoConcretizeButton.setSelected(true);
			stepSpinner.setEnabled(false);
			setEnabledParameter(false);
		} else if (parameters.isStepsTillEnd()) {
			stepTillEndButton.setSelected(true);
			stepSpinner.setEnabled(false);
			setEnabledParameter(true);
		} else {
			stepSizeButton.setSelected(true);
			stepSpinner.setEnabled(true);
			setEnabledParameter(true);
		}
	}

	/**
	 * Enable/Disable parameters panels
	 * 
	 * @param enabled
	 *            if true, panels are enabled
	 */
	private void setEnabledParameter(boolean enabled) {
		heuristicComboBox.setEnabled(enabled);
		concretizationComboBox.setEnabled(enabled);
	}
}