/**
 *   COMICS - Computing Minimal Counterexamples for Discrete-time Markov Chains
 *
 *   COMICS is a stand-alone tool which performs model checking and the generation
 *   of counterexamples for discrete-time Markov Chains (DTMCs). *
 *
 *   Copyright (C) <2012> <RWTH Aachen University>
 *   Authors: Nils Jansen, Erika Abraham, Jens Katelaan, Maik Scheffler, Matthias Volk, Andreas Vorpahl
 *
 *   This program is free software: you can redistribute it and/or modify
 *   it under the terms of the GNU General Public License as published by
 *   the Free Software Foundation, either version 3 of the License, or
 *   (at your option) any later version.
 *
 *   This program is distributed in the hope that it will be useful,
 *   but WITHOUT ANY WARRANTY; without even the implied warranty of
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *   GNU General Public License for more details.
 *
 *   You should have received a copy of the GNU General Public License
 *   along with this program.  If not, see <http://www.gnu.org/licenses/>.
 *
 *   Main Contact:
 *
 *   Nils Jansen
 *   Theory of Hybrid Systems
 *   RWTH Aachen
 *   52056 Aachen
 *   Germany
 *   nils.jansen@cs.rwth-aachen.de
 *
 */

package comics.gui.counterexample;

import java.awt.event.ActionEvent;
import java.awt.event.ActionListener;
import java.awt.event.KeyEvent;
import java.io.File;

import javax.swing.AbstractButton;
import javax.swing.JButton;
import javax.swing.JOptionPane;
import javax.swing.JToolBar;

import comics.SCC_MC;
import comics.data.CounterExampleSearchParameters;
import comics.graph.data.MarkovChain;
import comics.gui.GUI;
import comics.io.FileHandler;
import comics.io.input.XmlParser;
import comics.io.output.XmlWriter;
import comics.tools.counterexample.SccModelChecker;

/**
 * Toolbar for counter examples
 * 
 */
public class ToolbarCounterExample extends JToolBar implements ActionListener {

	private static final long serialVersionUID = 1L;
	private GUI gui;
	private JButton buttonModelCheck;
	private JButton buttonCounterExample;
	private SccModelChecker modelChecker = null;
	private boolean initialized;

	/**
	 * Constructor
	 * 
	 * @param gui
	 * @param scc
	 */
	public ToolbarCounterExample(GUI gui) {
		this.gui = gui;
		setFloatable(false);
		setRollover(true);

		initButtons();
	}

	/**
	 * Initialize buttons
	 */
	private void initButtons() {
		buttonModelCheck = new JButton("Model Check");
		buttonModelCheck.setVerticalTextPosition(AbstractButton.CENTER);
		buttonModelCheck.setHorizontalTextPosition(AbstractButton.LEADING);
		buttonModelCheck.setMnemonic(KeyEvent.VK_M);
		buttonModelCheck.setActionCommand("modelCheck");
		buttonModelCheck.addActionListener(this);
		add(buttonModelCheck);

		buttonCounterExample = new JButton("Search Counterexample");
		buttonCounterExample.setVerticalTextPosition(AbstractButton.CENTER);
		buttonCounterExample.setHorizontalTextPosition(AbstractButton.LEADING);
		buttonCounterExample.setMnemonic(KeyEvent.VK_F);
		buttonCounterExample.setActionCommand("searchCounterExample");
		buttonCounterExample.addActionListener(this);
		add(buttonCounterExample);
	}

	@Override
	public void actionPerformed(ActionEvent e) {
		SCC_MC scc = SCC_MC.getInstance();

		// Model Check
		if (e.getActionCommand().equals("modelCheck")) {
			gui.setStatusBarMessage("Model Checking");

			scc.lockGraph();
			modelChecker = new SccModelChecker();

			if (scc.getMarkovChain().isEmpty()) {
				GUI.showErrorMessage("No Markov Chain specified. Please load a file or create a graph!");
				scc.unlockGraph();
				return;
			}

			// Checking for initial states and target states
			if (!scc.getMarkovChain().hasInitialAndTarget()) {
				GUI.showErrorMessage("Markov Chain can't be model checked. Initial or target state is missing");
				scc.unlockGraph();
				return;
			}

			// Check for consistency
			if (!SCC_MC.getInstance().getMarkovChain().isMarkovChainConsistent()) {
				GUI.showErrorMessage("Warning! Markov chain ist not consistent!");
				boolean answer = GUI.showYesNoQuestion("Insert self loops?",
						"Would you like to insert self loops for absorbing states and continue model checking?");
				if (answer) {
					SCC_MC.getInstance().getMarkovChain().insertSelfLoops(gui);
				} else {
					reset();
					return;
				}
			}

			scc.setModelCheckResult(modelChecker.modelCheck(scc.getMarkovChain()));

			// Model checking
			if (scc.getModelCheckResult() == null) {
				GUI.showErrorMessage("No model checking result");
				scc.unlockGraph();
				return;
			}
			// Show result
			scc.executeParser(new XmlParser(scc.getModelCheckResult().getMarkovChain()));
			gui.hideAbsorbingNodes(true);
		}
		// Search counterexample
		else if (e.getActionCommand().equals("searchCounterExample")) {
			gui.setStatusBarMessage("Search counterexample");
			if (!initialized) {
				if (!initializeCounterExample())
					return;
			}

			// set step count and other parameters
			if (scc.counterExampleSearchParameters.isAbstractSearch()) {
				boolean successfull = new CounterExampleStepDialog().show(scc.counterExampleSearchParameters);
				if (!successfull) {
					return;
				}
				scc.counterExampleSearchParameters.setConcretizedNodes(gui.wrapper.getConcretizedGraphs());
			} else {
				// Concrete search is not stepwise
				scc.counterExampleSearchParameters.setStepsTillEnd(true);
				scc.counterExampleSearchParameters.setNoAutoConcretize(false);
			}
			scc.counterExampleResult = modelChecker.doCounterExampleSteps(scc.counterExampleSearchParameters);

			if (scc.counterExampleResult == null) {
				GUI.showErrorMessage("No counterexample");
				scc.unlockGraph();
				return;
			} else if (scc.counterExampleResult.getSteps() == 0) {
				GUI.showMessage("Nothing was concretized!", "No User Input!", JOptionPane.INFORMATION_MESSAGE);
			} else {
				// Show result
				gui.setStatusBarMessage("Stepsize: " + scc.counterExampleSearchParameters.getStepSize() + ", Stepcount: "
						+ scc.counterExampleResult.getSteps());
				if (scc.counterExampleSearchParameters.isClosure()) {
					scc.executeParser(new XmlParser(scc.counterExampleResult.getMarkovChain()));
					gui.hideAbsorbingNodes(true);
				} else {
					String pathsString = scc.counterExampleResult.getMarkovChain();
					new PathResult().show(pathsString);
					FileHandler.StringToFileContent(pathsString, new File("counter_example_paths.txt"));
				}
				// Check if finished
				if (scc.counterExampleResult.isComplete()) {
					GUI.showMessage("Counterexample generation finished", "Counterexample complete",
							JOptionPane.INFORMATION_MESSAGE);
					scc.unlockGraph();
				}
			}
		}
	}

	/**
	 * Initialize counterexample search and ask for probability bound
	 * 
	 * @return true, if initialization was successful
	 */
	public boolean initializeCounterExample() {
		CounterExampleSearchParameters parameters = SCC_MC.getInstance().counterExampleSearchParameters;
		if (parameters == null) {
			parameters = new CounterExampleSearchParameters();
		}
		boolean successfull = new CounterExampleInitDialog().show(parameters);
		if (!successfull) {
			return false;
		}
		if (parameters.getProbabilityBound() == 0) {
			// Use bound from model checking as default value
			parameters.setProbabilityBound(SCC_MC.getInstance().getModelCheckResult().getProbability());
		}

		double probability = GUI
				.getProbability("Enter probability for counterexample search", parameters.getProbabilityBound());
		if (probability == 0 || probability > SCC_MC.getInstance().getModelCheckResult().getProbability()) {
			return false;
		}
		parameters.setProbabilityBound(probability);

		MarkovChain mc = SCC_MC.getInstance().getMarkovChain();
		String xmlGraph = new XmlWriter(mc, false, SCC_MC.getInstance().getModelCheckResult().getProbability())
				.markovChainToXMLString(mc);
		SCC_MC.getInstance().counterExampleSearchParameters = parameters;
		initialized = modelChecker.initCounterExampleSearch(xmlGraph, parameters);
		if (!initialized) {
			GUI.showErrorMessage("Could not initialize counterexample module");
			SCC_MC.getInstance().unlockGraph();
			return false;
		}
		gui.getOutputPanel().getOverviewPanel().reloadValues();
		return true;
	}

	/**
	 * Reset toolbar and modelChecker
	 * 
	 */
	public void reset() {
		buttonModelCheck.setEnabled(true);
		buttonCounterExample.setEnabled(false);
		initialized = false;

		// Destroy actual model checker
		if (modelChecker != null) {
			modelChecker.cleanup();
			modelChecker = null;
		}
	}

	public void enableCounterExampleSearch() {
		buttonModelCheck.setEnabled(false);
		buttonCounterExample.setEnabled(true);
		if (modelChecker == null) {
			// Init model checker
			modelChecker = new SccModelChecker();
		}
	}
}
