/**
 *   COMICS - Computing Minimal Counterexamples for Discrete-time Markov Chains
 *
 *   COMICS is a stand-alone tool which performs model checking and the generation
 *   of counterexamples for discrete-time Markov Chains (DTMCs). *
 *
 *   Copyright (C) <2012> <RWTH Aachen University>
 *   Authors: Nils Jansen, Erika Abraham, Jens Katelaan, Maik Scheffler, Matthias Volk, Andreas Vorpahl
 *
 *   This program is free software: you can redistribute it and/or modify
 *   it under the terms of the GNU General Public License as published by
 *   the Free Software Foundation, either version 3 of the License, or
 *   (at your option) any later version.
 *
 *   This program is distributed in the hope that it will be useful,
 *   but WITHOUT ANY WARRANTY; without even the implied warranty of
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *   GNU General Public License for more details.
 *
 *   You should have received a copy of the GNU General Public License
 *   along with this program.  If not, see <http://www.gnu.org/licenses/>.
 *
 *   Main Contact:
 *
 *   Nils Jansen
 *   Theory of Hybrid Systems
 *   RWTH Aachen
 *   52056 Aachen
 *   Germany
 *   nils.jansen@cs.rwth-aachen.de
 *
 */

package comics.gui.output_panel;

import java.awt.BorderLayout;
import java.awt.Dimension;
import javax.swing.JPanel;
import javax.swing.JTabbedPane;

import comics.SCC_MC;
import comics.data.Config;
import comics.graph.data.Edge;
import comics.graph.data.MarkovChain;
import comics.graph.data.Node;

/**
 * List to display graph. If graph is too big, only this list is displayed and
 * no graphical nodes
 * 
 */
public class OutputPanel extends JPanel {
	private static final long serialVersionUID = 1L;
	private JTabbedPane tabbedPane = new JTabbedPane();
	private TabNodes tabNodes;
	private TabEdges tabEdges;
	private TabTargetNodes tabTargetNodes;
	private OverviewPanel overviewPanel;

	public OutputPanel() {
		setLayout(new BorderLayout());
		add(tabbedPane, BorderLayout.CENTER);
		overviewPanel = new OverviewPanel();
		tabNodes = new TabNodes(overviewPanel);
		tabEdges = new TabEdges(overviewPanel);
		tabTargetNodes = new TabTargetNodes(overviewPanel);
		tabbedPane.add(tabNodes);
		tabbedPane.add(tabEdges);
		tabbedPane.add(tabTargetNodes);
		add(overviewPanel, BorderLayout.PAGE_END);
		overviewPanel.setPreferredSize(new Dimension(Config.overviewPanelWidth, Config.overviewPanelHeight));

	}

	public TabNodes getTabNodes() {
		return tabNodes;
	}

	public TabTargetNodes getTabTargetNodes() {
		return tabTargetNodes;
	}

	public TabEdges getTabEdges() {
		return tabEdges;
	}

	public OverviewPanel getOverviewPanel() {
		return overviewPanel;
	}

	/**
	 * Clear list
	 * 
	 */
	public void clear() {
		tabNodes.clear();
		tabTargetNodes.clear();
		tabEdges.clear();
		overviewPanel.setDefault();
	}

	/**
	 * Load nodes and edges from markov chain and display them in list
	 * 
	 */
	public void loadMarkovChain() {
		MarkovChain mc = SCC_MC.getInstance().getMarkovChain();
		if (mc != null) {
			tabNodes.clear();
			tabTargetNodes.clear();
			for (Node n : mc.getNodes()) {
				tabNodes.addNode(n);
				tabTargetNodes.addNode(n);
			}
			tabEdges.clear();
			for (Edge e : mc.getEdges()) {
				tabEdges.addEdge(e);
			}
		}
		overviewPanel.reloadValues();
	}
}