/**
 *   COMICS - Computing Minimal Counterexamples for Discrete-time Markov Chains
 *
 *   COMICS is a stand-alone tool which performs model checking and the generation
 *   of counterexamples for discrete-time Markov Chains (DTMCs). *
 *
 *   Copyright (C) <2012> <RWTH Aachen University>
 *   Authors: Nils Jansen, Erika Abraham, Jens Katelaan, Maik Scheffler, Matthias Volk, Andreas Vorpahl
 *
 *   This program is free software: you can redistribute it and/or modify
 *   it under the terms of the GNU General Public License as published by
 *   the Free Software Foundation, either version 3 of the License, or
 *   (at your option) any later version.
 *
 *   This program is distributed in the hope that it will be useful,
 *   but WITHOUT ANY WARRANTY; without even the implied warranty of
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *   GNU General Public License for more details.
 *
 *   You should have received a copy of the GNU General Public License
 *   along with this program.  If not, see <http://www.gnu.org/licenses/>.
 *
 *   Main Contact:
 *
 *   Nils Jansen
 *   Theory of Hybrid Systems
 *   RWTH Aachen
 *   52056 Aachen
 *   Germany
 *   nils.jansen@cs.rwth-aachen.de
 *
 */

package comics.gui.output_panel;

import javax.swing.JComponent;
import javax.swing.JLabel;
import javax.swing.JPanel;
import javax.swing.SpringLayout;

import comics.SCC_MC;
import comics.data.Config;
import comics.graph.data.MarkovChain;
import comics.graph.data.Node;

/**
 * Overview over general properties of current markov chain
 * 
 */
public class OverviewPanel extends JPanel {
	private static final long serialVersionUID = 1L;
	private JLabel textNoNodes = new JLabel();
	private JLabel textNoEdges = new JLabel();
	private JLabel textInitial = new JLabel();
	private JLabel textFilename = new JLabel();
	private JLabel textProbabilityModelCheck = new JLabel();
	private JLabel textProbabilityBound = new JLabel();
	private JLabel textProbabilitySubsystem = new JLabel();
	private JLabel textNoSteps = new JLabel();

	public OverviewPanel() {
		super(new SpringLayout());
		addRow("Number states", textNoNodes);
		addRow("Number edges", textNoEdges);
		addRow("Initial", textInitial);
		addRow("Filename", textFilename);
		addRow("Prob. model check", textProbabilityModelCheck);
		addRow("Prob. bound", textProbabilityBound);
		addRow("Prob. crit. subsystem", textProbabilitySubsystem);
		addRow("Number steps", textNoSteps);
		SpringUtilities.makeCompactGrid(this, 8, 2, // rows, cols
				6, 6, // initX, initY
				6, 6); // xPad, yPad

		setDefault();
	}

	private void addRow(String text, JComponent component) {
		JLabel l = new JLabel(text);
		add(l);
		l.revalidate();
		l.repaint();
		add(component);
	}

	/**
	 * Set default values for properties
	 * 
	 */
	public void setDefault() {
		setText(textNoNodes, "0");
		setText(textNoEdges, "0");
		setText(textInitial, "none");
		setText(textFilename, "");
		setText(textProbabilityModelCheck, "0");
		setText(textProbabilityBound, "0");
		setText(textProbabilitySubsystem, "0");
		setText(textNoSteps, "0");
	}

	public void reloadValues() {
		MarkovChain mc = SCC_MC.getInstance().getMarkovChain();
		if (mc != null) {
			OutputPanel outputPanel = SCC_MC.getInstance().getGui().getOutputPanel();
			// Number of nodes
			int noNodes = outputPanel.getTabNodes().getNodeCount();
			setText(textNoNodes, "" + noNodes);

			// Number of edges
			int noEdges = outputPanel.getTabEdges().getEdgeCount();
			setText(textNoEdges, "" + noEdges);

			// Initial node
			if (mc.getInitialNode() != null) {
				setText(textInitial, mc.getInitialNode().toString());
			} else {
				String s = "";
				if (!mc.getInputNodes().isEmpty()) {
					for (Node n : mc.getInputNodes()) {
						s += n.toString() + ", ";
					}
					s = s.substring(0, s.length() - 2);
					if (s.isEmpty()) {
						setText(textInitial, "none");
					} else {
						setText(textInitial, s);
					}
				}
			}

			// Filename
			setText(textFilename, Config.filename);

			// Model checking probabilty bound
			if (SCC_MC.getInstance().getModelCheckResult() != null) {
				setText(textProbabilityModelCheck, "" + SCC_MC.getInstance().getModelCheckResult().getProbability());
			} else {
				setText(textProbabilityModelCheck, "0");
			}

			if (SCC_MC.getInstance().counterExampleResult != null) {
				// Probability subsystem
				setText(textProbabilitySubsystem, "" + SCC_MC.getInstance().counterExampleResult.getProbability());

				// Number of steps
				setText(textNoSteps, "" + SCC_MC.getInstance().counterExampleResult.getSteps());
			} else {
				setText(textProbabilitySubsystem, "0");
				setText(textNoSteps, "0");
			}

			if (SCC_MC.getInstance().counterExampleSearchParameters != null) {
				// Probability bound
				setText(textProbabilityBound, "" + SCC_MC.getInstance().counterExampleSearchParameters.getProbabilityBound());
			} else {
				setText(textProbabilityBound, "0");
			}
		} else {
			setDefault();
		}
	}

	private void setText(JLabel label, String text) {
		label.setText(text);
	}
}