/**
 *   COMICS - Computing Minimal Counterexamples for Discrete-time Markov Chains
 *
 *   COMICS is a stand-alone tool which performs model checking and the generation
 *   of counterexamples for discrete-time Markov Chains (DTMCs). *
 *
 *   Copyright (C) <2012> <RWTH Aachen University>
 *   Authors: Nils Jansen, Erika Abraham, Jens Katelaan, Maik Scheffler, Matthias Volk, Andreas Vorpahl
 *
 *   This program is free software: you can redistribute it and/or modify
 *   it under the terms of the GNU General Public License as published by
 *   the Free Software Foundation, either version 3 of the License, or
 *   (at your option) any later version.
 *
 *   This program is distributed in the hope that it will be useful,
 *   but WITHOUT ANY WARRANTY; without even the implied warranty of
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *   GNU General Public License for more details.
 *
 *   You should have received a copy of the GNU General Public License
 *   along with this program.  If not, see <http://www.gnu.org/licenses/>.
 *
 *   Main Contact:
 *
 *   Nils Jansen
 *   Theory of Hybrid Systems
 *   RWTH Aachen
 *   52056 Aachen
 *   Germany
 *   nils.jansen@cs.rwth-aachen.de
 *
 */

package comics.io;

import java.io.File;
import java.io.FileOutputStream;
import java.io.FileReader;
import java.io.FileWriter;
import java.io.IOException;
import java.io.OutputStreamWriter;

import javax.swing.JFileChooser;

import comics.SCC_MC;
import comics.data.Config;
import comics.gui.GUI;
import comics.utilities.Output;

/**
 * Handles files (open and save)
 * 
 */
public class FileHandler {

	public FileHandler() {
		loadDefaultFilePath();
	}

	/**
	 * Get File for Markov Chain
	 * 
	 * @param filetypes
	 *            Filetypes as ExtensionType array
	 * @return file
	 */
	public File getOpenFile(ExtensionType[] filetypes) {
		String names = "";
		for (int i = 0; i < filetypes.length - 1; i++) {
			names += filetypes[i] + ", ";
		}
		names += filetypes[filetypes.length - 1];
		ExtensionFileFilter filter = new ExtensionFileFilter(names, filetypes);
		return showFileDialog(filter, true);
	}

	/**
	 * Get File for Markov Chain
	 * 
	 * @param filetype
	 *            Filetype as ExtensionType
	 * @return file
	 */
	public File getOpenFile(ExtensionType filetype) {
		return getOpenFile(new ExtensionType[] { filetype });
	}

	/**
	 * Get File for saving Markov Chain
	 * 
	 * @param filetypes
	 *            Filetypes as ExtensionType
	 * @param defaultFile
	 *            Default selected file
	 * @return file
	 */
	public File getSaveFile(ExtensionType[] filetypes, String defaultFile) {
		String names = "";
		for (int i = 0; i < filetypes.length - 1; i++) {
			names += filetypes[i] + ", ";
		}
		names += filetypes[filetypes.length - 1];
		ExtensionFileFilter filter = new ExtensionFileFilter(names, filetypes);

		return showFileDialog(filter, false);
	}

	/**
	 * Get File for saving Markov Chain
	 * 
	 * @param filetype
	 *            Filetype as ExtensionType
	 * @param defaultFile
	 *            Default selected file
	 * @return file
	 */
	public File getSaveFile(ExtensionType filetype, String defaultFile) {
		return getSaveFile(new ExtensionType[] { filetype }, defaultFile);
	}

	/**
	 * Starts file-dialog and returns selected file
	 * 
	 * @param filter
	 *            Filter for file extension
	 * @param open
	 *            if true: open-file-dialog, if false: save-file-dialog
	 * @return selected file
	 */
	private File showFileDialog(ExtensionFileFilter filter, boolean open) {
		JFileChooser fileChooser = new JFileChooser();
		fileChooser.setFileFilter(filter);
		fileChooser.setCurrentDirectory(Config.path);
		int status = 0;
		if (open) {
			status = fileChooser.showOpenDialog(GUI.getFrame());
		} else {
			status = fileChooser.showSaveDialog(GUI.getFrame());
		}
		if (status == JFileChooser.APPROVE_OPTION) {
			File selectedFile = fileChooser.getSelectedFile();
			if (selectedFile == null) {
				Output.print("Couldn't open file");
				return null;
			}
			if (!filter.accept(selectedFile)) {
				// Add file extension
				selectedFile = new File(selectedFile.getAbsolutePath() + "." + filter.getFirstExtension());
			}
			SCC_MC.getInstance().getGui().setStatusBarMessage(selectedFile.getName());
			saveDefaultFilePath(selectedFile.getParentFile());
			return selectedFile;
		} else {
			Output.print(JFileChooser.CANCEL_OPTION + "");
			return null;
		}
	}

	/**
	 * Get file content
	 * 
	 * @param file
	 * @return file content as string
	 */
	public static String fileContentToString(File file) {
		StringBuffer sb = new StringBuffer();
		FileReader reader;
		try {
			reader = new FileReader(file);
			for (int c; (c = reader.read()) != -1;)
				sb.append((char) c);
		} catch (IOException e) {
			e.printStackTrace();
		}
		return sb.toString();
	}

	/**
	 * Write string to file
	 * 
	 * @param content
	 * @param file
	 * @return true, if succesfull
	 */
	public static boolean StringToFileContent(String content, File file) {
		OutputStreamWriter out;
		try {
			out = new OutputStreamWriter(new FileOutputStream(file));
			try {
				out.write(content);
			} finally {
				out.close();
			}
		} catch (Exception e) {
			Output.print("Couldn't write string to file");
			return false;
		}
		return true;
	}

	private void loadDefaultFilePath() {
		File defaultPath = new File("path.txt");
		if (defaultPath.exists()) {
			try {
				String pathWrittenInTextFile = fileContentToString(new File("path.txt"));
				Config.path = new File(pathWrittenInTextFile);
			} catch (Exception e) {
				Output.print("path.txt doesn't exist");
			}
		} else {
			Config.path = new File(".");
		}
		Output.print(Config.path.toString());
	}

	/**
	 * Save current path in File and Config
	 * 
	 * @param path
	 *            current path
	 */
	private void saveDefaultFilePath(File path) {
		Config.path = path;
		File filePath = new File("path.txt");
		try {
			FileWriter writer = new FileWriter(filePath, false);
			writer.write(path.getPath());
			writer.flush();
			writer.close();
		} catch (IOException e) {
			e.printStackTrace();
		}
	}
}
