/**
 *   COMICS - Computing Minimal Counterexamples for Discrete-time Markov Chains
 *
 *   COMICS is a stand-alone tool which performs model checking and the generation
 *   of counterexamples for discrete-time Markov Chains (DTMCs). *
 *
 *   Copyright (C) <2012> <RWTH Aachen University>
 *   Authors: Nils Jansen, Erika Abraham, Jens Katelaan, Maik Scheffler, Matthias Volk, Andreas Vorpahl
 *
 *   This program is free software: you can redistribute it and/or modify
 *   it under the terms of the GNU General Public License as published by
 *   the Free Software Foundation, either version 3 of the License, or
 *   (at your option) any later version.
 *
 *   This program is distributed in the hope that it will be useful,
 *   but WITHOUT ANY WARRANTY; without even the implied warranty of
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *   GNU General Public License for more details.
 *
 *   You should have received a copy of the GNU General Public License
 *   along with this program.  If not, see <http://www.gnu.org/licenses/>.
 *
 *   Main Contact:
 *
 *   Nils Jansen
 *   Theory of Hybrid Systems
 *   RWTH Aachen
 *   52056 Aachen
 *   Germany
 *   nils.jansen@cs.rwth-aachen.de
 *
 */

package comics.io.input;

import java.io.File;
import java.util.regex.Matcher;
import java.util.regex.Pattern;

import comics.data.Config;
import comics.data.CounterExampleSearchParameters;
import comics.io.FileHandler;
import comics.utilities.Output;

/**
 * @author jansen
 * 
 *         This is for parsing .conf-Files
 * 
 *         Syntax:
 * 
 *         TASK counterexample | modelchecking PROBABILITY_BOUND double
 *         DTMC_FILE|XML_FILE *.dtmc | *.xml | *.tra REPRESENTATION subsystem |
 *         pathset SEARCH_ALGORITHM local | global ABSTRACTION abstract |
 *         concrete
 */
public class ConfReader extends InputParser {

	public int count = 0;
	String fileContent = null;
	CounterExampleSearchParameters parameters;

	public ConfReader(File fileToReadFrom, CounterExampleSearchParameters parameters) {
		super(fileToReadFrom);
		this.parameters = parameters;
	}

	@Override
	public boolean parse() {
		if (fileToReadFrom != null) {
			fileContent = FileHandler.fileContentToString(fileToReadFrom);
		}
		if (fileContent != null) {
			try {
				String[] lines = fileContent.split("\\r?" + Output.getLineBreak());
				for (String line : lines) {
					Pattern pat1 = Pattern.compile("((?i)TASK) ((?i)" + Config.TASK_MODELCHECKING + ")");
					Matcher mat1 = pat1.matcher(line);

					Pattern pat2 = Pattern.compile("((?i)TASK) ((?i)" + Config.TASK_COUNTEREXAMPLE + ")");
					Matcher mat2 = pat2.matcher(line);

					Pattern pat3 = Pattern.compile("((?i)PROBABILITY_BOUND) ((\\b(\\d+)\\.(\\d+)\\b))");
					Matcher mat3 = pat3.matcher(line);

					Pattern pat4 = Pattern.compile("((?i)REPRESENTATION) ((?i)" + Config.REPRESENTATION_SUBSYSTEM + ")");
					Matcher mat4 = pat4.matcher(line);

					Pattern pat5 = Pattern.compile("((?i)REPRESENTATION) ((?i)" + Config.REPRESENTATION_PATHSET + ")");
					Matcher mat5 = pat5.matcher(line);

					Pattern pat6 = Pattern.compile("((?i)SEARCH_ALGORITHM) ((?i)" + Config.SEARCH_ALGORITHM_LOCAL + ")");
					Matcher mat6 = pat6.matcher(line);

					Pattern pat7 = Pattern.compile("((?i)SEARCH_ALGORITHM) ((?i)" + Config.SEARCH_ALGORITHM_GLOBAL + ")");
					Matcher mat7 = pat7.matcher(line);

					Pattern pat8 = Pattern.compile("((?i)ABSTRACTION) ((?i)" + Config.ABSTRACTION_ABSTRACT + ")");
					Matcher mat8 = pat8.matcher(line);

					Pattern pat9 = Pattern.compile("((?i)ABSTRACTION) ((?i)" + Config.ABSTRACTION_CONCRETE + ")");
					Matcher mat9 = pat9.matcher(line);

					Pattern pat10 = Pattern.compile("((?i)[DTMC|XML]_FILE) ((?i).*\\.(dtmc|xml|tra))");
					Matcher mat10 = pat10.matcher(line);

					Pattern pat11 = Pattern.compile("((?i)DTMC_FILE) ((?i).*\\.conf)");
					Matcher mat11 = pat11.matcher(line);

					if (mat1.find()) {
						Output.print("TASK: Model Checking");
						Config.task = Config.TASK_MODELCHECKING;
						continue;
					}

					if (mat2.find()) {
						Output.print("TASK: Counterexample Generation");
						Config.task = Config.TASK_COUNTEREXAMPLE;
						continue;
					}

					if (mat3.find()) {
						parameters.setProbabilityBound(Double.valueOf(mat3.group(3)).doubleValue());
						Output.print("Probability Bound: " + parameters.getProbabilityBound());
						continue;
					}

					if (mat4.find()) {
						Output.print("Representation: Critical Subsystem");
						parameters.setClosure(true);
						continue;
					}

					if (mat5.find()) {
						Output.print("Representation: Set of Paths");
						parameters.setClosure(false);
						continue;
					}

					if (mat6.find()) {
						Output.print("Search Algorithm: Local Search");
						parameters.setGlobalSearch(false);
						continue;
					}

					if (mat7.find()) {
						Output.print("Search Algorithm: Global Search");
						parameters.setGlobalSearch(true);
						continue;
					}

					if (mat8.find()) {
						Output.print("Abstraction: abstract");
						parameters.setAbstractSearch(true);
						continue;
					}

					if (mat9.find()) {
						Output.print("Abstraction: concrete");
						parameters.setAbstractSearch(false);
						continue;
					}

					if (mat10.find()) {
						handleDTMCFile(mat10);
						continue;
					}

					if (mat11.find()) {
						Output.print(".conf File is no proper dtmc file...");
						continue;
					} else {
						Output.print("Line: '" + line + "' is not valid.");
						break;
					}

				}
			} catch (Exception ausnahme) {
				return false;
			}
		} else {
			Output.print("No file and no string");
			return false;
		}

		return true;
	}

	private void handleDTMCFile(Matcher match) {
		Config.filename = match.group(2);
		result = null;
		Output.print("DTMC File: " + Config.filename);
	}
}
