/**
 *   COMICS - Computing Minimal Counterexamples for Discrete-time Markov Chains
 *
 *   COMICS is a stand-alone tool which performs model checking and the generation
 *   of counterexamples for discrete-time Markov Chains (DTMCs). *
 *
 *   Copyright (C) <2012> <RWTH Aachen University>
 *   Authors: Nils Jansen, Erika Abraham, Jens Katelaan, Maik Scheffler, Matthias Volk, Andreas Vorpahl
 *
 *   This program is free software: you can redistribute it and/or modify
 *   it under the terms of the GNU General Public License as published by
 *   the Free Software Foundation, either version 3 of the License, or
 *   (at your option) any later version.
 *
 *   This program is distributed in the hope that it will be useful,
 *   but WITHOUT ANY WARRANTY; without even the implied warranty of
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *   GNU General Public License for more details.
 *
 *   You should have received a copy of the GNU General Public License
 *   along with this program.  If not, see <http://www.gnu.org/licenses/>.
 *
 *   Main Contact:
 *
 *   Nils Jansen
 *   Theory of Hybrid Systems
 *   RWTH Aachen
 *   52056 Aachen
 *   Germany
 *   nils.jansen@cs.rwth-aachen.de
 *
 */

package comics.io.input;

import java.io.ByteArrayInputStream;
import java.io.IOException;
import java.io.InputStream;
import javax.xml.parsers.ParserConfigurationException;
import javax.xml.parsers.SAXParserFactory;

import org.xml.sax.Attributes;
import org.xml.sax.SAXException;
import org.xml.sax.helpers.DefaultHandler;

import javax.xml.parsers.SAXParser;

import comics.SCC_MC;
import comics.data.ModelCheckResult;
import comics.graph.data.MarkovChain;
import comics.graph.data.Node;
import comics.graph.data.ProbabilityExceededException;
import comics.utilities.Output;

public class XmlParser extends InputParser {

	private String xmldoc;

	public XmlParser(String xmldoc) {
		super(null); // We don't parse a file here, as we get the results
						// directly via JNI!
		this.xmldoc = xmldoc;
	}

	@Override
	public boolean parse() {
		InputStream in = new ByteArrayInputStream(xmldoc.getBytes());
		SAXParserFactory factory = SAXParserFactory.newInstance();
		try {
			SAXParser saxParser = factory.newSAXParser();
			DTMCHandler handler = new DTMCHandler();
			saxParser.parse(in, handler);
			result = handler.getResult();
			return result != null;
		} catch (ParserConfigurationException e) {
			e.printStackTrace();
			return false;
		} catch (SAXException e) {
			e.printStackTrace();
			return false;
		} catch (IOException e) {
			e.printStackTrace();
			return false;
		}
	}

	private class DTMCHandler extends DefaultHandler {
		// The MC that we currently assemble
		MarkovChain mc = null;
		// Needed to find node references for the edges
		MarkovChain toplevelMC = null;
		char nextNodeType;
		// If top level MC, we have to create the nodes; if not, we have to find
		// them in the parent MC to avoid duplication
		boolean isToplevel = true;

		// Positions of nodes
		double xpos, ypos;

		public MarkovChain getResult() {
			return mc;
		}

		@Override
		public void startDocument() {
			// Nothing
		}

		@Override
		public void endDocument() {
			Node initial = toplevelMC.getInputNodes().first();
			// Check if we have concrete graph
			if (toplevelMC.getSubgraphs().size() == 0) {
				toplevelMC.setReduced(false);
				toplevelMC.setInputOfSccNode(initial, false);
				toplevelMC.setInitialNode(initial);
			} else {
				toplevelMC.setInitialNode(initial);
			}
		}

		@Override
		public void startElement(String uri, String localName, String qName, Attributes atts) {
			// Reset
			xpos = -1;
			ypos = -1;
			if (qName.equals("vtx")) {
				nextNodeType = 'v';
				setPosition(atts);
			} else if (qName.equals("inp")) {
				nextNodeType = 'i';
				setPosition(atts);
			} else if (qName.equals("out")) {
				nextNodeType = 'o';
				setPosition(atts);
			} else if (qName.equals("target")) {
				nextNodeType = 't';
			} else if (qName.equals("prob")) {
				nextNodeType = 'p';
			} else if (qName.equals("edg")) {
				nextNodeType = '-';

				// Edges are only found after all nodes have been parsed,
				// so we can safely call the Edge constructor (which relies on
				// finding source/target Node objects)
				boolean isReduced = false;
				int src = -1;
				int trg = -1;
				// TODO: Allow for exact arithmetic!
				double probability = -1d;

				for (int i = 0; i < atts.getLength(); i++) {
					if (atts.getQName(i).equals("abs")) {
						isReduced = atts.getValue(i).equals("1");
					} else if (atts.getQName(i).equals("src")) {
						src = Integer.parseInt(atts.getValue(i));
					} else if (atts.getQName(i).equals("trg")) {
						trg = Integer.parseInt(atts.getValue(i));
					} else if (atts.getQName(i).equals("prb")) {
						probability = Double.parseDouble(atts.getValue(i));
					}
				}

				try {
					mc.addEdge(toplevelMC.getNodeByIntName(src), toplevelMC.getNodeByIntName(trg), probability, isReduced);
				} catch (ProbabilityExceededException exception) {
					Output.print(exception.toString());
				}
			} else if (qName.equals("scc")) {
				nextNodeType = '-';

				// First SCC --> global graph
				if (mc == null) {
					mc = new MarkovChain();
					toplevelMC = mc;
				}
				// Create subgraph
				else {
					MarkovChain child = new MarkovChain();
					if (atts.getLength() > 0) {
						child.setId(Integer.parseInt(atts.getValue(0)));
					}
					mc.addSubgraph(child);
					child.setParent(mc);
					// MC should always hold the current (sub)graph
					mc = child;
					// Not top level any more --> important info for node
					// parsing (see below)
					isToplevel = false;
				}
				// We only have reduced DTMCs, as we are parsing the model
				// checking result!
				mc.setReduced(true);

				// Set graph id
				for (int i = 0; i < atts.getLength(); i++) {
					if (atts.getQName(i).equals("id")) {
						mc.setId(Integer.parseInt(atts.getValue(i)));
					}
				}
			} else {
				nextNodeType = '-';
			}
		}

		@Override
		public void endElement(String uri, String localName, String qName) {
			// If we are leaving an SCC, our current MC reference has to be set
			// back to the parent MC, if existent
			if (qName.equals("scc")) {
				if (mc.getParent() != null) {
					mc = mc.getParent();
				}
			}
		}

		@Override
		public void characters(char[] ch, int start, int length) {
			// Get probability from model checking
			if (nextNodeType == 'p') {
				String sid = "";
				for (int i = start; i < (start + length); i++) {
					sid += ch[i];
				}
				double prob = Double.parseDouble(sid);
				// String is model check result
				ModelCheckResult result = new ModelCheckResult(xmldoc, prob);
				SCC_MC.getInstance().setModelCheckResult(result);
				nextNodeType = '-';
				return;
			}
			// We are only interested in (input/output/general) vertex content
			if (nextNodeType == '-')
				return;

			// We have a node --> first compute node id
			// JK: Inefficient! (Could compute directly by casting chars to
			// digits and raising to powers)
			String sid = "";
			for (int i = start; i < (start + length); i++) {
				sid += ch[i];
			}
			int nodeId = Integer.parseInt(sid);

			Node node = null;
			// If we are in the top level graph, we have to create the node
			// object...
			if (isToplevel && nextNodeType != 't' && !mc.containsNode(nodeId)) {
				node = mc.addNode(nodeId);
				if (xpos >= 0 && ypos >= 0) {
					node.setPosition(xpos, ypos);
				}
			}
			// ...if not, we have to get the node reference from the parent MC
			else {
				node = toplevelMC.getNodeByIntName(nodeId);
				if (node == null) {
					// JK: The current implementation of getNodeByIntName
					// doesn't allow this case,
					// but I think we should change that
					Output.print("Should not be reached");
				}
			}

			// Node reference obtained --> add as general/input/output node
			if (nextNodeType == 'i') {
				mc.setInputOfSccNode(node);
				mc.setCorrespondingMarkovChainNode(node);
			} else if (nextNodeType == 'o') {
				mc.setOutputOfSccNode(node);
			} else if (nextNodeType == 'v') {
				if (!node.isInputOfScc()) {
					mc.setNormalNode(node);
				}
			} else if (nextNodeType == 't') {
				mc.setTargetNode(node);
			}

			// To suppress possible parsing of whitespace following the element
			nextNodeType = '-';
		}

		private void setPosition(Attributes atts) {
			if (isToplevel) {
				for (int i = 0; i < atts.getLength(); i++) {
					if (atts.getQName(i).equals("xpos")) {
						xpos = Double.parseDouble(atts.getValue(i));
					} else if (atts.getQName(i).equals("ypos")) {
						ypos = Double.parseDouble(atts.getValue(i));
					}
				}
			}
		}
	}
}
