/**
 *   COMICS - Computing Minimal Counterexamples for Discrete-time Markov Chains
 *
 *   COMICS is a stand-alone tool which performs model checking and the generation
 *   of counterexamples for discrete-time Markov Chains (DTMCs). *
 *
 *   Copyright (C) <2012> <RWTH Aachen University>
 *   Authors: Nils Jansen, Erika Abraham, Jens Katelaan, Maik Scheffler, Matthias Volk, Andreas Vorpahl
 *
 *   This program is free software: you can redistribute it and/or modify
 *   it under the terms of the GNU General Public License as published by
 *   the Free Software Foundation, either version 3 of the License, or
 *   (at your option) any later version.
 *
 *   This program is distributed in the hope that it will be useful,
 *   but WITHOUT ANY WARRANTY; without even the implied warranty of
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *   GNU General Public License for more details.
 *
 *   You should have received a copy of the GNU General Public License
 *   along with this program.  If not, see <http://www.gnu.org/licenses/>.
 *
 *   Main Contact:
 *
 *   Nils Jansen
 *   Theory of Hybrid Systems
 *   RWTH Aachen
 *   52056 Aachen
 *   Germany
 *   nils.jansen@cs.rwth-aachen.de
 *
 */

package comics.io.output;

import java.io.File;
import com.mxgraph.model.mxGeometry;

import comics.graph.data.Edge;
import comics.graph.data.MarkovChain;
import comics.graph.data.Node;
import comics.utilities.Output;

public class XmlWriter extends OutputFormatter {
	private boolean keepAbstract;
	private double probModelCheck;
	private String lb;

	/**
	 * Create and set file null for only exporting as string
	 * 
	 * @param markovChain
	 *            MarkovChain to write
	 * @param keepAbstract
	 *            if false, all expanded SCCs are written as concrete SCCs, i.e.
	 *            they can not be reduced anymore
	 * @param probModelCheck
	 *            probability from modelChecking
	 */
	public XmlWriter(MarkovChain markovChain, boolean keepAbstract, double probModelCheck) {
		this(markovChain, null, keepAbstract, probModelCheck);
		lb = Output.getLineBreak();
	}

	/**
	 * Create
	 * 
	 * @param markovChain
	 *            MarkovChain to write
	 * @param file
	 *            File to write
	 * @param keepAbstract
	 *            if false, all expanded SCCs are written as concrete SCCs, i.e.
	 *            they can not be reduced anymore
	 * @param probModelCheck
	 *            probability from modelChecking
	 */
	public XmlWriter(MarkovChain markovChain, File file, boolean keepAbstract, double probModelCheck) {
		super(markovChain, file);
		this.keepAbstract = keepAbstract;
		this.probModelCheck = probModelCheck;
		lb = Output.getLineBreak();
	}

	@Override
	protected String generateString(MarkovChain mc) {
		return markovChainToXMLString(mc);
	}

	/**
	 * Export markov chain to xml string
	 * 
	 * @return String in xml format containing the markov chain
	 */
	public String markovChainToXMLString(MarkovChain mc) {
		String str = "";
		str += "<?xml version=" + "\"" + "1.0" + "\"" + " encoding=" + "\"" + "utf-8" + "\"" + "?>";
		str += lb + "<dtmc>" + lb;
		// Write labels
		//str += printLabels(mc, "  ");

		// Traverse markov chain recursively
		str += recursiveTransformation(mc, 1);
		// Write target states explicitly
		for (Node n : mc.getTargetNodes()) {
			if (n.isTarget()) {
				str += "  <target>" + n.getNumber() + "</target>" + lb;
			}
		}

		// Write probability from model checking
		str += "  <prob>" + probModelCheck + "</prob>" + lb;

		str += "</dtmc>";

		return str;

	}

	/**
	 * Returns a line with an edge tag representing the given edge.
	 * 
	 * @param e
	 *            Edge
	 * @return edge tag
	 */
	String edgeTag(Edge e) {
		String s = "";
		s += "<edg abs=\"" + (e.isReduced() ? 1 : 0);
		s += "\" prb=\"" + e.getProbability();
		s += "\" src=\"" + e.getSourceNode().getNumber();
		s += "\" trg=\"" + e.getTargetNode().getNumber();
		s += "\" />" + lb;
		return s;
	}

	/**
	 * Traverses the graph structure recursively and maps it to the XML format.
	 * 
	 * @param mc
	 *            MarkovChain to travers
	 * @param depth
	 *            is used to keep track of the indentation
	 * @return String in xml format
	 */
	String recursiveTransformation(MarkovChain mc, int depth) {
		String str = "";
		String indent = "";
		for (int i = 1; i < depth; i++) {
			indent += "  ";
		}

		if (keepAbstract || mc.getRecursionDepth() == 0 || !mc.isExpandedInGraph()) {
			// Only new SCC if root or reduced or abstract SCCs should be kept
			indent += "  ";
			str += indent + "<scc";
			str += " id=\"" + mc.getId() + "\"";

			// Add initial nodes for better debugging
			int i = 0;
			if (mc.getInitialNode() != null) {
				str += " node0=\"" + mc.getInitialNode().getNumber() + "\"";
			} else {
				for (Node n : mc.getInputNodes()) {
					str += " node" + i + "=\"" + n.getNumber() + "\"";
					i++;
				}
			}

			str += ">" + lb;

			// (1) Vertices
			if (mc.isReduced()) {
				for (Node n : mc.getInputNodes()) {
					str += indent + "  " + "<inp";
					if (mc.getRecursionDepth() == 0) {
						str += getXMLPos(n);
					}
					str += ">";
					str += n.getNumber() + "</inp>" + lb;
				}
			} else {
				str += indent + "  " + "<inp";
				if (mc.getRecursionDepth() == 0) {
					str += getXMLPos(mc.getInitialNode());
				}
				str += ">";
				str += mc.getInitialNode() + "</inp>" + lb;
			}
			for (Node n : mc.getTargetNodes()) {
				str += indent + "  " + "<out";
				if (mc.getRecursionDepth() == 0) {
					str += getXMLPos(n);
				}
				str += ">";
				str += n.getNumber() + "</out>" + lb;
			}
			for (Node n : mc.getNodes()) {
				// Write not input and target nodes
				if (!mc.getInputNodes().contains(n) && !n.equals(mc.getInitialNode()) && !mc.getTargetNodes().contains(n)) {
					str += indent + "  " + "<vtx";
					if (mc.getRecursionDepth() == 0) {
						str += getXMLPos(n);
					}
					str += ">";
					str += n.getNumber() + "</vtx>" + lb;
				}
			}
		}
		// (2) Subgraphs
		for (MarkovChain subgraph : mc.getSubgraphs()) {
			str += recursiveTransformation(subgraph, depth + 1);
		}

		// (3) Edges
		// Concrete edges
		for (Edge e : mc.getEdges()) {
			str += indent + "  " + edgeTag(e);
		}
		// Abstract edges
		if (mc.isReduced()) {
			if (keepAbstract || !mc.isExpandedInGraph()) {
				for (Edge e : mc.getReducedEdges()) {
					str += indent + "  " + edgeTag(e);
				}
			}
		}

		if (keepAbstract || mc.getRecursionDepth() == 0 || !mc.isExpandedInGraph()) {
			str += indent + "</scc>" + lb;
		}

		return str;
	}

	private String getXMLPos(Node n) {
		if (n.getGraphCell() == null) {
			return "";
		}
		mxGeometry geometry = n.getGraphCell().getGeometry();
		String s = " xpos=\"" + geometry.getX() + "\" ";
		s += "ypos=\"" + geometry.getY() + "\"";
		return s;
	}

	private String printLabels(MarkovChain mc, String indent) {
		String s = indent + "<labels>" + lb;

		// Print all defined labels
		s += indent + "  <declaration>" + lb;
		for (String label : mc.getLabels()) {
			s += indent + "    " + "<lab>" + label + "</lab>" + lb;
		}
		s += indent + "  </declaration>" + lb;

		// Print nodes with their labels
		for (Node n : mc.getNodes()) {
			s += indent + "  <lbvtx id=\"" + n.getNumber() + "\">" + lb;
			for (String label : mc.getLabelsNode(n)) {
				s += indent + "    " + "<lab>" + label + "</lab>" + lb;
			}
			s += indent + "  </lbvtx>" + lb;
		}
		s += indent + "</labels>" + lb;
		return s;
	}
}