/**
 *   COMICS - Computing Minimal Counterexamples for Discrete-time Markov Chains
 *
 *   COMICS is a stand-alone tool which performs model checking and the generation
 *   of counterexamples for discrete-time Markov Chains (DTMCs). *
 *
 *   Copyright (C) <2012> <RWTH Aachen University>
 *   Authors: Nils Jansen, Erika Abraham, Jens Katelaan, Maik Scheffler, Matthias Volk, Andreas Vorpahl
 *
 *   This program is free software: you can redistribute it and/or modify
 *   it under the terms of the GNU General Public License as published by
 *   the Free Software Foundation, either version 3 of the License, or
 *   (at your option) any later version.
 *
 *   This program is distributed in the hope that it will be useful,
 *   but WITHOUT ANY WARRANTY; without even the implied warranty of
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *   GNU General Public License for more details.
 *
 *   You should have received a copy of the GNU General Public License
 *   along with this program.  If not, see <http://www.gnu.org/licenses/>.
 *
 *   Main Contact:
 *
 *   Nils Jansen
 *   Theory of Hybrid Systems
 *   RWTH Aachen
 *   52056 Aachen
 *   Germany
 *   nils.jansen@cs.rwth-aachen.de
 *
 */

package comics.tools;

import java.util.ArrayList;
import java.util.HashMap;
import java.util.HashSet;

import prismparser.casestudies.*;
import prismparser.external.*;
import comics.graph.data.Node;
import comics.graph.data.ProbabilityExceededException;
import comics.utilities.Either;
import comics.utilities.Output;

/**
 * This is the interface to the PRISM parser. Just call one of the various
 * parseModel / getModel methods. Note: Parsing may take several seconds, so it
 * might make sense to execute the PrismParser methods in a separate thread.
 * 
 * @author jens
 */
public class PrismParser {

	/**
	 * Parse the model defined by the given CaseStudyGenerator
	 * 
	 * @param csg
	 *            The object that generates the model code
	 * @return Either a list of errors or a MarkovChain corresponding to the
	 *         model
	 */
	public Either<ArrayList<String>, comics.graph.data.MarkovChain> parseModel(CaseStudyGenerator csg) {
		ParserCall pc = new ParserCall(csg);
		return coordinateParsing(pc);
	}

	/**
	 * Generate a DTMC for the Crowds protocol.
	 * 
	 * @param crowdSize
	 *            Members in the crowd
	 * @param totalRuns
	 *            Number of runs to simulate
	 * @return Either a list of errors or a MarkovChain corresponding to the
	 *         model
	 */
	public Either<ArrayList<String>, comics.graph.data.MarkovChain> getCrowdsModel(int crowdSize, int totalRuns) {
		return parseModel(new CrowdsGenerator(crowdSize, totalRuns));
	}

	/**
	 * Generate a DTMC for the Synchronous Leader Election protocol.
	 * 
	 * @param N
	 *            Number of processes
	 * @param K
	 *            Range of probabilistic choice
	 * @return Either a list of errors or a MarkovChain corresponding to the
	 *         model
	 */
	public Either<ArrayList<String>, comics.graph.data.MarkovChain> getLeaderModel(int N, int K) {
		return parseModel(new SyncLeaderGenerator(N, K));
	}

	/**
	 * Construct a model from a PRISM model specification only (i.e. without
	 * passing in a property) Note: The resulting DTMC will not have any target
	 * states if you don't pass a property
	 * 
	 * @param prismmodel
	 *            Model specification in the PRISM language
	 * @return Either a list of errors or a MarkovChain corresponding to the
	 *         model
	 */
	public Either<ArrayList<String>, comics.graph.data.MarkovChain> parseModel(String prismmodel) {
		return parseModel(prismmodel, "");
	}

	/**
	 * Construct a model from a PRSIM model specification and a property.
	 * 
	 * @param prismmodel
	 *            Model specification in the PRISM language
	 * @param property
	 *            A property -- either a simple condition expression
	 *            (representing ttUa) or (condition1) U (condition2)
	 *            representing aUb
	 * @return Either a list of errors or a MarkovChain corresponding to the
	 *         model
	 */
	public Either<ArrayList<String>, comics.graph.data.MarkovChain> parseModel(String prismmodel, String property) {
		ParserCall pc = new ParserCall(prismmodel, property);
		return coordinateParsing(pc);
	}

	/**
	 * The actual work is done here (i.e., parser is called, result is
	 * transformed to MarkovChain
	 * 
	 * @param pc
	 *            The parser call to be executed
	 * @return Either a list of errors or a MarkovChain corresponding to the
	 *         model
	 */
	private Either<ArrayList<String>, comics.graph.data.MarkovChain> coordinateParsing(ParserCall pc) {
		MarkovChain scalaDTMC = pc.runParser();
		if (scalaDTMC == null) {
			return new Either<ArrayList<String>, comics.graph.data.MarkovChain>(pc.errors(), null);
		} else {
			Output.print("The following Markov Chain was generated:");
			Output.print(scalaDTMC.toString());

			comics.graph.data.MarkovChain javaDTMC = transformMarkovChain(scalaDTMC);

			return new Either<ArrayList<String>, comics.graph.data.MarkovChain>(null, javaDTMC);
		}
	}

	/**
	 * Construct a comics MarkovChain from the given PRISM parser MarkovChain.
	 * 
	 * @param scalaDTMC
	 *            Parser result
	 * @return Corresponding comics MarkovChain representation
	 */
	private comics.graph.data.MarkovChain transformMarkovChain(MarkovChain scalaDTMC) {
		comics.graph.data.MarkovChain dtmc = new comics.graph.data.MarkovChain();

		ArrayList<Vertex> vertices = scalaDTMC.vertices();
		HashSet<Integer> initStates = scalaDTMC.initStates();
		HashSet<Integer> targetStates = scalaDTMC.targetStates();

		HashMap<Integer, comics.graph.data.Node> nodeBuffer = new HashMap<Integer, comics.graph.data.Node>();

		// Node construction
		for (Vertex v : vertices) {
			// Add nodes to DTMC
			Node node = dtmc.addNode(v.getId());
			// Also assemble map from node ids to nodes --> makes adding edges
			// and set init/target properties easier
			nodeBuffer.put(v.getId(), node);
		}

		// Node properties
		if (initStates != null) {
			for (int iv : initStates) {
				comics.graph.data.Node n = nodeBuffer.get(iv);
				dtmc.setInitialNode(n);
			}
		}
		if (targetStates != null) {
			for (int tv : targetStates) {
				comics.graph.data.Node n = nodeBuffer.get(tv);
				dtmc.setTargetNode(n);
			}
		}

		// Edge construction
		for (Vertex v : vertices) {
			comics.graph.data.Node src = nodeBuffer.get(v.getId());
			for (Edge e : v.getEdges()) {
				comics.graph.data.Node trg = nodeBuffer.get(e.getTarget());
				try {
					dtmc.addEdge(src, trg, e.getProbability());
				} catch (ProbabilityExceededException exception) {
					Output.print(exception.toString());
				}
			}
		}

		return dtmc;
	}

}
