/**
 *   COMICS - Computing Minimal Counterexamples for Discrete-time Markov Chains
 *
 *   COMICS is a stand-alone tool which performs model checking and the generation
 *   of counterexamples for discrete-time Markov Chains (DTMCs). *
 *
 *   Copyright (C) <2012> <RWTH Aachen University>
 *   Authors: Nils Jansen, Erika Abraham, Jens Katelaan, Maik Scheffler, Matthias Volk, Andreas Vorpahl
 *
 *   This program is free software: you can redistribute it and/or modify
 *   it under the terms of the GNU General Public License as published by
 *   the Free Software Foundation, either version 3 of the License, or
 *   (at your option) any later version.
 *
 *   This program is distributed in the hope that it will be useful,
 *   but WITHOUT ANY WARRANTY; without even the implied warranty of
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *   GNU General Public License for more details.
 *
 *   You should have received a copy of the GNU General Public License
 *   along with this program.  If not, see <http://www.gnu.org/licenses/>.
 *
 *   Main Contact:
 *
 *   Nils Jansen
 *   Theory of Hybrid Systems
 *   RWTH Aachen
 *   52056 Aachen
 *   Germany
 *   nils.jansen@cs.rwth-aachen.de
 *
 */

package comics.tools.counterexample;

import comics.SCC_MC;
import comics.data.CounterExampleResult;
import comics.data.CounterExampleSearchParameters;
import comics.data.ModelCheckResult;
import comics.graph.data.InconsistentMarkovChainException;
import comics.graph.data.MarkovChain;
import comics.utilities.Output;

public class SccModelChecker {

	private JniMethods nativeInterface = null;

	private boolean wasModelChecked = false, wasCexInitialized = false, wasCexStarted = false, wasCexCompleted = false;

	public SccModelChecker() {
		nativeInterface = new JniMethods();
	}

	/**
	 * Model check
	 * 
	 * @param mc
	 *            markov chain
	 * @return result with xml string for graph and probability
	 */
	public ModelCheckResult modelCheck(MarkovChain mc) {
		try {
			String inputData = mc.to_scc_mc_string();
			// Couldn't load data -> Return
			if (!nativeInterface.loadData(inputData)) {
				return null;
			}
		} catch (InconsistentMarkovChainException e) {
			return null;
		}
		return modelCheck();
	}

	/**
	 * Model check
	 * 
	 * @param filename
	 *            file name with dtmc file
	 * @return result with xml string for graph and probability
	 */
	public ModelCheckResult modelCheck(String filename) {
		// Couldn't load data -> Return
		if (!nativeInterface.loadFile(filename)) {
			return null;
		}
		return modelCheck();
	}

	/**
	 * Model check. Markov chain or file must be set correctly before
	 * 
	 * @return result with xml string for graph and probability
	 */
	private ModelCheckResult modelCheck() {
		String resultChain = nativeInterface.modelCheck();
		double resultValue = 0;
		if (resultChain != null && !resultChain.equals("")) {
			// Modelchecking was successful -> get probability as well
			resultValue = nativeInterface.getResultProbability();
			wasModelChecked = true;
			return new ModelCheckResult(resultChain, resultValue);
		} else {
			return null;
		}
	}

	/**
	 * Initialize counter example search
	 * 
	 * @param xmlGraph
	 *            string with xml representation of markov chain
	 * @param parameters
	 *            parameters for probability, abstract or concrete search,
	 *            global or local search path, subsystem- or path-based counter
	 *            example
	 * @return true, if initialization was successful
	 */
	public boolean initCounterExampleSearch(String xmlGraph, CounterExampleSearchParameters parameters) {
		if (!wasModelChecked) {
			if (SCC_MC.getInstance().getModelCheckResult() != null) {
				if (SCC_MC.getInstance().getModelCheckResult().getProbability() > 0) {
					wasModelChecked = true;
				} else {
					return false;
				}
			} else {
				return false;
			}
		}
		if (parameters.getProbabilityBound() > SCC_MC.getInstance().getModelCheckResult().getProbability()) {
			return false;
		}

		boolean result = nativeInterface.initCounterExampleSearch(xmlGraph, parameters.getProbabilityBound(),
				parameters.isAbstractSearch(), parameters.isGlobalSearch(), parameters.isClosure());

		wasCexInitialized = result;
		return wasCexInitialized;
	}

	/**
	 * Search for counter example
	 * 
	 * @param parameters
	 *            Several parameters (StepCount, user input for concretized
	 *            nodes, heuristic, ...)
	 * @return
	 */
	public CounterExampleResult doCounterExampleSteps(CounterExampleSearchParameters parameters) {
		if (!wasCexStarted && !wasCexInitialized) {
			Output.print("Initialization failed -> can't do a thing");
			return null;
		}
		if (isCounterExampleComplete()) {
			Output.print("Counterexample already complete -> nothin left to do");
			return null;
		}

		// Actually execute the steps
		int heuristic = parameters.getHeuristic().getNo();
		int concretizeMode = parameters.getConcretizationMode().getNo();
		String result = nativeInterface.doCounterExampleSteps(parameters.getConcretizedNodes(),
				!parameters.isNoAutoConcretize(), parameters.isStepsTillEnd(), parameters.getStepSize(), concretizeMode,
				heuristic);
		if (result == null) {
			return null;
		} else {
			CounterExampleResult counterExampleResult = new CounterExampleResult();
			if (result.contains("No step")) {
				// No step was done
				counterExampleResult.setSteps(0);
				return counterExampleResult;
			} else {
				// Successfully done steps in search
				wasCexStarted = true;
				counterExampleResult.setMarkovChain(result);
				if (!parameters.isAbstractSearch()) {
					// Concrete search is done till end -> complete
					wasCexCompleted = true;
				}
				counterExampleResult.setComplete(isCounterExampleComplete());
				counterExampleResult.setSteps(getCounterExampleSteps());
				counterExampleResult.setProbability(getCounterExampleProbability());
				return counterExampleResult;
			}
		}
	}

	public boolean isCounterExampleComplete() {
		if (!wasCexCompleted) {
			wasCexCompleted = nativeInterface.isCounterExampleComplete();
		}

		return wasCexCompleted;
	}

	public int getCounterExampleSteps() {
		return nativeInterface.getCounterExampleSteps();
	}

	public double getCounterExampleProbability() {
		return nativeInterface.getCounterExampleProbability();
	}

	public void cleanup() {
		nativeInterface.destroy();
		nativeInterface = null;
	}
}
