/**
 * Parametric SCC based Model Checking
 *  
 * This is a stand-alone tool which performs model checking
 * for parametric discrete-time Markov Chains (PDTMCs).
 * 
 * Copyright (c) 2013 RWTH Aachen University.
 * Authors: Florian Corzilius, Nils Jansen, Matthias Volk
 * 
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 * 
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the
 * GNU General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public License
 * along with this program.  If not, see
 * http://www.gnu.org/licenses/gpl.html.
 * 
 * 
 * Main Contact:
 * 
 * Nils Jansen
 * Theory of Hybrid Systems
 * RWTH Aachen
 * 52056 Aachen
 * Germany
 * nils.jansen@cs.rwth-aachen.de
 */

#include "AbstractParameter.h"
#include "Cancellator2.h"
#include "Parameters.h"
#include "../defines.h"
#include <sstream>

using namespace std;

namespace parametric {

unsigned int AbstractParameter::count = 0;
exmap AbstractParameter::substitutions;

/**
 * Constructor
 * @param rat underlying rational functions for this abstracted transition
 * @return
 */
AbstractParameter::AbstractParameter(const Rational& rat) {
	stringstream stream;
	stream << "abs" << count;
	string paramString = stream.str();
	parameterSymbol = symbol(paramString);
	parameterRational = Rational(Parameters::getInstance().createPolynomial(parameterSymbol));
	value = rat;
	count++;

	//Simplify
	Cancellator::getInstance().cancel(value, true);
}

/**
 * Destructor
 * @return
 */
AbstractParameter::~AbstractParameter() {
}

/**
 * Output operator
 * @param stream
 * @param parameter
 * @return
 */
std::ostream &operator<<(std::ostream& stream, const AbstractParameter& parameter) {
	stream << parameter.parameterSymbol;
	stream << " := " << parameter.value.toString();
	return stream;
}

/**
 * Getter for all abstracted parameters and their corresponding rational functions
 * @return substitutions
 */
const exmap& AbstractParameter::getSubstitutions() {
	vector<AbstractParameter*> abstractParameters = Parameters::getInstance().getAbstractParameters();
	for (unsigned int i = substitutions.size(); i < AbstractParameter::count; i++) {
		AbstractParameter* tmpAbsParam = abstractParameters[i];
		substitutions[tmpAbsParam->parameterSymbol] = tmpAbsParam->value.getExpression();
	}
	return substitutions;
}

/**
 * Returns string with prefix notation for some tools as input
 * @return prefix notation
 */
std::string AbstractParameter::toPrefixNotation() const {
	std::stringstream stream;
	stream << "=";
	stream << " " << parameterRational.toPrefixNotation();
	stream << " " << value.toPrefixNotation();
	return stream.str();
}

}
