/**
 * Parametric SCC based Model Checking
 *  
 * This is a stand-alone tool which performs model checking
 * for parametric discrete-time Markov Chains (PDTMCs).
 * 
 * Copyright (c) 2013 RWTH Aachen University.
 * Authors: Florian Corzilius, Nils Jansen, Matthias Volk
 * 
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 * 
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the
 * GNU General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public License
 * along with this program.  If not, see
 * http://www.gnu.org/licenses/gpl.html.
 * 
 * 
 * Main Contact:
 * 
 * Nils Jansen
 * Theory of Hybrid Systems
 * RWTH Aachen
 * 52056 Aachen
 * Germany
 * nils.jansen@cs.rwth-aachen.de
 */

#include "Constraint.h"
#include <sstream>

namespace parametric {

/**
 * Output
 * @param stream
 * @param constraint
 * @return
 */
std::ostream &operator<<(std::ostream &stream, const Constraint& constraint) {
	stream << constraint.rational1;
	switch (constraint.operatorType) {
	case OPERATOR_LEQ:
		stream << " <= ";
		break;
	case OPERATOR_LESS:
		stream << " < ";
		break;
	case OPERATOR_EQ:
		stream << " == ";
		break;
	}
	stream << constraint.rational2;
	return stream;
}

/**
 * Output in prefix notation for some tools
 * @return string with constraint in prefix notation
 */
std::string Constraint::toPrefixNotation() const {
	std::stringstream stream;
	switch (operatorType) {
	case OPERATOR_LEQ:
		stream << "<=";
		break;
	case OPERATOR_LESS:
		stream << "<";
		break;
	case OPERATOR_EQ:
		stream << "=";
		break;
	}
	stream << " " << rational1.toPrefixNotation();
	stream << " " << rational2.toPrefixNotation();
	return stream.str();
}

/**
 * Output for xml (< is &lt;)
 * @return string with constraint
 */
std::string Constraint::toXMLString() const {
	std::stringstream stream;
	stream << rational1;
	switch (operatorType) {
	case OPERATOR_LEQ:
		stream << " &le; ";
		break;
	case OPERATOR_LESS:
		stream << " &lt; ";
		break;
	case OPERATOR_EQ:
		stream << " = ";
		break;
	}
	stream << rational2;
	return stream.str();
}

}
