/**
 * Parametric SCC based Model Checking
 *  
 * This is a stand-alone tool which performs model checking
 * for parametric discrete-time Markov Chains (PDTMCs).
 * 
 * Copyright (c) 2013 RWTH Aachen University.
 * Authors: Florian Corzilius, Nils Jansen, Matthias Volk
 * 
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 * 
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the
 * GNU General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public License
 * along with this program.  If not, see
 * http://www.gnu.org/licenses/gpl.html.
 * 
 * 
 * Main Contact:
 * 
 * Nils Jansen
 * Theory of Hybrid Systems
 * RWTH Aachen
 * 52056 Aachen
 * Germany
 * nils.jansen@cs.rwth-aachen.de
 */

#include "../defines.h"
#include "Factorization.h"
#include "Parameters.h"
#include "Cancellator2.h"
#include <sstream>

namespace parametric {

//#################
//# FACTORIZATION #
//#################

/**
 * Constructor
 * @param pol polynomial
 * @return
 */
Factorization::Factorization(constPoly pol) {
	originalPol = pol;
}

/**
 * Destructor
 * @return
 */
Factorization::~Factorization() {
	factorization.clear();
	originalPol = NULL;
}

/**
 * Output
 * @param stream
 * @param factorization
 * @return
 */
std::ostream &operator<<(std::ostream &stream, const Factorization& factors) {
	if (factors.isEmpty()) {
		//Original polynomial
		Factor fac(factors.originalPol, 1);
		stream << fac;
	} else {
		//Notice: factorization is not refined
		for (factors_const_it iter = factors.factorization.begin(); iter != factors.factorization.end();/*iter++ later*/) {
			Factor fac(iter->first, iter->second);
			stream << "(" << fac << ")";
			iter++;
			if (iter != factors.factorization.end()) {
				stream << "*";
			}
		}
	}
	return stream;
}

/**
 * Insert factor in factorization. If factor already exists, the exponent is increased
 * @param factor new factor to insert
 * @param exponent exponent of factor. Default value is 1
 */
void Factorization::insertFactor(constPoly factor, unsigned int exponent) {
	assert(factor != originalPol);
	factors_it itFound = factorization.find(factor);
	if (itFound != factorization.end()) {
		//Factor already there
		itFound->second += exponent;
	} else {
		//New factor
		factorization.insert(make_pair(factor, exponent));

#ifdef USE_POLY_MANAGEMENT
		//Increment usageCounter
		factor->incUsageCounter();
#endif
	}
}

/**
 * Build factorization tree by adding two factor with pol1*pol2=this
 * @param pol1 polynomial1
 * @param pol2 polynomial2
 */
void Factorization::setFactors(constPoly pol1, constPoly pol2) {
	assert(pol1 != NULL);
	assert(pol2 != NULL);

	//Avoid trivial cases
	if (pol1->isNumber()) {
		numeric num1 = pol1->getNumeric();
		if (num1 == 1 || num1 == 0 || num1 == -1) {
			return;
		}
	}
	if (pol2->isNumber()) {
		numeric num2 = pol2->getNumeric();
		if (num2 == 1 || num2 == 0 || num2 == -1) {
			return;
		}
	}

	if (isEmpty()) {
		insertFactor(pol1, 1);
		insertFactor(pol2, 1);

		assert(checkFactorization());

	} else {
		//TODO implement proper
		assert(checkFactorization());
	}

	//Refine factorization w.r.t. factorization of factors
	//getAllFactors();
}

/**
 * Set factor as power pol^exponent
 * @param pol polynomial
 * @param exponent exponent
 */
void Factorization::setFactorPower(constPoly pol, unsigned int exponent) {
	assert(pol != NULL);
	//Avoid trivial cases

	if (pol->isNumber()) {
		numeric num1 = pol->getNumeric();
		if (num1 == 1 || num1 == 0 || num1 == -1) {
			return;
		}
	}

	if (isEmpty()) {
		insertFactor(pol, exponent);

		assert(checkFactorization());
	} else {
		//TODO implement proper
		assert(checkFactorization());
	}

	//Refine factorization w.r.t. factorization of factors
	//getAllFactors();
}

/**
 * Delete factorization and decrement usageCounter of factors
 */
void Factorization::clearFactorization() {
#ifdef USE_POLY_MANAGEMENT
	for (factors_const_it iter = factorization.begin(); iter != factorization.end(); iter++) {
		iter->first->decUsageCounter();
	}
#endif
	factorization.clear();
}

/**
 * Set factorization from expression
 * @param expr expression
 */
void Factorization::setFactorization(const ex& expr) {
	if (!is_a<mul>(expr) && !is_a<power>(expr)) {
		//No factorization possible
		return;
	}

	clearFactorization();

	//Iterate over all factors
	for (const_iterator iter = expr.begin(); iter != expr.end(); ++iter) {
		if (is_a<power>(*iter)) {
			//Power
			assert(iter->nops() == 2);
			assert(is_a<numeric>(iter->op(1)));
			numeric exponent = ex_to<numeric>(iter->op(1));
			assert(exponent.is_integer());

			constPoly factor = Parameters::getInstance().createPolynomial(iter->op(0));
			factorization.insert(make_pair(factor, exponent.to_int()));
#ifdef USE_POLY_MANAGEMENT
			//Increment usageCounter
			factor->incUsageCounter();
#endif
		} else {
			//No power
			constPoly factor = Parameters::getInstance().createPolynomial(*iter);
			factorization.insert(make_pair(factor, 1));
#ifdef USE_POLY_MANAGEMENT
			//Increment usageCounter
			factor->incUsageCounter();
#endif
		}
	}

	assert(checkFactorization());
}

/**
 * Set factorization from list of factors
 * @param factors list of factors
 */
void Factorization::setFactorization(const FactorList& factors) {
	if (factors.size() <= factorization.size()) {
		//Old factorization seems to be better
		//TODO better merge both factorizations
		return;
	}

	if (factors.size() == 1 && factors.front().pol == originalPol) {
		//No factorization
		return;
	}

	clearFactorization();

	//Iterate over all factors
	for (factorList_const_it iter = factors.begin(); iter != factors.end(); iter++) {
		constPoly factor = iter->pol;
		insertFactor(factor, iter->exponent);
#ifdef USE_POLY_MANAGEMENT
		//Increment usageCounter
		factor->incUsageCounter();
#endif
	}

	assert(checkFactorization());
}

const ex Factorization::getExpressionFromFactorization(const FactorList& factorization) {
	//Construct expression
	ex poly = 1;
	for (factorList_const_it iter = factorization.begin(); iter != factorization.end(); iter++) {
		poly *= pow(iter->pol->getExpression(), iter->exponent);
	}
	return poly;
}

/**
 * Getter
 * @param refine if true, factorization will be refined before
 * @return copy of factorization as FactorMap
 */
//TODO everything okay?
FactorMap Factorization::getFactors(bool refine) {
	if (refine) {
		refineFactorization();
	}
	FactorMap map;
	if (isEmpty()) {
		map.insert(make_pair(originalPol, 1));
	} else {
		for (factors_const_it iter = factorization.begin(); iter != factorization.end(); iter++) {
			map.insert(make_pair(iter->first, iter->second));
		}
	}
	return map;
}

/**
 * Refine factorization. The internal data structure is updated, if factors themselves have a factorization.
 */
void Factorization::refineFactorization() {
	if (isEmpty()) {
		return;
	}

	//Store all factors, which have own factorization
	vector<factors_const_it> iterators;
	for (factors_const_it iter = factorization.begin(); iter != factorization.end(); iter++) {
		if (!iter->first->getFactorization()->isEmpty()) {
			//Factor has factorization for itself
			iterators.push_back(iter);
		}
	}

	FactorMap tmpFactors;
	unsigned int exponent = 1;
	//Update these factors with their factorization
	for (vector<factors_const_it>::iterator iter = iterators.begin(); iter != iterators.end(); iter++) {
		(*iter)->first->getFactorization()->refineFactorization();
		tmpFactors = (*iter)->first->getFactorization()->factorization;
		exponent = (*iter)->second;

		//Insert factorization of factor
		for (factors_const_it iter2 = tmpFactors.begin(); iter2 != tmpFactors.end(); iter2++) {
			factors_it itFound = factorization.find(iter2->first);
			if (itFound != factorization.end()) {
				//Factor already there
				itFound->second += iter2->second * exponent;
			} else {
				//New factor
				factorization.insert(make_pair(iter2->first, iter2->second * exponent));
#ifdef USE_POLY_MANAGEMENT
				iter2->first->incUsageCounter();
#endif
			}
		}
		//Decrement usageCounter
#ifdef USE_POLY_MANAGEMENT
		(*iter)->first->decUsageCounter();
#endif
		//Erase old factor
		factorization.erase((*iter));
	}

	assert(checkFactorization());
}

/**Check factorization for correctness
 * @return true, if correct, false otherwise
 */
bool Factorization::checkFactorization() const {
	if (isEmpty()) {
		return true;
	} else {
		ex expr = 1;
		for (factors_const_it iter = factorization.begin(); iter != factorization.end(); iter++) {
			expr *= pow(iter->first->getExpression(), iter->second);
#ifdef USE_POLY_MANAGEMENT
			assert(iter->first->getUsageCounter() > 0);
#endif
		}
		return expr.expand() == originalPol->getExpression().expand();
	}
}

//##########
//# FACTOR #
//##########

/**
 * Constructor
 * @param pol polynomial
 * @param exponent exponent
 * @return
 */
Factor::Factor(constPoly pol, unsigned int exponent) :
		pol(pol), exponent(exponent) {
}

/**
 * Destructor
 * @return
 */
Factor::~Factor() {
}

/**
 * Output of factor with brackets
 * @param factor factor
 * @return string
 */
std::ostream &operator<<(std::ostream &stream, const Factor& factor) {
	return stream << factor.toString();
}

/**
 * Output
 * @return string with factor
 */
string Factor::toString() const {
	stringstream stream;
	bool bracket = (exponent > 1) && (pol->getExpression().nops() > 0);
	if (bracket) {
		stream << "(";
	}
	stream << pol->getExpression();
	if (bracket) {
		stream << ")";
	}
	if (exponent > 1) {
		stream << "^" << exponent;
	}

	return stream.str();
}
}
