/**
 * Parametric SCC based Model Checking
 *  
 * This is a stand-alone tool which performs model checking
 * for parametric discrete-time Markov Chains (PDTMCs).
 * 
 * Copyright (c) 2013 RWTH Aachen University.
 * Authors: Florian Corzilius, Nils Jansen, Matthias Volk
 * 
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 * 
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the
 * GNU General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public License
 * along with this program.  If not, see
 * http://www.gnu.org/licenses/gpl.html.
 * 
 * 
 * Main Contact:
 * 
 * Nils Jansen
 * Theory of Hybrid Systems
 * RWTH Aachen
 * 52056 Aachen
 * Germany
 * nils.jansen@cs.rwth-aachen.de
 */

#include "GraphTypes.h"

using namespace std;

namespace scc_cex {

bool IsMarked(const Edge<PTYPE>* arg) { return arg->getMark(); }

bool KeepAfterModelchecking(const Edge<PTYPE>* arg) { return arg->isConcrete() && arg->target->oindex != UINT_MAX; };

/* --------------- Tracing / Debugging functions ------------------ */

template <class T>
string vertexToString(const Vertex<T>& v) {
	stringstream ss;
	ss << v.oindex;
	return ss.str();
}

template <class T>
string edgeToString(const Edge<T>& e) {
	stringstream ss;
	if (e.state==ABSTRACT) ss << "^";
	ss << "(";
	ss << vertexToString(*(e.source));
	ss << ",";
	ss << vertexToString(*(e.target));
	ss << "):";
	ss << e.prob << " ";
	return ss.str();
}

template <class T>
string graphToString(const Graph<T>& g) {
	return graphToString(g, 0);
}

template <class T>
string graphToString(const Graph<T>& g, const int depth) {
	stringstream ss;

	string space;
	for (int i=0; i<depth; i++)
		space += " ";
	ss << space.c_str() << "[\n";
	ss << space.c_str() << "inputVertices(g) = ";
	for (unsigned int i = 0; i < g.inputVertices.size(); i++) {
		ss << vertexToString(*(g.inputVertices[i]));
	}
	ss << endl;

	ss << space.c_str() << "edges(g) = ";
	for (unsigned int i = 0; i < g.inputVertices.size(); i++) {
		for (unsigned int j = 0; j < g.inputVertices[i]->outEdges.size(); j++) {
			Edge<PTYPE>* e = g.inputVertices[i]->outEdges[j];
			if (e->state == g.state)
				ss << edgeToString(*e);
		}
	}
	ss << endl;

	if (g.state==CONCRETE){
		ss << space.c_str() << "subgraphs : \n";
		for (unsigned int i = 0; i < g.subgraphs.size(); i++) {
			ss << graphToString(*(g.subgraphs[i]), depth+1);
		}
	}
	ss << space.c_str() << "]\n";
	return ss.str();
}

//template <class T>
std::ostream &operator<<(std::ostream &stream, Edge<PTYPE>& obj) {
	stream << edgeToString(obj);
	return stream;
}

//template <class T>
std::ostream &operator<<(std::ostream &stream, Vertex<PTYPE>& obj) {
	stream << vertexToString(obj);
	return stream;
}

//template <class T>
std::ostream &operator<<(std::ostream &stream, Graph<PTYPE>& obj) {
	stream << graphToString(obj);
	return stream;
}

std::ostream &operator<<(std::ostream &stream, EdgeContainer& obj) {
	stringstream ss;

	ss << "[EdgeContainer isPartitioned=" << obj.isPartitioned;

	for (edge_it it = obj.begin(); it != obj.end(); it++) {
		if (it == obj.markedUpperBound) {
			ss << endl << "--- END MARKED ---";
		}
		ss << endl << (**it);
	}

	ss << "]";

	stream << ss.str();
	return stream;
}

}
