/**
 * Parametric SCC based Model Checking
 *  
 * This is a stand-alone tool which performs model checking
 * for parametric discrete-time Markov Chains (PDTMCs).
 * 
 * Copyright (c) 2013 RWTH Aachen University.
 * Authors: Florian Corzilius, Nils Jansen, Matthias Volk
 * 
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 * 
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the
 * GNU General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public License
 * along with this program.  If not, see
 * http://www.gnu.org/licenses/gpl.html.
 * 
 * 
 * Main Contact:
 * 
 * Nils Jansen
 * Theory of Hybrid Systems
 * RWTH Aachen
 * 52056 Aachen
 * Germany
 * nils.jansen@cs.rwth-aachen.de
 */

#ifndef GRAPHTYPES_H_
#define GRAPHTYPES_H_

#ifdef USE_EXACT
#include <gmpxx.h>
#endif

#ifdef USE_PARAMETRIC
#include "Rational.h"
#endif

#ifdef USE_LOGLIB
#include <log4cplus/logger.h>
#endif

#include <vector>
#include <set>
#include <iostream>
#include <algorithm>
#include <queue>
#include <sstream>
#include <string>
#include <stdio.h>
#include <stdlib.h>
#include <pthread.h>

#include "../defines.h"
#include "TriBool.h"

namespace scc_cex {

class Path;

template<class T>
class Edge;
template<class T>
class Vertex;
template<class T>
class Graph;

extern bool IsMarked(const Edge<PTYPE>* arg);
extern bool KeepAfterModelchecking(const Edge<PTYPE>* arg);

typedef std::vector<Edge<PTYPE>*> edge_vector;
typedef std::set<Edge<PTYPE>*> edge_set;
typedef std::vector<Edge<PTYPE>*>::iterator edge_it;

typedef std::vector<Vertex<PTYPE>*> vtx_vector;
typedef std::set<Vertex<PTYPE>*> vtx_set;
typedef std::vector<Vertex<PTYPE>*>::iterator vtx_it;

typedef std::vector<Graph<PTYPE>*> graph_vector;
typedef std::set<Graph<PTYPE>*> graph_set;
typedef std::vector<Graph<PTYPE>*>::iterator graph_it;

class EdgeContainer;

template<class T>
class Edge {
public:
	Vertex<T>* source;
	Vertex<T>* target;
	T prob;
	//state: edge is abstract or concrete
	bool state;

	//to mark new edges added in concretization
	//note: this flag is used to efficiently decide if paths contain new edges
	//      it's not used for removing unnecessary edges, that would be too expensive (look at CexGenerator)
	bool not_m_min;

	bool visited; //For closure computation
	Edge<PTYPE>* sisterEdge; //For abstract cex search: So that we can get rid of this closure edge in O(1)

	friend std::ostream &operator<<(std::ostream &stream, Edge<PTYPE>& obj);

	Edge(Vertex<T>* _source, Vertex<T>* _target, T _prob, bool _state = CONCRETE) :
			source(_source), target(_target), prob(_prob), state(_state) {
		source->outEdges.push_back(this);
		target->inEdges.push_back(this);
		flagged = false;
		not_m_min = false;
		visited = false;
		sisterEdge = NULL;

		if (source == target) {
			//Set self loop
			source->selfLoop = this;
		}
	}

	~Edge() {
	}

	inline void clear() {
		source->outEdges.remove(this);
		target->inEdges.remove(this);
	}

	bool operator<(const Edge<PTYPE>& e) const;

	inline bool isAbstract() const {
		return state == ABSTRACT;
	}
	inline bool isConcrete() const {
		return state == CONCRETE;
	}

	inline bool getMark() const {
		return flagged;
	}

	void setMark(bool mark);

	inline bool getXMLMark() const {
		return xmlMark;
	}

	inline void setXMLMark(bool mark) {
		xmlMark = mark;
	}
private:
	//this flag...
	//- to keep track of edge-printing in XML generation
	//- to keep track of concretization level in cex generation
	bool flagged;

	//Flag for xml
	bool xmlMark;
};

/*
 * This is a container for edge lists, to be used to store the outgoing / incoming edges of vertices.
 * This container should be used instead of simple vectors because it deals efficiently with the need
 * of accessing only visible edges / removing invisible edges etc.
 * It is fairly lightweight because all methods are declared inline
 */
class EdgeContainer {
public:
	EdgeContainer() {
		isPartitioned = true;
		markedUpperBound = edges.end();
	}
	virtual ~EdgeContainer() {
		edges.clear();
	}

	inline void push_back(Edge<PTYPE>* e) {
		edges.push_back(e);
		if (e->getMark())
			isPartitioned = false;
	}

	/**
	 * 	Note: This method does not check whether endMarked() becomes invalid
	 */
	inline void pop_back() {
		edges.pop_back();
	}

	inline void remove(Edge<PTYPE>* e) {
		edge_it it = edges.begin();
		for (; it != edges.end(); it++) {
			if (*it == e)
				break;
		}
		if (it != edges.end()) {
			edges.erase(it);
		}
	}

	/**
	 * Don't bother to repartition
	 */
	inline edge_it unsafe_begin() {
		return edges.begin();
	}

	/**
	 * Safe begin: repartition if necessary, then return iterator to begin of marked edges
	 */
	inline edge_it begin() {
		if (!isPartitioned) {
			callPartition();
		}
		return edges.begin();
	}

	inline edge_it end() {
		return edges.end();
	}

	//iterator pointing to the 1st invisible element (i.e. same semantics as vector.end())
	inline edge_it endMarked() {
		if (!isPartitioned) {
			callPartition();
		}
		return markedUpperBound;
	}

	/*
	 * (Re)set the flag for all edges; should be called between model checking & cex generation
	 */
	inline void markAll(bool mark) {
		for (edge_it it = edges.begin(); it != edges.end(); it++)
			(*it)->setMark(mark);

		//Avoid unnecessary re-partitioning
		isPartitioned = true;

		markedUpperBound = edges.end();
	}

	/**
	 * As it says: Mark concrete, unmark abstract
	 */
	inline void markConcrete_unmarkAbstract() {
		for (edge_it it = edges.begin(); it != edges.end(); it++) {
			if ((*it)->isAbstract())
				(*it)->setMark(false);
			else {
				(*it)->setMark(true);
				(*it)->not_m_min = true;
			}
		}

		repartition();
	}

	inline void removeSisterReferences() {
		for (edge_it it = edges.begin(); it != edges.end(); it++) {
			(*it)->sisterEdge = NULL;
		}
	}

	inline bool containsMarkedEdges() {
		return begin() != endMarked();
	}

	/*
	 * _Mark_ this container for repartitioning. This is pretty cheap, because the repartitionting
	 * won't be carried out before we actually access critical functions
	 */
	inline void repartition() {
		isPartitioned = false;
	}

	//Erase all elements outside the marked range. This only resizes the container - the caller must see to it that the objects are destroyed!
	inline void cutOutUnmarked() {
		if (!isPartitioned) {
			callPartition();
		}
		edges.erase(markedUpperBound, edges.end());
	}
	inline unsigned int size() const {
		return edges.size();
	}
	inline Edge<PTYPE>*& operator[](unsigned int i) {
		return edges[i];
	}

	inline void cleanUpEdges(bool doDelete) {
		edge_it fstAbstract = stable_partition(edges.begin(), edges.end(), KeepAfterModelchecking);

		if (doDelete) {
			for (edge_it it = fstAbstract; it != edges.end(); it++)
				delete *it;
		}

		edges.erase(fstAbstract, edges.end());

		for (edge_it it = edges.begin(); it != edges.end(); it++) {
			(*it)->visited = false;
		}
	}

	friend std::ostream &operator<<(std::ostream &stream, EdgeContainer& obj);

private:
	std::vector<Edge<PTYPE>*> edges;
	edge_it markedUpperBound;
	bool isPartitioned;

	void callPartition() {
		markedUpperBound = partition(edges.begin(), edges.end(), IsMarked);
		isPartitioned = true;
	}
};

class Path {
public:
	// Constructor for external callers (e.g. Barni ;))
	Path(Path* _pred, Vertex<PTYPE>* _node, Edge<PTYPE>* _edge); //Constructor for Barni
	// Constructor for search algorithms
	Path(const unsigned int _k, PTYPE _L, Path* _pred, Vertex<PTYPE>* _node, Edge<PTYPE>* _edge);

	std::string toString() const;
	std::string toStringDebug() const;
	std::string toStringOutput() const;
	std::string recursiveOutput(const Path* path) const;

	Path* deepCopy();
	void deepDelete();

	virtual ~Path();

	unsigned int k; //kth shortest path
	PTYPE L; //length of path
	Path* pred; //predecessor
	Vertex<PTYPE>* node; //node where this path leads; this gives speed, costs space
	Edge<PTYPE>* edge; //edge from predecessor to this node
	bool terminateHere; //for local search: terminate loop (source and target can be the same)

	bool operator<(const Path &other) const {
		//TODO check if correct assumption to set to false
		return param_sccmc::TriBool::toBoolUndefFalse(L < other.L);
	}

	friend std::ostream &operator<<(std::ostream &stream, Path& obj);
};

template<class T>
class Vertex {
public:
	EdgeContainer inEdges;
	EdgeContainer outEdges;
	//	std::vector<Edge<T>*> inEdges;
	//	std::vector<Edge<T>*> outEdges;

	Graph<T>* g;

	Vertex<T>* sisterVtx;

	T weight; //auxiliary field to compute the probabilities (see reduceGraph)
	bool seen; //auxiliary field for all the dfs-based searches
	unsigned int distance; //needed for state elimination approach
	bool eliminated; //needed for state elimination approach
	bool tmpMarking; //auxiliary field for e.g. concretization
	bool isInClosure;

	bool isConcretized; //to keep track of concretization in cex generation

	bool inXML; //For xml export

	Edge<T>* selfLoop; //needed for state elimination approach

	//original index of the vertex in the input file
	unsigned int oindex;

	//dfs and lowlink are needed for SCC detection
	int dfs;
	int lowlink;

	//atomic propositions
	AtPropType ap;

	bool isInputVertex;

	//Shortest Path attributes & functionality
	std::vector<Path*> mShortestPaths;
	std::priority_queue<Path, std::vector<Path> > mCandidates;

	friend std::ostream &operator<<(std::ostream &stream, Vertex<PTYPE>& obj);

	inline void setkthShortestPath(const unsigned int k, Path* path) {
		while (mShortestPaths.size() < k)
			mShortestPaths.push_back(NULL);
		mShortestPaths[k - 1] = path;
	}
	inline Path* getkthShortestPath(const unsigned int k) const {
		if (mShortestPaths.size() >= k && mShortestPaths[k - 1] != NULL)
			return mShortestPaths[k - 1];
		else
			return NULL;
	}
	/**
	 * Removes and deletes all shortest paths
	 * Returns whether there was anything to clean.
	 */
	inline bool cleanShortestPaths() {
		if (mShortestPaths.size() == 0 || mShortestPaths[0] == NULL)
			return true;
		if (mShortestPaths[0]->L == 0)
			return false;

		for (unsigned int i = 0; i < mShortestPaths.size(); i++) {
			if (mShortestPaths[i] != NULL) {
				if (mShortestPaths[i]->edge != NULL)
					mShortestPaths[i]->edge->visited = false;
				delete mShortestPaths[i];
			}
		}
		mShortestPaths.clear();
		return true;
	}

	inline std::string kthShortestPathToString(const unsigned int k) {
		return getkthShortestPath(k)->toStringDebug();
	}

	void clear();

	inline std::string getName() const {
		std::stringstream ss;
		if (!isConcretized)
			ss << g->graph_id << "/";
		ss << oindex;
		return ss.str();
	}

	inline PTYPE getSelfLoopProb() {
		if (selfLoop != NULL) {
			return selfLoop->prob;
		} else {
			return 0.0;
		}
	}

	Vertex() {
		init(0);
	}

	Vertex(unsigned int name) {
		init(name);
	}

	~Vertex() {
		clear();
	}
private:
	inline void init(unsigned int name) {
		g = NULL;
		weight = -1.0;
		seen = false;
		distance = UINT_MAX;
		eliminated = false;
		isInClosure = false;
		isConcretized = true;
		isInputVertex = false;
		sisterVtx = NULL;
		selfLoop = NULL;

		oindex = name;
		dfs = 0;
		lowlink = 0;
	}
};

template<class T>
class Graph {
private:
	static int maxId;
public:

	std::vector<Vertex<T>*> inputVertices;
	std::vector<Vertex<T>*> outputVertices;
	std::vector<Vertex<T>*> allVertices;
	std::vector<Graph<T>*> subgraphs;
	//state: abstract or concrete
	bool state;
	Graph* parent;

	//atomic propositions
	std::vector<bool> abstract_ap;

	bool concretized;

	int graph_id;

	bool inXML; //For xml export

	bool limitedFree; //When deleting the graph, don't delete its vertices
	bool blockRecursion; //For concretization: Don't look deeper into this SCC subtree (because it is unreachable)

	friend std::ostream &operator<<(std::ostream &stream, Graph<PTYPE>& obj);

	Graph() {
		state = CONCRETE;
		parent = NULL;
		concretized = false;
		Graph<T>::maxId++;
		graph_id = Graph<T>::maxId;
		limitedFree = false;
		blockRecursion = false;
	}

	~Graph() {
		outputVertices.clear();

		if (!limitedFree) {
			for (unsigned int i = 0; i < allVertices.size(); i++)

				if (allVertices[i]->g == this) {
					delete allVertices[i];
				}
		}
		allVertices.clear();

		for (unsigned int i = 0; i < subgraphs.size(); i++) {
			subgraphs[i]->limitedFree = limitedFree;
			delete subgraphs[i];
		}
		subgraphs.clear();
	}

	Graph* addSubgraph();
	void addSubgraph(Graph<T> *gra);
	std::string toString(int prefixes);

	inline bool isChildOf(const Graph<T>* parent) const {
		return this->parent == parent;
	}

	inline bool isParentOf(const Graph<T>* child) const {
		for (unsigned int i = 0; i < subgraphs.size(); i++) {
			if (subgraphs[i] == child)
				return true;
		}
		return false;
	}

	inline void cleanShortestPaths() {
		//Loop through both all vertices and output vertices
		vtx_it it = allVertices.begin();

		while (it != outputVertices.end()) {

			if ((*it)->cleanShortestPaths()) {
				Path* initialPath = new Path(1, 0, NULL, (*it), NULL);
				(*it)->setkthShortestPath(1, initialPath);

				//Must also empty the candidate sets!
				while (!(*it)->mCandidates.empty())
					(*it)->mCandidates.pop();
			}

			it++;
			if (it == allVertices.end())
				it = outputVertices.begin();
		}
	}

	inline bool isReachableInMarking(Graph<T>* subg) const {
		if (subg == this)
			return true;

		bool result = false;

		for (vtx_it it = subg->inputVertices.begin(); it != subg->inputVertices.end(); it++) {
			result |= (*it)->inEdges.containsMarkedEdges();
			if (result)
				break;
		}

		return result;
	}

	inline void resetStructureAfterModelChecking() {
		for (vtx_it it = allVertices.begin(); it != allVertices.end(); it++) {
			(*it)->g = this;
		}

		for (unsigned int i = 0; i < subgraphs.size(); i++) {
			subgraphs[i]->limitedFree = true;
			delete subgraphs[i];
		}
		subgraphs.clear();
	}

	inline void deepUnconcretize() {
		concretized = false;
		blockRecursion = false;

		for (vtx_it it = inputVertices.begin(); it != inputVertices.end(); it++) {
			(*it)->isConcretized = false;
		}

		for (graph_it it = subgraphs.begin(); it != subgraphs.end(); it++) {
			(*it)->deepUnconcretize();
		}
	}

};

/* --------------- Graph ----------------- */
template<class T> int Graph<T>::maxId = -1;

template<class T>
Graph<T>* Graph<T>::addSubgraph() {
	Graph<T>* g = new Graph<T>();
	subgraphs.push_back(g);
	g->parent = this;
	return g;
}

template<class T>
void Graph<T>::addSubgraph(Graph<T> *gra) {
	subgraphs.push_back(gra);
	gra->parent = this;
}

template<class T>
std::string Graph<T>::toString(int prefixes) {
	std::string out = "";
	std::string pr = "";
	for (int i = 0; i < prefixes; i++) {
		pr += "-";
	}
	//char temp[100];
	//std::itoa(this->graph_id, temp, 10);
	std::ostringstream oss;
	oss << this->graph_id;
	out += "Graphid: " + oss.str() + "\n";
	oss.str(pr);
	oss << this->state;
	out += "-state: " + oss.str() + "\n";
	oss.str(pr);

	oss << this->concretized;
	out += "-concretized: " + oss.str() + "\n";

	for (vtx_it it = this->allVertices.begin(); it != this->allVertices.end(); it++) {
		oss.str(pr);
		oss << (*it)->oindex;
		out += "allVertices contains: " + oss.str() + "\n";
		if (((*it)->g != NULL) && ((*it)->g != this)) {
			oss.str(pr);
			oss << (*it)->g;
			out += "-- Adresse des Graphs: " + oss.str() + "\n";
			oss.str(pr);
			oss << (*it)->g->graph_id;
			out += "-- Und dieser hat die ID " + oss.str() + "\n";
			//out += (*it)->g->toString(prefixes);
		}
		if ((*it)->g == this) {
			out += "Ich gehoere meinem Graphen\n";
		}
		for (edge_it eit = (*it)->outEdges.begin(); eit != (*it)->outEdges.end(); eit++) {
			oss.str(pr);
			oss << (*eit)->target->oindex;
			out += "---- edge to: " + oss.str() + " with prob: ";
			oss.str(pr);
			oss << (*eit)->prob;
			out += oss.str() + " abstract: ";
			oss.str(pr);
			oss << (*eit)->isAbstract();
			out += oss.str() + "\n";
		}
	}

	for (vtx_it it = this->outputVertices.begin(); it != this->outputVertices.end(); it++) {
		oss.str(pr);
		oss << (*it)->oindex;
		out += "outVertices contains: " + oss.str() + "\n";
	}

	oss.str("");
	oss << this->subgraphs.size();
	out += "Dieser Graph hat " + oss.str() + " subgraphs\n";

	return out;

}

/* --------------- Edge ------------------- */

template<class T>
void Edge<T>::setMark(bool mark) {
	if (mark != flagged) {
		flagged = mark;
		source->outEdges.repartition();
		target->inEdges.repartition();
	}
}

template<class T>
bool Edge<T>::operator<(const Edge<PTYPE>& e) const {
	if (this->source->oindex < e.source->oindex) {
		return true;
	} else if (this->source->oindex == e.source->oindex) {
		if (this->target->oindex < e.target->oindex) {
			return true;
		} else {
			return false;
		}
	}

	return false;
}

/* --------------- Vertex ----------------- */
template<class T>
void Vertex<T>::clear() {
	while (inEdges.size() > 0) {
		Edge<T>* e = inEdges[inEdges.size() - 1];
		inEdges.pop_back();
		Vertex<T>& source = *(e->source);
		for (unsigned int j = 0; j < source.outEdges.size(); j++) {
			if (source.outEdges[j] == e) {
				source.outEdges[j] = source.outEdges[source.outEdges.size() - 1];
				source.outEdges.pop_back();
				break;
			}
			assert(j < source.outEdges.size() - 1);
		}
		delete e;
	}

	while (outEdges.size() > 0) {
		Edge<T>* e = outEdges[outEdges.size() - 1];
		outEdges.pop_back();
		Vertex& target = *(e->target);
		for (unsigned int j = 0; j < target.inEdges.size(); j++) {
			if (target.inEdges[j] == e) {
				target.inEdges[j] = target.inEdges[target.inEdges.size() - 1];
				target.inEdges.pop_back();
				break;
			}
			assert(j < target.inEdges.size() - 1);
		}
		delete e;
	}
}

/* ------------- utility ---------- */
template<class T>
void reinitializeEdges(std::vector<Vertex<T>*> vertices) {
	for (unsigned int i = 0; i < vertices.size(); i++) {
		vertices[i]->outEdges.markAll(false);
		vertices[i]->outEdges.removeSisterReferences();
		vertices[i]->inEdges.repartition();
	}
}

template<class T>
void reinitializeEdgesForConcreteSearch(std::vector<Vertex<T>*> vertices) {
	for (unsigned int i = 0; i < vertices.size(); i++) {
		vertices[i]->outEdges.markConcrete_unmarkAbstract();
		vertices[i]->outEdges.removeSisterReferences();
		//vertices[i]->inEdges.repartition();
	}
}

/* tracing functions */
template<class T>
extern std::string vertexToString(const Vertex<T> & v);
template<class T>
extern std::string edgeToString(const Edge<T> & e);
template<class T>
extern std::string graphToString(const Graph<T>& g);
template<class T>
extern std::string graphToString(const Graph<T>& g, const int depth);

}

#endif /* GRAPHTYPES_H_ */
