/**
 * Parametric SCC based Model Checking
 *  
 * This is a stand-alone tool which performs model checking
 * for parametric discrete-time Markov Chains (PDTMCs).
 * 
 * Copyright (c) 2013 RWTH Aachen University.
 * Authors: Florian Corzilius, Nils Jansen, Matthias Volk
 * 
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 * 
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the
 * GNU General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public License
 * along with this program.  If not, see
 * http://www.gnu.org/licenses/gpl.html.
 * 
 * 
 * Main Contact:
 * 
 * Nils Jansen
 * Theory of Hybrid Systems
 * RWTH Aachen
 * 52056 Aachen
 * Germany
 * nils.jansen@cs.rwth-aachen.de
 */

#include "../defines.h"
#include "Parameters.h"
#include "assert.h"
#include <sstream>

#ifdef USE_LOGLIB
#include <log4cplus/logger.h>
#endif

using namespace log4cplus;

namespace parametric {
Logger parametriclogger = Logger::getInstance("Parametric");

/**
 * Constructor
 */
Parameters::Parameters() :
		minusOne(new Polynomial(-1)), one(new Polynomial(1)), zero(new Polynomial(0)) {
	bool result = polynomials.insert(std::pair<const ex, constPoly>(-1, minusOne)).second;
	assert(result);
	result = polynomials.insert(std::pair<const ex, constPoly>(1, one)).second;
	assert(result);
	result = polynomials.insert(std::pair<const ex, constPoly>(0, zero)).second;
	assert(result);
	parametriclogger.setLogLevel(DEFAULT_LOG_LEVEL);
}

/**
 * Constructor
 * @param param parameters
 * @return
 */
Parameters::Parameters(const Parameters& param) :
		polynomials(param.polynomials), minusOne(param.minusOne), one(param.one), zero(param.zero) {
}

/**
 * Destructor
 * @return
 */
Parameters::~Parameters() {
	while (!constraints.empty()) {
		const Constraint* constraint = *constraints.begin();
		constraints.erase(constraints.begin());
		delete constraint;
	}

	while (!abstractParameters.empty()) {
		const AbstractParameter* abstractParameter = *abstractParameters.begin();
		abstractParameters.erase(abstractParameters.begin());
		delete abstractParameter;
	}

	while (!polynomials.empty()) {
		const Polynomial* pol = polynomials.begin()->second;
		polynomials.erase(polynomials.begin());
		delete pol;
	}

#ifdef USE_POLY_MANAGEMENT
	queueUnusedPol.clear();
	unusedPolynomials.clear();

	while (queueOldPol.size() > 0) {
		constPoly pol = queueOldPol.front();
		queueOldPol.pop_front();
		size_t count = oldPolynomials.erase(pol->poly);
		assert(count == 1);
		delete pol;
	}
	assert(oldPolynomials.empty());

#endif
}

/**
 * Used for singleton
 * @return instance
 */
Parameters& Parameters::getInstance() {
	static Parameters instance;
	return instance;
}

/**Check constraint and if result is undefined add a ensuring constraint
 * @param rat1 rational function 1
 * @param rat2 rational function 2
 * @param opType operatorType (<, <=, ==)
 */
void Parameters::checkConstraint(const Rational& rat1, const Rational& rat2, OperatorType opType) {
	TRIBOOL result;
	switch (opType) {
	case OPERATOR_LEQ:
		result = (rat1 <= rat2);
		break;
	case OPERATOR_LESS:
		result = (rat1 < rat2);
		break;
	case OPERATOR_EQ:
		assert(rat1 == rat2);
		return;
	}

	if (result == TRIBOOL::TRUE) {
		return;
	} else if (result == TRIBOOL::FALSE) {
		assert(false);
		return;
	} else {
		//Result is undefined => check for existing constraint
		Constraint *constraint = new Constraint(rat1, rat2, opType);
		constraint_const_it iter = constraints.find(constraint);
		if (iter == constraints.end()) {
			//Not found -> add as new constraint
			constraints.insert(constraint);
		} else {
			delete constraint;
		}
		//Constraint now ensures satisfiability
	}
}

/**Checks constraint rat>0 and if result is undefined add a ensuring constraint
 * @param rat rational function
 */
void Parameters::checkConstraintGreaterZero(const Rational& rat) {
	Rational ratZero = getZero();
	checkConstraint(ratZero, rat, OPERATOR_LESS);
}

/**Checks constraint rat<1 and if result is undefined add a ensuring constraint
 * @param rat rational function
 */
void Parameters::checkConstraintLessOne(const Rational& rat) {
	Rational ratOne = getOne();
	checkConstraint(rat, ratOne, OPERATOR_LESS);
}

/**
 * Check constraint "0 < rat < 1" except for 0 and 1 which are okay
 * @param rat rational function
 */
void Parameters::checkConstraintOneZero(const Rational& rat) {
	if (rat != 0 && rat != 1) {
		checkConstraintGreaterZero(rat);
		checkConstraintLessOne(rat);
	}
}

/**
 * Add new abstract transition probability
 * @param rat rational function which is abstracted
 * @return
 */
const Rational& Parameters::addAbstractTransition(const Rational& rat) {
	//Create new parameter for this abstract transition
	AbstractParameter* parameter = new AbstractParameter(rat);
	abstractParameters.push_back(parameter);
	checkConstraintOneZero(parameter->getValue());
	return parameter->getValue();
}


/**
 * Create new polynomial in pool or get existing one, if already created
 * @param expr ex for new polynomial
 * @return polynomial with this expression
 */
constPoly Parameters::createPolynomial(const ex& expr) {
#if defined(USE_POLY_MANAGEMENT) && defined(DEBUGGING)
	count++;
	if (count == CHECK_COUNT) {
		count = 0;
		checkPolyManagement();
	}
#endif

	polynomial_const_it iter = polynomials.find(expr);
	if (iter != polynomials.end()) {
		//Polynomial already exists
		return iter->second;
	} else {

#ifdef USE_POLY_MANAGEMENT
		//Try to restore from old polynomial pool
		constPoly oldPol = restorePolynomial(expr);
		if (oldPol != NULL) {
			if (oldPol->getUsageCounter() == 0) {
				//Polynomial might not be used
				queueUnusedPol.push_back(oldPol);	
				bool inserted = unusedPolynomials.insert(std::pair<const ex, QueuePol::iterator>(oldPol->poly, --queueUnusedPol.end())).second;
				if (!inserted){
					queueUnusedPol.pop_back();
				}
			}
			return oldPol;
		}
#endif

		//Create new polynomial
		Polynomial* pol = new Polynomial(expr);
		bool result = polynomials.insert(std::pair<const ex, Polynomial*>(expr, pol)).second;
		assert(result);
		
#ifdef USE_POLY_MANAGEMENT
		queueUnusedPol.push_back(pol);
		bool inserted = unusedPolynomials.insert(std::pair<const ex, QueuePol::iterator>(pol->poly, --queueUnusedPol.end())).second;
		if (!inserted){
			queueUnusedPol.pop_back();
		}
#endif

		return pol;
	}
}

/**
 * Create new polynomial in pool or get existing one, if already created
 * @param factorization factorization of new polynomial
 * @return polynomial constructed from the factorization
 */
constPoly Parameters::createPolynomial(const FactorList& factorization) {
	ex expr = Factorization::getExpressionFromFactorization(factorization);
	constPoly pol = createPolynomial(expr);
	pol->getFactorization()->setFactorization(factorization);
	return pol;
}

#ifdef USE_POLY_MANAGEMENT
/**
 * Restore polynomial by reinserting in pool and deleting from old pool
 * @param expr polynomial
 * @return polynomial recreated in pool, NULL if polynomial is not in old pool
 */
constPoly Parameters::restorePolynomial(const ex& expr) {
	//LOG4CPLUS_TRACE(parametriclogger, "Restore polynomial: " << expr);
	polyManage_const_it iterOld = oldPolynomials.find(expr);

	if (iterOld == oldPolynomials.end()) {
		//Polynomial does not exists in old pool
		return NULL;
	}

	queuePol_const_it iterQueue = iterOld->second;
	constPoly pol = *iterQueue;

	//Add polynomial again
	polynomials.insert(std::pair<const ex, constPoly>(pol->poly, pol));

	//Delete in old pool
	oldPolynomials.erase(iterOld);
	queueOldPol.erase(iterOld->second);
	pol->usageCounter = 0;

	//Update factors
	if (!pol->getFactorization()->isEmpty()) {
		//Increment usageCount for factors of restored polynomial
		FactorMap factorization = pol->getFactorization()->getFactors(false);
		for (factors_it iter = factorization.begin(); iter != factorization.end(); iter++) {
			assert(iter->first->getUsageCounter() >= 0);
			if (iter->first->getUsageCounter() == 0) {
				restorePolynomial(iter->first->poly);
			}
			iter->first->incUsageCounter();

		}
	}

	return pol;
}

/**
 * Delete polynomial from set if not used anymore
 * @param pol polynomial to delete
 * @param root signals root for recursion and enables cleanup
 */
void Parameters::deletePolynomial(constPoly pol, bool root) {
	if (pol->isNumber() && (*pol == 0 || *pol == 1 || *pol == -1)) {
		//Do not delete important polynomials
		return;
	}

	//LOG4CPLUS_TRACE(parametriclogger, "Delete polynomial: " << pol->poly);
	assert(pol->usageCounter == 0);

	//We remove polynomial before its factors -> will delete it before it factors later
	// > we avoid problems with non-existing factors
	size_t count = polynomials.erase(pol->poly);
	assert(count == 1);
	queueOldPol.push_back(pol);
	bool result = oldPolynomials.insert(std::pair<const ex, QueuePol::iterator>(pol->poly, --queueOldPol.end())).second;
	assert(result);

	if (!pol->getFactorization()->isEmpty()) {
		//Decrement usageCount for factors of deleted polynomial
		FactorMap factorization = pol->getFactorization()->getFactors(true);
		for (factors_it iter = factorization.begin(); iter != factorization.end(); iter++) {
			iter->first->decUsageCounter(false);
		}
	}

	//Only clean up after cascade of deletion is finished
	if (root) {
		cleanUp();

#ifdef DEBUGGING
		count++;
		if (count == CHECK_COUNT) {
			count = 0;
			checkPolyManagement();
		}
#endif
	}
}

/**
 * Clean up: completely remove oldest polynomials and check possibly unused polynomials in pool
 */
void Parameters::cleanUp() {
	//Delete oldest polynomials
	while (queueOldPol.size() > POOL_OLD_SIZE) {
		constPoly pol = queueOldPol.front();
		//LOG4CPLUS_TRACE(parametriclogger, "Completely remove polynomial: " << pol->poly);
		queueOldPol.pop_front();
		size_t count = oldPolynomials.erase(pol->poly);
		assert(count == 1);
		
		//Delete from unused polynomials
		polyManage_const_it iterUnused = unusedPolynomials.find(pol->poly);
		if (iterUnused != unusedPolynomials.end()){
			queueUnusedPol.erase(iterUnused->second);
			unusedPolynomials.erase(iterUnused);
		}
		
		delete pol;
	}

	if (queueUnusedPol.size() > POOL_UNUSED_UPPER) {
		//Check possibly unused polynomials
		while (queueUnusedPol.size() > POOL_UNUSED_LOWER) {
			constPoly pol = queueUnusedPol.front();
			queueUnusedPol.pop_front();
			unusedPolynomials.erase(pol->poly);

			//Delete
			if (pol->getUsageCounter() == 0) {
				if (polynomials.find(pol->poly) != polynomials.end()) {
					deletePolynomial(pol, false);
				}
			}
		}
	}
}

#ifdef DEBUGGING
void Parameters::checkPolyManagement() const {
	FactorMap polyTmp;
	for (polynomial_const_it iter = polynomials.begin(); iter != polynomials.end(); iter++) {
		polyTmp.insert(make_pair(iter->second, 0));
		if (iter->second->usageCounter <= 0) {
			if (iter->second != minusOne && iter->second != zero && iter->second != one) {
				assert(iter->second->usageCounter == 0);

				//Possibly unused polynomials have own pool
				assert(unusedPolynomials.find(iter->second->poly) != unusedPolynomials.end());
			}
		}
	}

	for (polynomial_const_it iter = polynomials.begin(); iter != polynomials.end(); iter++) {
		if (!iter->second->factors->isEmpty() && iter->second->getUsageCounter() > 0) {
			FactorMap factors = iter->second->factors->getFactors(false);
			for (factors_it iterFac = factors.begin(); iterFac != factors.end(); iterFac++) {
				factors_it itFound = polyTmp.find(iterFac->first);
				if (itFound != polyTmp.end()) {
					itFound->second++;
				} else {
					LOG4CPLUS_DEBUG(parametriclogger, iterFac->first->poly << " not found! (1)");
					//assert(false);
				}
			}
		}
	}

	for (polynomial_const_it iter = polynomials.begin(); iter != polynomials.end(); iter++) {
		factors_it iterFound = polyTmp.find(iter->second);
		if (iterFound != polyTmp.end()) {
			if (iterFound->second > iter->second->getUsageCounter()) {
				LOG4CPLUS_DEBUG(parametriclogger, "Usage counter too low! " << iter->second->poly << ": " << iter->second->getUsageCounter() << " < " << iterFound->second);
				assert(false);
			}
		} else {
			LOG4CPLUS_DEBUG(parametriclogger, iter->second->poly << " not found! (2)");
			assert(false);
		}
	}
	polyTmp.clear();
}
#endif
#endif

/**
 * Add new variable
 * @param name string with name
 */
void Parameters::addVariable(const std::string& name) {
	symbol sym(name);
	parameterTable[name] = sym;
	parameterList.append(sym);
	LOG4CPLUS_TRACE(parametriclogger, "Parameter: " << name);
}

/**
 * Output information about parameters, abstract parameters, constraints and polynomial pool
 * @param substituted not used currently
 * @return string with all information
 */
std::string Parameters::toString(bool substituted) const {
	//TODO use substituted
	std::stringstream stream;
	lst substitutions;

	stream << std::endl << "Parameters:" << std::endl;
	for (symtab::const_iterator iter = parameterTable.begin(); iter != parameterTable.end(); iter++) {
		if (iter != parameterTable.begin()) {
			stream << ", ";
		}
		stream << iter->second;
	}
	stream << std::endl;
	stream << std::endl;

	stream << "Constraints:" << std::endl;

	for (constraint_const_it iter = constraints.begin(); iter != constraints.end(); iter++) {
		Constraint* constraint = *iter;
		stream << *constraint << std::endl;
	}
	stream << std::endl;

	stream << "Abstract Parameters: " << abstractParameters.size() << std::endl;
	for (unsigned int i = 0; i < abstractParameters.size(); i++) {
		stream << *abstractParameters[i] << std::endl;
	}
	stream << std::endl;

	stream << "Count polynomials: " << getNumberPolynomials() << endl;

#ifdef USE_POLY_MANAGEMENT
	stream << "Count old polynomials: " << getNumberOldPolynomials() << endl;
#endif

	return stream.str();
}

/**
 * Output
 * @param stream
 * @param parameters
 * @return
 */
std::ostream & operator<<(std::ostream & stream, Parameters & parameters) {
	stream << parameters.toString();
	return stream;
}

/**
 * Output for dReal
 * @return string with dReal output
 */
std::string Parameters::convertToDRealInput() const {
	std::stringstream stream;
	stream << "//[Variables]" << std::endl;
	for (symtab::const_iterator iter = parameterTable.begin(); iter != parameterTable.end(); iter++) {
		//Original parameters with bound
		//TODO check bounds
		stream << "[0, 1] ";
		stream << (*iter).second << ";" << std::endl;
	}

	for (unsigned int i = 0; i < abstractParameters.size(); i++) {
		//Original parameters with bound
		stream << "[0, 1] ";
		stream << abstractParameters[i]->getParameterRational().toString() << ";" << std::endl;
	}

	stream << "//[Constraints]" << std::endl;
	for (constraint_const_it iter = constraints.begin(); iter != constraints.end(); iter++) {
		Constraint* constraint = *iter;
		stream << "(" << constraint->toPrefixNotation() << ");" << std::endl;
	}

	for (unsigned int i = 0; i < abstractParameters.size(); i++) {
		//Original parameters with bound
		stream << "(" << abstractParameters[i]->toPrefixNotation() << ");" << std::endl;
	}

	return stream.str();
}

/**
 * Instantiate parameters with values and get a result for given rational function
 * @param rat rational function which should be evaluated w.r.t. given parameter instantiations
 */
void Parameters::instantiate(const Rational& rat) const {
	bool first = true;
	symtab sym;
	parser reader(sym);
	reader.strict = true;
	exmap instantiation;
	std::cout << std::endl;
	std::cout << "Instantiate the rational function " << rat << std::endl;
	for (symtab::const_iterator it = parameterTable.begin(); it != parameterTable.end(); it++) {
		std::cout << "Insert value for " << it->first << ":" << std::endl;
		std::string readValue;
		//getline(std::cin, readValue);

		//Hardcoding
		if (first) {
			readValue = "0.8";
			first = false;
		} else {
			readValue = "1/6";
		}

		ex e;
		try {
			e = reader(readValue);
		} catch (parse_error& err) {
			std::cerr << err.what() << " in " << readValue << std::endl;
		}
		if (!is_a<numeric>(e)) {
			std::cerr << "Invalid number!" << std::endl;
		}
		numeric n = ex_to<numeric>(e);
		std::cout << "Value is " << n << std::endl;
		instantiation[it->second] = n;
	}

	std::cout << "Result is: " << rat.getExpression().subs(instantiation) << std::endl;
	std::cout << std::endl;
}
}
