/**
 * Parametric SCC based Model Checking
 *  
 * This is a stand-alone tool which performs model checking
 * for parametric discrete-time Markov Chains (PDTMCs).
 * 
 * Copyright (c) 2013 RWTH Aachen University.
 * Authors: Florian Corzilius, Nils Jansen, Matthias Volk
 * 
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 * 
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the
 * GNU General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public License
 * along with this program.  If not, see
 * http://www.gnu.org/licenses/gpl.html.
 * 
 * 
 * Main Contact:
 * 
 * Nils Jansen
 * Theory of Hybrid Systems
 * RWTH Aachen
 * 52056 Aachen
 * Germany
 * nils.jansen@cs.rwth-aachen.de
 */

#ifndef PARAMETERS_H_
#define PARAMETERS_H_
#include "Rational.h"
#include "Constraint.h"
#include "AbstractParameter.h"
#include "ParametricSets.h"
#include "../typedefs.h"
#include <ginac/ginac.h>
#include <vector>
#include <string>
#include <boost/unordered_set.hpp>

using namespace GiNaC;

namespace parametric {

typedef std::vector<AbstractParameter*>::iterator abstractParameter_it;

/**
 * Singleton for all information about parametric computation. Has pool of all used polynomials, abstract transition probabilities and constraints.
 */
class Parameters {
public:
	static Parameters& getInstance();
	void checkConstraintOneZero(const Rational&);
	void checkConstraintGreaterZero(const Rational&);
	void checkConstraintLessOne(const Rational&);
	const Rational& addAbstractTransition(const Rational&);
	std::string convertToDRealInput() const;
	constPoly createPolynomial(const ex& ex);
	constPoly createPolynomial(const FactorList&);
	void addVariable(const std::string&);
	void instantiate(const Rational&) const;

	friend std::ostream &operator<<(std::ostream&, Parameters&);

	inline const Rational& addAbstractTransitionOne() {
		return addAbstractTransition(getOne());
	}

	inline std::string toString() const {
		return toString(false);
	}

	inline std::string toStringSubstituted() const {
		return toString(true);
	}

	inline const symtab& getParameterTable() const {
		return parameterTable;
	}

	inline const lst& getParameterList() const {
		return parameterList;
	}

	inline const ConstraintSet& getConstraints() const {
		return constraints;
	}

	inline const std::vector<AbstractParameter*>& getAbstractParameters() const {
		return abstractParameters;
	}

	inline constPoly getMinusOne() {
		return minusOne;
	}

	inline constPoly getOne() {
		return one;
	}

	inline constPoly getZero() {
		return zero;
	}

	inline unsigned int getNumberPolynomials() const {
		return polynomials.size();
	}

#ifdef USE_POLY_MANAGEMENT
	inline unsigned int getNumberOldPolynomials() const {
		return oldPolynomials.size();
	}

	void deletePolynomial(constPoly, bool root);
#endif

private:
	Parameters();
	Parameters(const Parameters&);
	virtual ~Parameters();
	void checkConstraint(const Rational&, const Rational&, OperatorType);
	std::string toString(bool substituted) const;

	symtab parameterTable;
	lst parameterList;
	std::vector<AbstractParameter*> abstractParameters;
	ConstraintSet constraints;
	PolynomialSet polynomials;

	constPoly minusOne;
	constPoly one;
	constPoly zero;

#ifdef USE_POLY_MANAGEMENT
	#define POOL_OLD_SIZE 100
	#define POOL_UNUSED_UPPER 50
	#define POOL_UNUSED_LOWER 20
	
	constPoly restorePolynomial(const ex&);
	void cleanUp();
	
	PolyManageSet oldPolynomials;
	PolyManageSet unusedPolynomials;
	QueuePol queueOldPol;
	QueuePol queueUnusedPol;
#endif

#ifdef DEBUGGING
	#define CHECK_COUNT 20
	unsigned int count = 0;
	void checkPolyManagement() const;
#endif
};

}
#endif /* PARAMETERS_H_ */
