/**
 * Parametric SCC based Model Checking
 *  
 * This is a stand-alone tool which performs model checking
 * for parametric discrete-time Markov Chains (PDTMCs).
 * 
 * Copyright (c) 2013 RWTH Aachen University.
 * Authors: Florian Corzilius, Nils Jansen, Matthias Volk
 * 
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 * 
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the
 * GNU General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public License
 * along with this program.  If not, see
 * http://www.gnu.org/licenses/gpl.html.
 * 
 * 
 * Main Contact:
 * 
 * Nils Jansen
 * Theory of Hybrid Systems
 * RWTH Aachen
 * 52056 Aachen
 * Germany
 * nils.jansen@cs.rwth-aachen.de
 */

#include "ParametricSets.h"
#include "Polynomial2.h"
#include "Constraint.h"
#include "Rational.h"

using namespace GiNaC;

namespace parametric {

/**
 * Get hash
 * @param expr expression
 * @return hash
 */
size_t ParametricSets::computeHash(const ex& expr) {
	return expr.gethash();
}

/**
 * Get hash
 * @param pol polynomial
 * @return hash
 */
size_t ParametricSets::computeHash(const Polynomial* pol) {
	return computeHash(pol->getExpression());
}

/**
 * Get hash
 * @param rat rational function
 * @return hash
 */
size_t ParametricSets::computeHash(const Rational& rat) {
	return computeHash(rat.getExpression());
}

unsigned int expressionEqual::counter = 0;

/**
 * Check for equality of expressions
 * @param expr1 expression1
 * @param expr2 expression2
 * @return true, if equal
 */
bool expressionEqual::operator ()(const ex expr1, const ex expr2) const {
	counter++;
	return expr1.is_equal(expr2);
}

/**
 * Get hash of expression
 * @param expr expression
 * @return hash
 */
size_t expressionHash::operator ()(const ex expr) const {
	return ParametricSets::computeHash(expr);
}

/**
 * Check for equality of polynomials
 * @param pol1 polynomial1
 * @param pol2 polynomial2
 * @return true, if equal
 */
bool polynomialEqual::operator ()(constPoly pol1, constPoly pol2) const {
	if (pol1->isIrreducible() || pol2->isIrreducible()) {
		return expressionEqual().operator ()(pol1->getExpression(), pol2->getExpression());
	} else {
		return pol1 == pol2;
	}
}

/**
 * Get hash of polynomial
 * @param pol polynomial
 * @return hash
 */
size_t polynomialHash::operator ()(constPoly pol) const {
	return ParametricSets::computeHash(pol);
}

/**
 * Check for equality of constraints
 * @param con1 constraint1
 * @param con2 constraint2
 * @return true, if equal
 */
bool constraintEqual::operator ()(const Constraint* const con1, const Constraint* const con2) const {
	if (con1->operatorType == con2->operatorType) {
		if (con1->rational1 == con2->rational1) {
			return con1->rational2 == con2->rational2;
		}
	}
	return false;
}

/**
 * Get hash of constraint
 * @param con constraint
 * @return hash
 */
size_t constraintHash::operator ()(const Constraint* const con) const {
	size_t hash = 0;
	boost::hash_combine(hash, ParametricSets::computeHash(con->rational1));
	boost::hash_combine(hash, ParametricSets::computeHash(con->rational2));
	boost::hash_combine(hash, con->operatorType);
	return hash;
}

/**
 * Check for equality of pairs for gcd
 * @param pair1 first pair
 * @param pair2 second pair
 * @return true, if equal
 */
bool gcdPairEqual::operator ()(const gcdPair pair1, const gcdPair pair2) const {
	return ((pair1.first == pair2.first) && (pair1.second == pair2.second));
}

/**
 * Get hash of pair for gcd. First polynomial has lesser hash value than second.
 * @param pair gcdPair
 * @return hash
 */
size_t gcdPairHash::operator ()(const gcdPair pair) const {
	size_t seed = 0;
	boost::hash_combine(seed, ParametricSets::computeHash(pair.first));
	boost::hash_combine(seed, ParametricSets::computeHash(pair.second));
	return seed;
}

/**
 *Getter
 *@param pol1 first polynomial
 *@param pol2 second polynomial
 *@return true, if pair is in set, false otherwise
 */
bool GCDPairSet::isInGCDPairSet(constPoly pol1, constPoly pol2) const {
	return set.find(makePair(pol1, pol2)) != set.end();
}

/**
 *Insert pair into set
 *@param pol1 first polynomial
 *@param pol2 second polynomial
 */
void GCDPairSet::insertGCDPair(constPoly pol1, constPoly pol2) {
	set.insert(makePair(pol1, pol2));
}

/**
 * Make pair of polynomials where first polynomial has lesser hash value than second
 * @param pol1 first polynomial
 * @param pol2 second polynomial
 * @return pair of these polynomials
 */
gcdPair GCDPairSet::makePair(constPoly pol1, constPoly pol2) const {
	size_t hash1 = ParametricSets::computeHash(pol1);
	size_t hash2 = ParametricSets::computeHash(pol2);
	if (hash1 < hash2) {
		return make_pair(pol1, pol2);
	} else {
		return make_pair(pol2, pol1);
	}
}
}
