/**
 * Parametric SCC based Model Checking
 *  
 * This is a stand-alone tool which performs model checking
 * for parametric discrete-time Markov Chains (PDTMCs).
 * 
 * Copyright (c) 2013 RWTH Aachen University.
 * Authors: Florian Corzilius, Nils Jansen, Matthias Volk
 * 
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 * 
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the
 * GNU General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public License
 * along with this program.  If not, see
 * http://www.gnu.org/licenses/gpl.html.
 * 
 * 
 * Main Contact:
 * 
 * Nils Jansen
 * Theory of Hybrid Systems
 * RWTH Aachen
 * 52056 Aachen
 * Germany
 * nils.jansen@cs.rwth-aachen.de
 */

#ifndef PARAMETRIC_SETS_H_
#define PARAMETRIC_SETS_H_
#include <boost/unordered_set.hpp>
#include <boost/unordered_map.hpp>
#include <ginac/ginac.h>

using namespace GiNaC;

namespace parametric {
class Polynomial;
typedef const Polynomial* const constPoly;
class Constraint;
class Rational;

enum IRREDUCIBLE {
	NO, YES, DONTKNOW
};

/**
 * Hash functions for several data types
 */
class ParametricSets {
public:
	static size_t computeHash(const ex& expr);
	static size_t computeHash(const Polynomial* pol);
	static size_t computeHash(const Rational& rat);
};

/**
 * Check for equality of expressions
 */
class expressionEqual {
public:
	bool operator ()(const ex expr1, const ex expr2) const;
	static unsigned int counter;
};

/**
 * Get hash of expression
 */
class expressionHash {
public:
	size_t operator ()(const ex expr) const;
};

/**
 * Check for equality of polynomials
 */
class polynomialEqual {
public:
	bool operator ()(constPoly pol1, constPoly pol2) const;
};

/**
 * Get hash of polynomial
 */
class polynomialHash {
public:
	size_t operator ()(constPoly pol) const;
};

typedef boost::unordered_map<const ex, const Polynomial*, expressionHash, expressionEqual> PolynomialSet;
typedef PolynomialSet::const_iterator polynomial_const_it;

typedef std::list<const Polynomial*> QueuePol;
typedef QueuePol::const_iterator queuePol_const_it;
typedef boost::unordered_map<const ex, QueuePol::iterator, expressionHash, expressionEqual> PolyManageSet;
typedef PolyManageSet::const_iterator polyManage_const_it;

/**
 * Check for equality of constraints
 */
class constraintEqual {
public:
	bool operator ()(const Constraint* const con1, const Constraint* const con2) const;
};

/**
 * Get hash of constraint
 */
class constraintHash {
public:
	size_t operator ()(const Constraint* const con) const;
};

typedef boost::unordered_set<Constraint*, constraintHash, constraintEqual> ConstraintSet;
typedef ConstraintSet::const_iterator constraint_const_it;

typedef std::pair<constPoly, constPoly> gcdPair;

/**
 * Check for equality of pairs for gcd computation
 */
class gcdPairEqual {
public:
	bool operator ()(const gcdPair pair1, const gcdPair pair2) const;
};

/**
 * Get hash of pairs for gcd computation
 */
class gcdPairHash {
public:
	size_t operator ()(const gcdPair pair) const;
};

class GCDPairSet {
public:
	GCDPairSet() {
	}
	GCDPairSet(const GCDPairSet& pairset) :
			set(pairset.set) {
	}
	~GCDPairSet() {
		set.clear();
	}
	bool isInGCDPairSet(constPoly, constPoly) const;
	void insertGCDPair(constPoly, constPoly);

private:
	gcdPair makePair(constPoly, constPoly) const;
	boost::unordered_set<gcdPair, gcdPairHash, gcdPairEqual> set;
};
}
#endif /* PARAMETRIC_SETS_H_ */
