/**
 * Parametric SCC based Model Checking
 *  
 * This is a stand-alone tool which performs model checking
 * for parametric discrete-time Markov Chains (PDTMCs).
 * 
 * Copyright (c) 2013 RWTH Aachen University.
 * Authors: Florian Corzilius, Nils Jansen, Matthias Volk
 * 
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 * 
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the
 * GNU General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public License
 * along with this program.  If not, see
 * http://www.gnu.org/licenses/gpl.html.
 * 
 * 
 * Main Contact:
 * 
 * Nils Jansen
 * Theory of Hybrid Systems
 * RWTH Aachen
 * 52056 Aachen
 * Germany
 * nils.jansen@cs.rwth-aachen.de
 */

#include "../defines.h"
#include "Polynomial2.h"
#include "Parameters.h"
#include "Cancellator2.h"
#include <sstream>

#ifdef USE_LOGLIB
#include <log4cplus/logger.h>
#endif

using namespace std;
using namespace log4cplus;

namespace parametric {

/**
 * Constructor
 * @param expr expression
 * @return
 */
Polynomial::Polynomial(const ex &expr) :
		poly(expr) {
	number = is_a<numeric>(poly);

#ifdef DEBUGGING
	if (number) {
		assert(ex_to<numeric>(poly).is_integer());
	} else {
		assert(poly.is_polynomial(Parameters::getInstance().getParameterList()));
	}
#endif

#ifdef USE_POLY_MANAGEMENT
	usageCounter = 0;
#endif

	factors = new Factorization(this);
	irreducible = DONTKNOW;
	checkIrreducibility();
}

/**
 * Copy constructor. Should not be called
 * @param pol polynomial to copy
 */
Polynomial::Polynomial(const Polynomial& pol) {
	factors = pol.factors;
	number = pol.number;
	poly = pol.poly;
	irreducible = pol.irreducible;

#ifdef USE_POLY_MANAGEMENT
	usageCounter = pol.usageCounter;
#endif

	assert(false);
}

/**
 * Destructor
 * @return
 */
Polynomial::~Polynomial() {
	delete factors;
}

/**
 * Output
 * @param stream
 * @param pol
 * @return
 */
std::ostream &operator<<(std::ostream &stream, constPoly pol) {
	stream << *(pol->factors);
	return stream;
}

/**
 * Addition
 * @param pol polynomial
 * @return result
 */
constPoly Polynomial::add(constPoly pol) const {
//Avoid identity element 0
	if (this->isNumber() && this->getNumeric() == 0) {
		return pol;
	}
	if (pol->isNumber() && pol->getNumeric() == 0) {
		return this;
	}

//Default case
	constPoly result = Parameters::getInstance().createPolynomial((this->poly + pol->poly).expand());
//Factors are ignored after addition. They are computed by GCD for rationals.
	return result;
}

/**
 * Multiplication
 * @param pol polynomial
 * @return result
 */
constPoly Polynomial::times(constPoly pol) const {
//Avoid identity element 1
	if (*this == (numeric) 1) {
		return pol;
	}
	if (*pol == (numeric) 1) {
		return this;
	}

//Avoid 0
	if (*this == (numeric) 0) {
		return this;
	}
	if (*pol == (numeric) 0) {
		return pol;
	}

//Default case
	constPoly result = Parameters::getInstance().createPolynomial(this->poly * pol->poly);
	result->factors->setFactors(this, pol);
	return result;
}

/**
 * Power function this^exponent
 * @param exponent exponent
 * @return result
 */
constPoly Polynomial::powerOwn(unsigned int exponent) const {
//Avoid identity element 1 and 0
	if (exponent == 1 || *this == (numeric) 1 || *this == (numeric) 0) {
		return this;
	}

//Avoid 0
	if (exponent == 0) {
		return Parameters::getInstance().getOne();
	}

//Default case
	constPoly result = Parameters::getInstance().createPolynomial(pow(this->poly, exponent));
	result->factors->setFactorPower(this, exponent);
	return result;
}

bool operator==(const Polynomial& pol, const numeric& num) {
	if (pol.isNumber()) {
		numeric num1 = ex_to<numeric>(pol.poly);
		return num1 == num;
	} else {
		return false;
	}
}

TRIBOOL operator==(const Polynomial& pol1, const Polynomial& pol2) {
	if (pol2.isNumber()) {
		numeric num2 = ex_to<numeric>(pol2.poly);
		if (pol1 == num2) {
			return TRIBOOL::TRUE;
		} else {
			return TRIBOOL::FALSE;
		}
	} else {
		if (&pol1 == &pol2) {
			//Point to same polynomial
			return TRIBOOL::TRUE;
#ifdef BETTER_COMP
		} else if (pol1.poly.expand().is_equal(pol2.poly.expand())) {
			return TRIBOOL::TRUE;
#endif
		} else {
			return TRIBOOL::UNDEF;
		}
	}
}

bool operator!=(const Polynomial& pol1, const numeric& num) {
	return !(pol1 == num);
}

TRIBOOL operator!=(const Polynomial& pol1, const Polynomial& pol2) {
	TRIBOOL result = pol1 == pol2;
	if (result == TRIBOOL::TRUE) {
		return TRIBOOL::FALSE;
	} else if (result == TRIBOOL::FALSE) {
		return TRIBOOL::TRUE;
	} else {
		return TRIBOOL::UNDEF;
	}
}

TRIBOOL operator<(const Polynomial& pol1, const Polynomial& pol2) {
	if (pol1.isNumber() && pol2.isNumber()) {
		numeric num1 = ex_to<numeric>(pol1.poly);
		numeric num2 = ex_to<numeric>(pol2.poly);
		if (num1 < num2) {
			return TRIBOOL::TRUE;
		} else {
			return TRIBOOL::FALSE;
		}
	} else {

#ifdef BETTER_COMP
		ex expr = pol2.poly - pol1.poly;
		if (expr.expand().is_zero()) {
			return TRIBOOL::FALSE;
		}
#endif
		return TRIBOOL::UNDEF;
	}
}

TRIBOOL operator<=(const Polynomial& pol1, const Polynomial& pol2) {
	if ((pol1 == pol2) == TRIBOOL::TRUE) {
		return TRIBOOL::TRUE;
	} else {
		return pol1 < pol2;
	}
}

TRIBOOL operator>(const Polynomial& pol1, const Polynomial& pol2) {
	return pol2 < pol1;
}

TRIBOOL operator>=(const Polynomial& pol1, const Polynomial& pol2) {
	if ((pol1 == pol2) == TRIBOOL::TRUE) {
		return TRIBOOL::TRUE;
	} else {
		return pol1 > pol2;
	}
}

/**Checks, if polynomial is irreducible
 * @return true, if irreducible
 */
bool Polynomial::isIrreducible() const {
	checkIrreducibility();
	return (irreducible == YES) ? true : false;
}

/**
 * Checks, if polynomial is irreducible and update variable
 */
void Polynomial::checkIrreducibility() const {
	if (irreducible != DONTKNOW) {
		return;
	}

	if (!factors->isEmpty()) {
		irreducible = NO;
		return;
	}

	if (isNumber()) {
		//TODO avoid numbers?
		if (getNumeric() == 1 || getNumeric() == 0 || getNumeric().is_prime()) {
			irreducible = YES;
		} else {
			irreducible = NO;
		}
		return;
	}

//Check if linear
	unsigned upperDegree;
	signed result = sign(poly, upperDegree);
	assert(collectProperties(poly.expand()) >= (result == 2 || result == -2 || upperDegree <= 1));
	irreducible = result ? YES : NO;
}

/**
 * Collects some properties of the constraint. Needs only to be applied once.
 */
bool Polynomial::collectProperties(const ex& polynomial) {
	bool mIsNeverPositive = true;
	bool mIsNeverNegative = true;
	unsigned int mMaxMonomeDegree = 1;
	numeric mConstantPart = 0;
	if (GiNaC::is_exactly_a<GiNaC::add>(polynomial)) {
		for (GiNaC::const_iterator summand = polynomial.begin(); summand != polynomial.end(); ++summand) {
			const ex summandEx = *summand;
			if (is_exactly_a<mul>(summandEx)) {
				unsigned monomDegree = 0;
				numeric coefficient = 1;
				for (GiNaC::const_iterator factor = summandEx.begin(); factor != summandEx.end(); ++factor) {
					const ex factorEx = *factor;
					if (is_exactly_a<symbol>(factorEx)) {
						mIsNeverPositive = false;
						mIsNeverNegative = false;
						++monomDegree;
					} else if (is_exactly_a<numeric>(factorEx)) {
						if (factorEx.info(info_flags::negative)) {
							mIsNeverNegative = false;
						} else {
							mIsNeverPositive = false;
						}
						coefficient *= ex_to<numeric>(factorEx);
					} else if (is_exactly_a<power>(factorEx)) {
						assert(factorEx.nops() == 2);
						ex exponent = *(++(factorEx.begin()));
						assert(!exponent.info(info_flags::negative));
						unsigned exp = static_cast<unsigned>(exponent.integer_content().to_int());
						if (fmod(exp, 2.0) != 0.0) {
							mIsNeverPositive = false;
							mIsNeverNegative = false;
						}
						ex subterm = *factorEx.begin();
						assert(is_exactly_a<symbol>(subterm));
						monomDegree += exp;
					} else
						assert(false);
				}
				if (monomDegree > mMaxMonomeDegree) {
					mMaxMonomeDegree = monomDegree;
				}
			} else if (is_exactly_a<symbol>(summandEx)) {

				mIsNeverPositive = false;
				mIsNeverNegative = false;
			} else if (is_exactly_a<numeric>(summandEx)) {
				mConstantPart += ex_to<numeric>(summandEx);
				if (summandEx.info(info_flags::negative)) {
					mIsNeverNegative = false;
				} else {
					mIsNeverPositive = false;
				}
			} else if (is_exactly_a<power>(summandEx)) {
				assert(summandEx.nops() == 2);
				ex exponent = *(++(summandEx.begin()));
				assert(!exponent.info(info_flags::negative));
				unsigned exp = static_cast<unsigned>(exponent.integer_content().to_int());
				mIsNeverPositive = false;
				if (fmod(exp, 2.0) != 0.0) {
					mIsNeverNegative = false;
				}
				ex subterm = *summandEx.begin();
				assert(is_exactly_a<symbol>(subterm));
				if (exp > mMaxMonomeDegree) {
					mMaxMonomeDegree = exp;
				}
			} else
				assert(false);
		}
	} else if (is_exactly_a<mul>(polynomial)) {
		unsigned monomDegree = 0;
		bool hasCoefficient = false;
		for (GiNaC::const_iterator factor = polynomial.begin(); factor != polynomial.end(); ++factor) {
			if (mMaxMonomeDegree > 1 && !(mMaxMonomeDegree == 2 && mConstantPart.is_negative() && mIsNeverPositive)
					&& !(mMaxMonomeDegree == 2 && mConstantPart.is_positive() && mIsNeverNegative)) {
				return false;
			}

			const ex factorEx = *factor;
			if (is_exactly_a<symbol>(factorEx)) {
				mIsNeverPositive = false;
				mIsNeverNegative = false;
				++monomDegree;
			} else if (is_exactly_a<numeric>(factorEx)) {
				if (factorEx.info(info_flags::negative)) {
					mIsNeverNegative = false;
				} else {
					mIsNeverPositive = false;
				}
				hasCoefficient = true;
			} else if (is_exactly_a<power>(factorEx)) {
				assert(factorEx.nops() == 2);
				ex exponent = *(++factorEx.begin());
				assert(!exponent.info(info_flags::negative));
				unsigned exp = static_cast<unsigned>(exponent.integer_content().to_int());
				if (fmod(exp, 2.0) != 0.0) {
					mIsNeverPositive = false;
					mIsNeverNegative = false;
				}
				ex subterm = *factorEx.begin();
				assert(is_exactly_a<symbol>(subterm));
				monomDegree += exp;
			} else
				assert(false);
		}
		if (!hasCoefficient) {
			mIsNeverPositive = false;
		}
		if (monomDegree > mMaxMonomeDegree) {
			mMaxMonomeDegree = monomDegree;
		}
	} else if (is_exactly_a<symbol>(polynomial)) {
		return true;
	} else if (is_exactly_a<numeric>(polynomial)) {
		return ex_to<numeric>(polynomial).is_prime();
	} else if (is_exactly_a<power>(polynomial)) {
		return false;
	} else {
		assert(false);
	}

	if (mMaxMonomeDegree <= 1
			|| (mMaxMonomeDegree == 2 && ((mConstantPart < 0 && mIsNeverPositive) || (mConstantPart > 0 && mIsNeverNegative)))) {
		return true;
	}
	return false;
}

/**
 * @param _ex A polynomial.
 * @param _upperDegreeBound An upper bound for the degree of the given polynomial, which is calculated as a side product of this method.
 * @return 2, if the expression is greater 0 for all assignments of its variables
 *          1, if the expression is greater or equal 0 for all assignments of its variables
 *          -1, if the expression is less or equal 0 for all assignments of its variables
 *          -2, if the expression is less 0 for all assignments of its variables
 *          0, if any of the aforementioned properties cannot be decided by this method
 */
signed Polynomial::sign(const ex& _ex, unsigned& _upperDegreeBound) {
	if (is_exactly_a<GiNaC::add>(_ex)) {
		GiNaC::const_iterator summand = _ex.begin();
		signed result = sign(*summand, _upperDegreeBound);
		++summand;
		unsigned tmpLDB;
		while (summand != _ex.end()) {
			if (result == 0 && _upperDegreeBound > 1)
				return 0;
			switch (sign(*summand, tmpLDB)) {
			case -2: // .. +  something which is always negative (<0)
				if (result >= 0)
					result = 0;
				else if (result == -1)
					result = -2;
				break;
			case -1: // .. +  something which is never positive (<=0)
				if (result >= 0)
					result = 0;
				break;
			case 0:
				result = 0;
				break;
			case 1: // .. +  something which is never negative (>=0)
				if (result <= 0)
					result = 0;
				break;
			case 2: // .. +  something which is always positive (>0)
				if (result <= 0)
					result = 0;
				else if (result == 1)
					result = 2;
				break;
			default:
				assert(false);
				break;
			}
			if (tmpLDB > _upperDegreeBound)
				_upperDegreeBound = tmpLDB;
			++summand;
		}
		return result;
	} else if (is_exactly_a<mul>(_ex)) {
		GiNaC::const_iterator factor = _ex.begin();
		signed result = sign(*factor, _upperDegreeBound);
		++factor;
		unsigned tmpLDB;
		while (factor != _ex.end()) {
			if (result == 0 && _upperDegreeBound > 1)
				return 0;
			switch (sign(*factor, tmpLDB)) {
			case -2: // .. *  something which is always negative (<0)
				if (result >= 0)
					result = 0;
				break;
			case -1: // .. *  something which is never positive (<=0)
				if (result >= 0)
					result = 0;
				else if (result == -2)
					result = -1;
				break;
			case 0:
				result = 0;
				break;
			case 1: // .. *  something which is never negative (>=0)
				if (result <= 0)
					result = 0;
				else if (result == 2)
					result = 1;
				break;
			case 2: // .. *  something which is always positive (>0)
				if (result <= 0)
					result = 0;
				break;
			default:
				assert(false);
				break;
			}
			_upperDegreeBound += tmpLDB;
			++factor;
		}
		return result;
	} else if (is_exactly_a<symbol>(_ex)) {
		_upperDegreeBound = 1;
		return 0;
	} else if (is_exactly_a<numeric>(_ex)) {
		_upperDegreeBound = 0;
		if (_ex.info(info_flags::negative)) {
			return -2;
		} else {
			assert(_ex.info(info_flags::positive));
			return 2;
		}
	} else {
		assert(is_exactly_a<power>(_ex));
		assert(_ex.nops() == 2);
		ex exponent = *(++(_ex.begin()));
		assert(exponent > 1);
		const ex& subterm = *_ex.begin();
		signed result = sign(subterm, _upperDegreeBound);
		unsigned exp = static_cast<unsigned>(exponent.integer_content().to_int());
		_upperDegreeBound *= exp;
		if (fmod(exp, 2.0) == 0.0) {
			return 1;
		} else {
			return result;
		}
	}
}

#ifdef USE_POLY_MANAGEMENT
/**
 * Decrement usage counter and delete polynomial if not used anymore
 * @param root signal root for recursion
 */
void Polynomial::decUsageCounter(bool root) const {
	usageCounter--;
	if (usageCounter == 0) {
		Parameters::getInstance().deletePolynomial(this, root);
	} else if (usageCounter < 0) {
		assert(*this == 0 || *this == -1 || *this == 1);
	}
}
#endif

/**
 * Substitute variables and return polynomial
 * @param substitutions variable->number to apply
 * @return result as polynomial
 */
constPoly Polynomial::substitute(const exmap& substitutions) const {
	return Parameters::getInstance().createPolynomial(poly.subs(substitutions));
}

/**
 * Substitute variables and return expression
 * @param substitutions variable->number to apply
 * @return result as expression
 */
ex Polynomial::substituteEx(const exmap& substitutions) const {
	return poly.subs(substitutions);
}

/**
 * Factorize polynomial
 */
void Polynomial::factorize() const {
	ex factorization = factor(poly, factor_options::all);
	factors->setFactorization(factorization);
}

/**
 * Get string with prefix notation of polynomial as input for tools
 * @return string
 */
std::string Polynomial::toPrefixNotation() const {
	return Polynomial::exToPrefixNotation(poly);
}

/**
 * Helper function for recursive traversal of expression
 * @param e expression
 * @return string with prefix notation
 */
std::string Polynomial::exToPrefixNotation(const ex& e) {
	stringstream stream;
	size_t n = e.nops();
	if (n) {
		stream << "(";
	}

	if (is_a<expairseq>(e)) {
		if (is_a<mul>(e)) {
			stream << "* ";
		} else {
			stream << "+ ";
		}
	} else if (is_a<power>(e)) {
		stream << "^ ";
	} else if (is_a<numeric>(e)) {
		stream << e.evalf();
		return stream.str();
	}

	if (n) {
		for (size_t i = 0; i < n; i++) {
			stream << Polynomial::exToPrefixNotation(e.op(i));
			if (i != n - 1) {
				stream << " ";
			}
		}
		stream << ")";
	} else {
		stream << e;
	}

	return stream.str();
}
}
