/**
 * Parametric SCC based Model Checking
 *  
 * This is a stand-alone tool which performs model checking
 * for parametric discrete-time Markov Chains (PDTMCs).
 * 
 * Copyright (c) 2013 RWTH Aachen University.
 * Authors: Florian Corzilius, Nils Jansen, Matthias Volk
 * 
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 * 
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the
 * GNU General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public License
 * along with this program.  If not, see
 * http://www.gnu.org/licenses/gpl.html.
 * 
 * 
 * Main Contact:
 * 
 * Nils Jansen
 * Theory of Hybrid Systems
 * RWTH Aachen
 * 52056 Aachen
 * Germany
 * nils.jansen@cs.rwth-aachen.de
 */

#ifndef POLYNOMIAL_H
#define POLYNOMIAL_H

#include "ParametricSets.h"
#include "Factorization.h"
#include "TriBool.h"
#include <vector>
#include <string>
#include <ginac/ginac.h>
#include <boost/logic/tribool.hpp>

using namespace GiNaC;
using namespace boost;
using namespace param_sccmc;

namespace parametric {

/**
 * Polynomial as GiNaC's ex. A factorization tree is also stored for later simplification.
 */
class Polynomial {
	friend class Rational;
	friend class Parameters;
public:
	~Polynomial();

	friend std::ostream &operator<<(std::ostream&, constPoly);

	constPoly add(constPoly) const;
	constPoly times(constPoly) const;
	constPoly powerOwn(unsigned int) const;

	friend bool operator==(const Polynomial&, const numeric&);
	friend TRIBOOL operator==(const Polynomial&, const Polynomial&);
	friend bool operator!=(const Polynomial&, const numeric&);
	friend TRIBOOL operator!=(const Polynomial&, const Polynomial&);
	friend TRIBOOL operator<(const Polynomial&, const Polynomial&);
	friend TRIBOOL operator<=(const Polynomial&, const Polynomial&);
	friend TRIBOOL operator>(const Polynomial&, const Polynomial&);
	friend TRIBOOL operator>=(const Polynomial&, const Polynomial&);

	std::string toPrefixNotation() const;
	bool isIrreducible() const;
	void factorize() const;

	inline bool isNumber() const {
		return number;
	}

	inline numeric getNumeric() const {
		return ex_to<numeric>(poly);
	}

	inline const ex getExpression() const {
		return poly;
	}

	inline Factorization* getFactorization() const {
		return factors;
	}

#ifdef USE_POLY_MANAGEMENT
	inline void incUsageCounter() const {
		assert(usageCounter<INT_MAX);
		usageCounter++;
	}

	inline unsigned int getUsageCounter() const {
		return usageCounter;
	}

	void decUsageCounter(bool root = true) const;
#endif

private:
	Polynomial(const ex&);
	Polynomial(const Polynomial& pol);

	void checkIrreducibility() const;
	static signed sign(const ex&, unsigned&);
	static bool collectProperties(const ex&);
	static std::string exToPrefixNotation(const ex&);
	constPoly substitute(const exmap&) const;
	ex substituteEx(const exmap&) const;

	Factorization* factors;
	bool number;
	ex poly;
	mutable IRREDUCIBLE irreducible;

#ifdef USE_POLY_MANAGEMENT
	mutable int usageCounter;
#endif
};
}

#endif
