/**
 * Parametric SCC based Model Checking
 *  
 * This is a stand-alone tool which performs model checking
 * for parametric discrete-time Markov Chains (PDTMCs).
 * 
 * Copyright (c) 2013 RWTH Aachen University.
 * Authors: Florian Corzilius, Nils Jansen, Matthias Volk
 * 
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 * 
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the
 * GNU General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public License
 * along with this program.  If not, see
 * http://www.gnu.org/licenses/gpl.html.
 * 
 * 
 * Main Contact:
 * 
 * Nils Jansen
 * Theory of Hybrid Systems
 * RWTH Aachen
 * 52056 Aachen
 * Germany
 * nils.jansen@cs.rwth-aachen.de
 */

#include "../defines.h"
#include "Rational.h"
#include "Parameters.h"
#include "Cancellator2.h"
#include <sstream>

#ifdef USE_LOGLIB
#include <log4cplus/logger.h>
#endif

using namespace std;
using namespace log4cplus;

namespace parametric {

/**
 * Constructor
 * @param num numeric
 * @return
 */
Rational::Rational(const int& num) :
		numerator(Parameters::getInstance().createPolynomial(num)), denominator(Parameters::getInstance().getOne()) {
#ifdef USE_POLY_MANAGEMENT
	numerator->incUsageCounter();
	denominator->incUsageCounter();
#endif
}

/**
 * Constructor
 * @param expr1 numerator
 * @param expr2 denominator
 * @return
 */
Rational::Rational(const ex &expr1, const ex &expr2) :
		numerator(Parameters::getInstance().createPolynomial(expr1)), denominator(
				Parameters::getInstance().createPolynomial(expr2)) {
	assert(expr2.expand() != 0);
#ifdef USE_POLY_MANAGEMENT
	numerator->incUsageCounter();
	denominator->incUsageCounter();
#endif
}

/**
 * Constructor
 * @param pol polynomial
 * @return
 */
Rational::Rational(constPoly pol) :
		numerator(pol), denominator(Parameters::getInstance().getOne()) {
	assert(pol != NULL);
#ifdef USE_POLY_MANAGEMENT
	numerator->incUsageCounter();
	denominator->incUsageCounter();
#endif
}

/**
 * Constructor
 * @param rat rational function
 * @return
 */
Rational::Rational(const Rational &rat) :
		numerator(rat.numerator), denominator(rat.denominator) {
	assert(rat.numerator != NULL);
	assert(rat.denominator != NULL);
	assert(rat.denominator != 0);
#ifdef USE_POLY_MANAGEMENT
	numerator->incUsageCounter();
	denominator->incUsageCounter();
#endif
}

/**
 * Destructor
 * @return
 */
Rational::~Rational() {
#ifdef USE_POLY_MANAGEMENT
	numerator->decUsageCounter();
	denominator->decUsageCounter();
#endif
	numerator = NULL;
	denominator = NULL;
}

/**
 * Output
 * @param stream
 * @param rat
 * @return
 */
std::ostream &operator<<(std::ostream &stream, const Rational& rat) {
	stream << rat.toString();
	return stream;
}

/**
 * Assignment
 * @param rat
 * @return
 */
Rational &Rational::operator=(const Rational& rat) {
	assert(rat.numerator != NULL);
	assert(rat.denominator != NULL);
	setNumerator(rat.numerator);
	setDenominator(rat.denominator);
	return *this;
}

/**
 * Set new numerator and update usageCounters
 * @param pol numerator
 */
void Rational::setNumerator(constPoly pol) {
	assert(pol != NULL);
#ifdef USE_POLY_MANAGEMENT
	numerator->decUsageCounter();
#endif
	numerator = pol;
#ifdef USE_POLY_MANAGEMENT
	numerator->incUsageCounter();
#endif
}

/**
 * Set new denominator and update usageCounters
 * @param pol denominator
 */
void Rational::setDenominator(constPoly pol) {
	assert(pol != NULL);
#ifdef USE_POLY_MANAGEMENT
	denominator->decUsageCounter();
#endif
	denominator = pol;
#ifdef USE_POLY_MANAGEMENT
	denominator->incUsageCounter();
#endif
}

/**
 * Addition
 * @param pol polynomial
 */
void Rational::add(constPoly pol) {
	//Avoid identity element 0
	if (*pol == 0 || *this == 0) {
		//Nothing changes
	} else {
		//Default case
		setNumerator(numerator->add(denominator->times(pol)));
		Cancellator::getInstance().cancel(*this, true);
	}
}

/**
 * Addition
 * @param rat rational function
 */
void Rational::add(const Rational& rat) {
	//Avoid identity element 0
	if (rat == 0) {
		//Nothing changes
	} else if (*this == 0) {
		//Result is rat
		*this = rat;
	} else {
		//Default case
		*this = Cancellator::getInstance().additionFactorization(*this, rat);
	}
}

/**
 * Subtraction
 * @param pol polynomial
 */
void Rational::sub(constPoly pol) {
	add(pol->times(Parameters::getInstance().getMinusOne()));
}

/**
 * Subtraction
 * @param rat rational function
 */
void Rational::sub(const Rational& rat) {
	Rational tmp = rat;
	tmp.times(Parameters::getInstance().getMinusOne());
	add(tmp);
}

/**
 * Multiplication
 * @param pol polynomial
 */
void Rational::times(constPoly pol) {
	//Avoid identity element 1 and 0
	if (*pol == 1 || *this == 0) {
		//Nothing changes
	} else if (*pol == 0 || *this == 1) {
		//Result is pol
		setNumerator(pol);
		setDenominator(Parameters::getInstance().getOne());
	} else {
		//Default case
		setNumerator(numerator->times(pol));
		Cancellator::getInstance().cancel(*this, false);
	}
}

/**
 * Multiplication
 * @param rat rational function
 */
void Rational::times(const Rational& rat) {
	//Avoid identity element 1 and 0
	if (rat == 1 || *this == 0) {
		//Nothing changes
	} else if (rat == 0 || *this == 1) {
		//Result is rat
		*this = rat;
	} else {
		//Default case
		setNumerator(numerator->times(rat.numerator));
		setDenominator(denominator->times(rat.denominator));
		Cancellator::getInstance().cancel(*this, false);
	}
}

/**
 * Division
 * @param pol polynomial
 */
void Rational::div(constPoly pol) {
	assert(pol != 0);
	//Avoid identity element 1 and 0
	if (*pol == 1 || *this == 0) {
		//Nothing changes
	} else if (*this == 1) {
		//Result is 1/pol
		setNumerator(Parameters::getInstance().getOne());
		setDenominator(pol);
	} else {
		//Default case
		setDenominator(denominator->times(pol));
		Cancellator::getInstance().cancel(*this, false);
	}
}

/**
 * Division
 * @param rat rational function
 */
void Rational::div(const Rational&rat) {
	assert(rat != 0);
	//Avoid identity element 1 and 0
	if (rat == 1 || *this == 0) {
		//Nothing changes
	} else if (*this == 1) {
		//Result is inverse of rat
		setNumerator(rat.denominator);
		setDenominator(rat.numerator);
	} else {
		//Default case
		setNumerator(numerator->times(rat.denominator));
		setDenominator(denominator->times(rat.numerator));
		Cancellator::getInstance().cancel(*this, false);
	}
}

Rational &Rational::operator+=(constPoly pol) {
	add(pol);
	return *this;
}

Rational &Rational::operator+=(const Rational& rat) {
	add(rat);
	return *this;
}

Rational &Rational::operator-=(constPoly pol) {
	sub(pol);
	return *this;
}

Rational &Rational::operator-=(const Rational& rat) {
	sub(rat);
	return *this;
}

Rational &Rational::operator*=(constPoly pol) {
	times(pol);
	return *this;
}

Rational &Rational::operator*=(const Rational& rat) {
	times(rat);
	return *this;
}

Rational &Rational::operator/=(constPoly pol) {
	div(pol);
	return *this;
}

Rational &Rational::operator/=(const Rational& rat) {
	div(rat);
	return *this;
}

Rational operator+(constPoly pol, const Rational& rat) {
	Rational result = rat;
	return result += pol;
}

Rational operator+(const Rational& rat, constPoly pol) {
	Rational result = rat;
	return result += pol;
}

Rational operator+(const Rational& rat1, const Rational& rat2) {
	Rational result = rat1;
	return result += rat2;
}

Rational operator-(constPoly pol, const Rational& rat) {
	Rational result = rat;
	return result -= pol;
}

Rational operator-(const Rational& rat, constPoly pol) {
	Rational result = rat;
	return result -= pol;
}

Rational operator-(const Rational& rat1, const Rational& rat2) {
	Rational result = rat1;
	return result -= rat2;
}

Rational operator*(constPoly pol, const Rational& rat) {
	Rational result = rat;
	return result *= pol;
}

Rational operator*(const Rational& rat, constPoly pol) {
	Rational result = rat;
	return result *= pol;
}

Rational operator*(const Rational& rat1, const Rational& rat2) {
	Rational result = rat1;
	return result *= rat2;
}

Rational operator/(constPoly pol, const Rational& rat) {
	Rational result = rat;
	return result /= pol;
}

Rational operator/(const Rational& rat, constPoly pol) {
	Rational result = rat;
	return result /= pol;
}

Rational operator/(const Rational& rat1, const Rational& rat2) {
	Rational result = rat1;
	return result /= rat2;
}

Rational operator-(const Rational& rat) {
	return rat * Parameters::getInstance().getMinusOne();
}

bool operator==(const Rational& rat1, const int num) {
	if (rat1.isNumber()) {
		return rat1.getNumeric() == num;
	} else {
		return false;
	}
}

bool operator==(const Rational& rat1, const Rational& rat2) {
	if (rat1.isNumber()) {
		if (rat2.isNumber()) {
			return rat1.getNumeric() == rat2.getNumeric();
		} else {
			return false;
		}
	} else if (rat2.isNumber()) {
		return false;
	} else {
		//TODO make better
		//Do not compute new rational function which conflicts with polynomial management
		return rat1.getExpression() == rat2.getExpression();
	}
}

bool operator!=(const Rational& rat1, const int num) {
	return !(rat1 == num);
}

bool operator!=(const Rational& rat1, const Rational& rat2) {
	return !(rat1 == rat2);
}

TRIBOOL operator<(const Rational& rat1, const Rational& rat2) {
	if (rat1.isNumber()) {
		if (rat2.isNumber()) {
			//Both numbers
			if (rat1.getNumeric() < rat2.getNumeric()) {
				return TRIBOOL::TRUE;
			} else {
				return TRIBOOL::FALSE;
			}
		} else {
			//only rat1 is number
			return TRIBOOL::UNDEF;
		}
	} else {
		if (rat2.isNumber()) {
			//only rat2 is number
			return TRIBOOL::UNDEF;
		} else {
			//both are not numbers

			//TODO activate again
			//Do not compute new rational function which conflicts with polynomial management
			//Rational rat = rat2 - rat1;
			//if (rat.isNumber()) {
			//	if (rat.getNumeric() == 0) {
			//		return TRIBOOL::TRUE;
			//	} else {
			//		return TRIBOOL::FALSE;
			//	}
			//}

			//Prove at least inequality
			if (rat1 == rat2) {
				return TRIBOOL::FALSE;
			} else {
				return TRIBOOL::UNDEF;
			}
		}
	}
}

TRIBOOL operator<=(const Rational& rat1, const Rational& rat2) {
	if (rat1 == rat2) {
		return TRIBOOL::TRUE;
	} else {
		return rat1 < rat2;
	}
}

TRIBOOL operator>(const Rational& rat1, const Rational& rat2) {
	return rat2 < rat1;
}

TRIBOOL operator>=(const Rational& rat1, const Rational& rat2) {
	if (rat1 == rat2) {
		return TRIBOOL::TRUE;
	} else {
		return rat1 > rat2;
	}
}

/**
 * Checks if numeric
 * @return true, if numeric
 */
bool Rational::isNumber() const {
	return numerator->isNumber() && denominator->isNumber();
}

/**
 * Substitute parameters
 * @param substitutions substitutions to existing parameters
 * @return evaluated rational function
 */
Rational Rational::substitute(const exmap &substitutions) const {
	Rational rat(numerator->substitute(substitutions));
	rat.setDenominator(denominator->substitute(substitutions));
	return rat;
}

/**
 * Factorize rational function
 */
void Rational::factorize() const {
	numerator->factorize();
	denominator->factorize();
}

/**
 * Expression for GiNaC
 * @return ex
 */
ex Rational::getExpression() const {
	return (numerator->getExpression()) / (denominator->getExpression());
}

/**
 * Output
 * @return string
 */
std::string Rational::toString() const {
	std::stringstream stream;
	if (*denominator == 1) {
		stream << numerator;
	} else {
		if (numerator->isNumber()) {
			stream << numerator;
		} else {
			stream << "(" << numerator << ")";
		}
		stream << " / ";
		if (denominator->isNumber()) {
			stream << denominator;
		} else {
			stream << "(" << denominator << ")";
		}
	}
	return stream.str();
}

/**
 * Output in prefix notation
 * @return string
 */
std::string Rational::toPrefixNotation() const {
	std::stringstream stream;
	if (*denominator == 1) {
		stream << numerator->toPrefixNotation();
	} else {
		stream << "(/";
		stream << " " << numerator->toPrefixNotation();
		stream << " " << denominator->toPrefixNotation();
		stream << ")";
	}
	return stream.str();
}
}
