/**
 * Parametric SCC based Model Checking
 *  
 * This is a stand-alone tool which performs model checking
 * for parametric discrete-time Markov Chains (PDTMCs).
 * 
 * Copyright (c) 2013 RWTH Aachen University.
 * Authors: Florian Corzilius, Nils Jansen, Matthias Volk
 * 
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 * 
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the
 * GNU General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public License
 * along with this program.  If not, see
 * http://www.gnu.org/licenses/gpl.html.
 * 
 * 
 * Main Contact:
 * 
 * Nils Jansen
 * Theory of Hybrid Systems
 * RWTH Aachen
 * 52056 Aachen
 * Germany
 * nils.jansen@cs.rwth-aachen.de
 */

#ifndef RATIONAL_H
#define RATIONAL_H

#include <vector>
#include <string>
#include <ginac/ginac.h>
#include "Polynomial2.h"
#include "../typedefs.h"

using namespace GiNaC;

namespace parametric {

/**
 * Rational function "numerator/denominator"
 */
class Rational {
	friend class Cancellator;
public:
	Rational(const int &num = 0);
	Rational(const ex &expr1, const ex &expr2);
	Rational(constPoly);
	Rational(const Rational&);
	~Rational();

	friend std::ostream &operator<<(std::ostream&, const Rational&);

	Rational &operator=(const Rational&);

	void add(constPoly);
	void add(const Rational&);
	void sub(constPoly);
	void sub(const Rational&);
	void times(constPoly);
	void times(const Rational&);
	void div(constPoly);
	void div(const Rational&);

	Rational &operator+=(constPoly);
	Rational &operator+=(const Rational&);
	Rational &operator-=(constPoly);
	Rational &operator-=(const Rational&);
	Rational &operator*=(constPoly);
	Rational &operator*=(const Rational&);
	Rational &operator/=(constPoly);
	Rational &operator/=(const Rational&);

	friend Rational operator+(constPoly, const Rational&);
	friend Rational operator+(const Rational&, constPoly);
	friend Rational operator+(const Rational&, const Rational&);
	friend Rational operator-(constPoly, const Rational&);
	friend Rational operator-(const Rational&, constPoly);
	friend Rational operator-(const Rational&, const Rational&);
	friend Rational operator*(constPoly, const Rational&);
	friend Rational operator*(const Rational&, constPoly);
	friend Rational operator*(const Rational&, const Rational&);
	friend Rational operator/(constPoly, const Rational&);
	friend Rational operator/(const Rational&, constPoly);
	friend Rational operator/(const Rational&, const Rational&);

	friend Rational operator-(const Rational&);

	friend bool operator==(const Rational&, const int);
	friend bool operator==(const Rational&, const Rational&);
	friend bool operator!=(const Rational&, const int);
	friend bool operator!=(const Rational&, const Rational&);
	friend TRIBOOL operator<(const Rational&, const Rational&);
	friend TRIBOOL operator<=(const Rational&, const Rational&);
	friend TRIBOOL operator>(const Rational&, const Rational&);
	friend TRIBOOL operator>=(const Rational&, const Rational&);

	inline numeric getNumeric() const {
		return ex_to<numeric>(getExpression());
	}

	bool isNumber() const;
	std::string toString() const;
	std::string toPrefixNotation() const;
	ex getExpression() const;
	void factorize() const;

private:
	void setNumerator(constPoly);
	void setDenominator(constPoly);
	Rational substitute(const exmap&) const;

	const Polynomial* numerator;
	const Polynomial* denominator;
};
}
#endif
