/**
 * Parametric SCC based Model Checking
 *  
 * This is a stand-alone tool which performs model checking
 * for parametric discrete-time Markov Chains (PDTMCs).
 * 
 * Copyright (c) 2013 RWTH Aachen University.
 * Authors: Florian Corzilius, Nils Jansen, Matthias Volk
 * 
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 * 
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the
 * GNU General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public License
 * along with this program.  If not, see
 * http://www.gnu.org/licenses/gpl.html.
 * 
 * 
 * Main Contact:
 * 
 * Nils Jansen
 * Theory of Hybrid Systems
 * RWTH Aachen
 * 52056 Aachen
 * Germany
 * nils.jansen@cs.rwth-aachen.de
 */

#ifndef COMMON_HEADER_H_
#define COMMON_HEADER_H_

#include <limits.h>
#include <string>

#ifdef USE_LOGLIB
#include <log4cplus/logger.h>
#else
//If no logging installed, define namespace ourselves to get this compiled
namespace log4cplus {

#define TRACE_LOG_LEVEL (1)
#define DEBUG_LOG_LEVEL (2)
#define INFO_LOG_LEVEL  (3)
#define WARN_LOG_LEVEL  (4)
#define ERROR_LOG_LEVEL (5)

extern int global_loglevel;

class Logger {
public:
	bool write(int level);

	void setLogLevel(int loglevel);

	static Logger getInstance(std::string name) {
		return Logger(name);
	}

	std::string name;
private:
	bool isLogLevelSet;
	int loglevel;
	Logger(std::string _name); //Private for consistency reasons
};

#define LOG4CPLUS_ERROR(logger, msg) ({ if ((logger).write(ERROR_LOG_LEVEL)) std::cout << "Error: " << "[" << logger.name << "] " << msg << std::endl << std::flush; })
#define LOG4CPLUS_WARN(logger, msg) ({ if ((logger).write(WARN_LOG_LEVEL)) std::cout << "Warning: " << "[" << logger.name << "] " << msg << std::endl << std::flush; })
#define LOG4CPLUS_INFO(logger, msg) ({ if ((logger).write(INFO_LOG_LEVEL)) std::cout << "Info: " << "[" << logger.name << "] " << msg << std::endl << std::flush; })
#define LOG4CPLUS_DEBUG(logger, msg) ({ if ((logger).write(DEBUG_LOG_LEVEL)) std::cout << "Debug: " << "[" << logger.name << "] " << msg << std::endl << std::flush; })
#define LOG4CPLUS_TRACE(logger, msg) ({ if ((logger).write(TRACE_LOG_LEVEL)) std::cout << "Trace: " << "[" << logger.name << "] " << msg << std::endl << std::flush; })
}
#endif

#include "typedefs.h"

#ifdef USE_EXACT
#define PTYPE mpq_class
#else
#ifdef USE_PARAMETRIC
#define PTYPE parametric::Rational
//Disabled by default
//#define USE_ABSTRACT_TRANSITIONS
#else
#define PTYPE double
#endif
#endif

#define UNDEFINED INT_MAX

#ifdef DEBUGGING
#define DEFAULT_LOG_LEVEL TRACE_LOG_LEVEL
#else
#define DEFAULT_LOG_LEVEL INFO_LOG_LEVEL
#endif

const bool CONCRETE = false;
const bool ABSTRACT = true;

//Give this as 2nd parameter to modelCheck if you want the result returned to
//Java (XML-based) instead of written to a result file
extern const char* JAVA;
extern const char* DEFAULT_RESULT_FILE;

#include <cstdio>
/*
 * External vars
 */
namespace scc_cex {
//See SCC_MC.cpp
extern DebuggingType debuggingMode;
extern int next_graph_id;

bool doesFileExist(std::string filename);
}

#ifndef CLOCKS_PER_SEC
#define CLOCKS_PER_SEC 1000000
#endif

#include <cassert>

#endif /* COMMON_HEADER_H_ */
