/**
 * Parametric SCC based Model Checking
 *  
 * This is a stand-alone tool which performs model checking
 * for parametric discrete-time Markov Chains (PDTMCs).
 * 
 * Copyright (c) 2013 RWTH Aachen University.
 * Authors: Florian Corzilius, Nils Jansen, Matthias Volk
 * 
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 * 
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the
 * GNU General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public License
 * along with this program.  If not, see
 * http://www.gnu.org/licenses/gpl.html.
 * 
 * 
 * Main Contact:
 * 
 * Nils Jansen
 * Theory of Hybrid Systems
 * RWTH Aachen
 * 52056 Aachen
 * Germany
 * nils.jansen@cs.rwth-aachen.de
 */


//TODO: Fix: Several code segments are based on continuous node numbering in the original file
//(auxiliary node ids, target node flags). This is not exactly ideal. It could be fixed with relative
//ease by using a one-to-one mapping of given node ids to the numbers 0,1,...

#include <iostream>
#include <fstream>
#include <sstream>
#include <stdlib.h>
#include <utility>

#include <sys/types.h>
#include <sys/stat.h>
#include <dirent.h>

#include "IOWrapper.h"
#include "input/ProbReader.h"
#include "input/DTMCReader.h"
#include "input/XMLReader.h"
#include "input/PMCReader.h"
#include "input/PrismReader.h"
#include "output/OutputFormatters.h"
#include "Configuration.h"

#ifdef USE_LOGLIB
#include <log4cplus/logger.h>
#include <log4cplus/configurator.h>
#endif

#ifdef USE_PARAMETRIC
#include "../data/Parameters.h"
#endif

#ifdef JNI
#include "../SCCInterface.h"
#endif

#if defined _WIN32 || defined _WIN64
#define SLASH '\\'
#else
#define SLASH '/'
#endif

using namespace std;
using namespace log4cplus;
using namespace param_sccmc_io;

const char* JAVA = "RETURN_TO_JAVA";
const char* DEFAULT_RESULT_FILE = "result.dtmc";

namespace scc_cex {

Logger iologger = Logger::getInstance("IOWrapper");

bool doesFileExist(std::string filename) {
	FILE* fp = fopen(filename.c_str(), "rb");
	if (fp != NULL) {
		fclose(fp);
		return true;
	} else {
		return false;
	}
}

IOWrapper::IOWrapper(string str, unsigned int input_flags, unsigned int heuristic_flags) {
	initLogging();

	mResultFileName = "";
	
	ConfigurationSingleton *conf = ConfigurationSingleton::getInstance();

	//Configuration::setFlags(input_flags);
	mWaitForExternalCalls = conf->isIoflWaitForCalls();

	mModelChecker = NULL;

	istream* strm;
	if (conf->isIoflInputIsFile()) {

		if (!scc_cex::doesFileExist(str)) {
			LOG4CPLUS_ERROR(iologger, "The given file (" << str << ") doesn't exist");
			return;
		}

		inputFile = str;

		//create input stream from file
		strm = new ifstream(str.c_str(), ifstream::in);

		LOG4CPLUS_INFO(iologger, "Will parse " << str);
	} else {
		//create input stream from string
		strm = new istringstream(str, ifstream::in);
	}
	init(strm);
}

IOWrapper::IOWrapper(string str, unsigned int input_flags) {
	initLogging();

	ConfigurationSingleton *conf = ConfigurationSingleton::getInstance();

	//	Configuration::setFlags(input_flags);
	mWaitForExternalCalls = conf->isIoflWaitForCalls();

	mModelChecker = NULL;

	istream* strm;
	if (conf->isIoflInputIsFile()) {

		if (!scc_cex::doesFileExist(str)) {
			LOG4CPLUS_ERROR(iologger, "The given file (" << str << ") doesn't exist");
			return;
		}

		inputFile = str;

		//create input stream from file
		strm = new ifstream(str.c_str(), ifstream::in);

		LOG4CPLUS_INFO(iologger, "Will parse " << str);
	} else {
		//create input stream from string
		strm = new istringstream(str, ifstream::in);
	}
	init(strm);
}

IOWrapper::IOWrapper(string str) {
	initLogging();

	ConfigurationSingleton *conf = ConfigurationSingleton::getInstance();

	//	Configuration::setFlags(input_flags);
	mWaitForExternalCalls = conf->isIoflWaitForCalls();

	mModelChecker = NULL;

	istream* strm;
	if (conf->isIoflInputIsFile()) {

		if (!doesFileExist(str)) {
			LOG4CPLUS_ERROR(iologger, "The given file (" << str << ") doesn't exist");
			return;
		}

		inputFile = str;

		//create input stream from file
		strm = new ifstream(str.c_str(), ifstream::in);

		LOG4CPLUS_INFO(iologger, "Will parse " << str);
	} else {
		//create input stream from string
		strm = new istringstream(str, ifstream::in);
	}
	init(strm);
}

IOWrapper::~IOWrapper() {
	delete mModelChecker;

	for (vtx_it it = mVertexList.begin(); it != mVertexList.end(); it++)
		delete *it;
	mVertexList.clear();
	mInitialNode = NULL;
}

void IOWrapper::init(istream *strm) {
	if (ConfigurationSingleton::getInstance()->isIoflExternalCaller()) {
		mResultFile = JAVA;
	} else {
		mResultFile = DEFAULT_RESULT_FILE;
	}
	mStrm = strm;
	mAtPropsGiven = false;
	mInitialNode = NULL;
	mTargetNodes.clear();
	mTargetNodes.push_back(false);
	mEdgeCount = 0;
}

void IOWrapper::initLogging() {
#ifdef USE_LOGLIB
	BasicConfigurator config;
	config.configure();
#endif
	iologger.setLogLevel(DEFAULT_LOG_LEVEL);
}

bool IOWrapper::readFiles() {
	bool couldRead = readFiles(mStrm);
	if (!couldRead) {
		LOG4CPLUS_ERROR(iologger, "The input file(s) couldn't be read - terminating.");
		return false;
	}

	//Check for consistency of graph
	//TODO check
	return true;
}

Graph<PTYPE>* IOWrapper::runModelChecking() {
	clearSelfLoops();
	transformOneInput();
	transformOneOutput();

	ConfigurationSingleton *conf = ConfigurationSingleton::getInstance();

	//All input read -> now do the model checking
	if (conf->isIoflStateElim()) {
		LOG4CPLUS_INFO(iologger, "Model checking with state elimination approach");
		StateElim* tmp = new StateElim();
		tmp->setEliminationOrder(conf->getIoflElimOrder(), conf->isIoflIncreasingElimOrder());
		LOG4CPLUS_INFO(iologger,
				"Elimination order: " << EliminationOrder::orderToString(conf->getIoflElimOrder()) << ", " << (conf->isIoflIncreasingElimOrder() ? "increasing" : "decreasing"));
		mModelChecker = tmp;

	} else {
		LOG4CPLUS_INFO(iologger, "Model checking with SCC approach");
		mModelChecker = new SccMc();
	}

	mModelChecker->setAuxInputNode(mInitialNode);
	Graph<PTYPE>* g = mModelChecker->modelCheck(mVertexList, mInitialNodeID, mTargetNodes, mVtxToPropsMap, mEdgeCount);
	mResultFileName = writeMCResult(g);
	return g;
}

/**
 * A complete run of the MC/Cex Generator, i.e.
 * - Read files
 * - Run MC
 * - Possibly run counter example search
 * - Possibly run counter example search again if in benchmark mode
 * - Write the results to files
 */
void IOWrapper::run() {
	bool couldRead = readFiles();
	if (!couldRead) {
		return;
	}
	LOG4CPLUS_INFO(iologger, "Parsing successful");

	//SccMc::reachabilityAnalysis(mVertexList, mInitialNodeID, mTargetNodes, )

	if (!ConfigurationSingleton::getInstance()->isIoflReadFromXml()) {
		runModelChecking();
		setProbModelCheck(getResultProbability());
	} else {
		std::ifstream stream(inputFile.c_str());
		XMLReader parser(this);
		parser.getGraphFromXML(stream);
		runModelChecking();
		setProbModelCheck(getResultProbability());
	}

	//Display graph read from the input
	//	AbstractOutputFormatter<PTYPE> *formatter;
	//	if (ConfigurationSingleton::getInstance()->isIoflExternalCaller()) {
	//		formatter = new XMLOutputFormatter<PTYPE>(mInitialNode, mTargetNodes, true, false);
	//	} else {
	//		formatter = new XMLOutputFormatter<PTYPE>(mInitialNode, mTargetNodes, true, false);
	//	}

	//	LOG4CPLUS_TRACE(iologger, formatter->generateOutput(*g));

	//Generate counter example(s), if applicable
	LOG4CPLUS_INFO(iologger, "Model Checking result: " << getResultProbability());
	stringstream log;
	log << mModelChecker->getLog();
	log << "Model Checking result:\t" << getResultProbability() << endl;
#ifdef USE_PARAMETRIC
	//TODO make interactive
	//parametric::Parameters::getInstance().instantiate(getResultProbability());
#endif
	//Write log (appends if already exists)
	fstream outputFile(logFile.c_str(), fstream::app | fstream::out);
	outputFile << log.str();
	outputFile.close();
}

PTYPE IOWrapper::getResultProbability() const {
	if (mModelChecker != NULL) {
		return mModelChecker->getTargetResult(mTargetIndex);
	} else if (ConfigurationSingleton::getInstance()->isIoflExternalCaller()
			|| ConfigurationSingleton::getInstance()->isIoflReadFromXml()) {
		return mProbModelCheck;
	} else {
		return 0;
	}
}

bool IOWrapper::readFiles(std::istream *strm) {
	//Either we are provided with only a single DTMC file,
	//(this is how SCC_MC was invoked originally and for compatibility and
	// convenience we keep this possibility)
	//or an XML file with the location of DTMC, AP etc. is given

	//if(!Configuration::getInstance().readFromXml()) {

	string modelFile = inputFile;

	ConfigurationSingleton *conf = ConfigurationSingleton::getInstance();
	
	//read the input
	LOG4CPLUS_TRACE(iologger, "will now read model file from input...");
	bool ok = true;
	if (modelFile.find(".pmc") != string::npos) {
		LOG4CPLUS_TRACE(iologger, "will parse pmc file ...");
		PMCReader parser(this);
		ok = parser.parse(strm);
	} else if (modelFile.find(".pm") != string::npos) {
		LOG4CPLUS_TRACE(iologger, "will parse pm file ...");
		PrismReader parser(this);
		//Create file name for pctl file
		size_t posSlash = modelFile.find_last_of("/");
		size_t pos = modelFile.find("_", posSlash + 1);
		if (pos == string::npos) {
			//Not found
			pos = modelFile.find(".pm");
		}
		string propertyFile = modelFile.substr(0, pos) + ".pctl";
		LOG4CPLUS_TRACE(iologger, "Property file: " << propertyFile);
		ok = parser.parse(modelFile, propertyFile);
	} else if ((modelFile.find(".tra") != string::npos) || (modelFile.find(".dtmc") != string::npos)) {
		LOG4CPLUS_TRACE(iologger, "will parse tra/dtmc file ...");
		DTMCReader parser(this);
		ok = parser.parse(strm);
	} else if (modelFile.find(".xml") != string::npos || conf->isIoflExternalCaller()) {
		LOG4CPLUS_TRACE(iologger, "will parse xml file ...");
		XMLReader parser(this);
		ok = parser.parse(strm);
	} else {
		ok = false;
	}
	if (ok) {
		LOG4CPLUS_TRACE(iologger, "Finished reading the File");
		return true;
	} else {
		LOG4CPLUS_ERROR(iologger, "Could *not* read the File: " + modelFile);
		return false;
	}
}

string IOWrapper::mcResultToString(Graph<PTYPE>* g) const {
	//Choose output formatter depending on destination parameter:
	AbstractOutputFormatter<PTYPE> *formatter;
	if (ConfigurationSingleton::getInstance()->isIoflExternalCaller()) {
		//formatter = new DefaultOutputFormatter<PTYPE>(mInitialNode);
		formatter = new XMLOutputFormatter<PTYPE>(this, true, false);
	} else {
		formatter = new XMLOutputFormatter<PTYPE>(this, true, false);
	}
	return formatter->generateOutput(*g);
}

std::string IOWrapper::writeMCResult(Graph<PTYPE>* g) const {
	//unsigned int flags = Configuration::getInstance().getFlags();
	string output = "";
	AbstractOutputFormatter<PTYPE> *formatter;
	string filepath = mResultFile;
	if (mResultFileName != "") {
		filepath = mResultFileName;
	}
	string name = filepath.substr(0, filepath.find_first_of("."));
	bool writeFile = false;

	if (ConfigurationSingleton::getInstance()->isIoflOutputAsDtmc()) {
		formatter = new DefaultOutputFormatter<PTYPE>(mInitialNode, false);
		output = formatter->generateOutput(*g);
		name = name + ".dtmc";
		writeFile = true;
	} else if (ConfigurationSingleton::getInstance()->isIoflOutputAsXml()) {
		formatter = new XMLOutputFormatter<PTYPE>(this, true, ConfigurationSingleton::getInstance()->isIoflCexConcrete());
		output = formatter->generateOutput(*g);
		name = name + ".xml";
		writeFile = true;
	}

	LOG4CPLUS_DEBUG(iologger, output);
	if (writeFile) {
		fstream outputFile(name.c_str(), fstream::out);
		outputFile << output;
		outputFile.close();
		LOG4CPLUS_INFO(iologger, "Result can be found in: " << name);
	}

	return name;
}

void IOWrapper::transformOneInput() {
	//We don't want edges leading into the initial state, so if that's the case,
	//we add an auxiliary initial node
	if (mVertexList[mInitialNodeID]->inEdges.size() > 0) {
		mInitialNode = new Vertex<PTYPE>();
		mInitialNode->oindex = mVertexList.size();
		mVertexList.push_back(mInitialNode);
		new Edge<PTYPE>(mInitialNode, mVertexList[mInitialNodeID], 1.0);
		mInitialNodeID = mInitialNode->oindex;

		LOG4CPLUS_DEBUG(iologger, "Added auxiliary initial node with id " << mInitialNodeID);
	} else {
		if (mInitialNode != NULL) {
			mInitialNode = NULL;
		}
	}

}

void IOWrapper::transformOneOutput() {
	unsigned int numTargets = 0;
	for (unsigned int i = 0; i < mTargetNodes.size(); i++) {
		if (isTarget(i)) {
			numTargets++;
		}
	}

	//If we are in counter example mode, or we might later be there (external caller!),
	//we only want a single target node...
	if (numTargets > 1) { // && (mGenerateCEx || ConfigurationSingleton::getInstance()->isIoflExternalCaller()
		//|| ConfigurationSingleton::getInstance()->isIoflOnlyModelChecking())) {
		LOG4CPLUS_DEBUG(iologger, "For counter example mode we need a single target, will introduce auxiliary target.");

		//...therefore create new target node...
		Vertex<PTYPE>* auxTargetNode = new Vertex<PTYPE>();
		auxTargetNode->oindex = mVertexList.size();
		mVertexList.push_back(auxTargetNode);
		mTargetIndex = auxTargetNode->oindex;

		//...and edges to that target node
		for (unsigned int i = 0; i < mTargetNodes.size(); i++) {
			if (isTarget(i)) {
				new Edge<PTYPE>(mVertexList[i], auxTargetNode, 1.0);

				//now this is not a target state any more
				mTargetNodes.at(i) = false;
			}
		}

		//Finally mark new node as target node
		setTarget(auxTargetNode->oindex);

		LOG4CPLUS_DEBUG(iologger, "Added auxiliary target node with id " << auxTargetNode->oindex);
	}
}

void IOWrapper::clearSelfLoops() {
	for (unsigned int i = 0; i < mVertexList.size(); i++) {
		Vertex<PTYPE> *vtx = mVertexList[i];
		if (vtx == NULL) {
			continue;
		}
		PTYPE selfProb = vtx->getSelfLoopProb();
		if (selfProb != 0) {
#ifdef DEBUGGING
			parametric::Parameters::getInstance().checkConstraintGreaterZero(selfProb);
#endif
			if (selfProb != 1) {
#ifdef DEBUGGING
				parametric::Parameters::getInstance().checkConstraintLessOne(selfProb);
#endif
				//Scale other edges
				for (unsigned int j = 0; j < vtx->outEdges.size(); j++) {
					vtx->outEdges[j]->prob /= (1 - selfProb);
				}
			}
			vtx->selfLoop->clear();
			delete vtx->selfLoop;
			vtx->selfLoop = NULL;
		}
	}
}

}
